/* ops-api-client.js
   Minimaler Ersatz für ops-offline-client.js:
   - stellt OpsOffline.apiFetch + ensureLoggedIn bereit
   - KEIN LocalStorage-Sync, KEIN Patchen von localStorage
*/
(() => {
  "use strict";

  if (window.__opsApiClientLoaded) return;
  window.__opsApiClientLoaded = true;

  const API_BASE = (window.OpsDeck && window.OpsDeck.apiBase) || "/api";

  function getLang() {
    try {
      if (typeof window.getOpsLang === "function") return window.getOpsLang();
    } catch (_) {}
    const l = document.documentElement.getAttribute("lang") || "de";
    return String(l).toLowerCase().startsWith("en") ? "en" : "de";
  }

  function t(de, en) {
    return getLang() === "en" ? (en ?? de) : (de ?? en);
  }

  function ensureLoginDialog() {
    let dlg = document.getElementById("opsLoginDialog");
    if (dlg) return dlg;

    dlg = document.createElement("dialog");
    dlg.id = "opsLoginDialog";
    dlg.className = "dialog";

    dlg.innerHTML = `
      <div class="card" style="max-width:520px;width:min(520px,96vw);padding:16px 18px 18px;">
        <div style="display:flex;align-items:flex-start;justify-content:space-between;gap:12px;">
          <div>
            <h3 style="margin:0;font-size:18px;">${t("Anmeldung", "Login")}</h3>
            <p class="tiny" style="margin:6px 0 0;opacity:.85;">${t("Lokaler Server", "Local server")}</p>
          </div>
          <button type="button" class="btn btn-ghost" id="opsLoginCloseX" aria-label="close">✕</button>
        </div>

        <form id="opsLoginForm" style="margin:14px 0 0;">
          <label class="tiny" style="display:block;">${t("Benutzername", "Username")}
            <input id="opsLoginUser" autocomplete="username" style="width:100%;padding:8px 10px;margin-top:6px;" />
          </label>

          <label class="tiny" style="display:block;margin-top:10px;">${t("Passwort", "Password")}
            <input id="opsLoginPass" type="password" autocomplete="current-password" style="width:100%;padding:8px 10px;margin-top:6px;" />
          </label>

          <div id="opsLoginErr" class="tiny" style="color:#b00020;margin-top:10px;display:none;"></div>

          <div style="display:flex;justify-content:flex-end;gap:10px;margin-top:14px;">
            <button type="button" class="btn btn-ghost" id="opsLoginCancel">${t("Abbrechen", "Cancel")}</button>
            <button type="submit" class="btn btn-primary" id="opsLoginBtn">${t("Anmelden", "Login")}</button>
          </div>
        </form>
      </div>
    `;

    document.body.appendChild(dlg);

    const close = () => { try { dlg.close(); } catch (_) { dlg.removeAttribute("open"); } };
    dlg.querySelector("#opsLoginCloseX")?.addEventListener("click", close);
    dlg.querySelector("#opsLoginCancel")?.addEventListener("click", close);

    return dlg;
  }

  function showLoginError(dlg, msg) {
    const el = dlg.querySelector("#opsLoginErr");
    if (!el) return;
    el.textContent = msg || "";
    el.style.display = msg ? "block" : "none";
  }

  async function apiFetch(path, options) {
    const opts = Object.assign(
      {
        method: "GET",
        credentials: "include",
        headers: { "Content-Type": "application/json" }
      },
      options || {}
    );

    if (opts.body && typeof opts.body !== "string" && !(opts.body instanceof FormData)) {
      opts.body = JSON.stringify(opts.body);
    }

    if (opts.body instanceof FormData) {
      if (opts.headers) delete opts.headers["Content-Type"];
    }

    const res = await fetch(API_BASE + path, opts);

    const isJson = (res.headers.get("content-type") || "").includes("application/json");
    const data = isJson ? await res.json().catch(() => ({})) : await res.text().catch(() => "");

    if (!res.ok) {
      const msg =
        (data && data.detail) ? data.detail :
        (typeof data === "string" && data) ? data :
        `Request failed (${res.status})`;

      const err = new Error(msg);
      err.status = res.status;
      err.data = data;
      throw err;
    }

    return data;
  }

  async function me() {
    try {
      return await apiFetch("/me");
    } catch (e) {
      if (e && e.status === 401) return null;
      throw e;
    }
  }

  async function login(username, password) {
    return await apiFetch("/auth/login", { method: "POST", body: { username, password } });
  }

  async function logout() {
    try { await apiFetch("/auth/logout", { method: "POST", body: {} }); } catch (_) {}
  }

  async function ensureLoggedIn() {
    const current = await me();
    if (current) return current;

    const dlg = ensureLoginDialog();
    const form = dlg.querySelector("#opsLoginForm");
    const userEl = dlg.querySelector("#opsLoginUser");
    const passEl = dlg.querySelector("#opsLoginPass");

    showLoginError(dlg, "");

    return await new Promise((resolve) => {
      const cleanup = () => {
        form?.removeEventListener("submit", onSubmit);
        dlg?.removeEventListener("close", onClose);
      };

      const onClose = () => {
        cleanup();
        resolve(null);
      };

      const onSubmit = async (ev) => {
        ev.preventDefault();
        showLoginError(dlg, "");
        const u = (userEl?.value || "").trim();
        const p = (passEl?.value || "").trim();
        if (!u || !p) {
          showLoginError(dlg, t("Bitte Benutzername und Passwort eingeben.", "Please enter username and password."));
          return;
        }

        try {
          await login(u, p);
          try { dlg.close(); } catch (_) {}
          const who = await me();
          cleanup();
          resolve(who);
        } catch (e) {
          showLoginError(dlg, (e && e.message) ? e.message : t("Login fehlgeschlagen.", "Login failed."));
        }
      };

      form?.addEventListener("submit", onSubmit);
      dlg?.addEventListener("close", onClose);

      try { dlg.showModal(); } catch (_) { dlg.setAttribute("open", ""); }
      setTimeout(() => userEl?.focus?.(), 0);
    });
  }

  // Kompatibilität für bestehende Module (rbac erwartet OpsOffline.apiFetch)
  window.OpsOffline = window.OpsOffline || {};
  window.OpsOffline.apiBase = API_BASE;
  window.OpsOffline.apiFetch = apiFetch;
  window.OpsOffline.ensureLoggedIn = ensureLoggedIn;
  window.OpsOffline.me = me;
  window.OpsOffline.logout = logout;

  // optional: readiness promise
  function startBoot() {
    if (window.__opsOfflineReady) return window.__opsOfflineReady;
    const p = ensureLoggedIn().catch((e) => { console.error(e); return null; });
    window.__opsOfflineReady = p;
    return p;
  }

  if (document.readyState === "loading") {
    document.addEventListener("DOMContentLoaded", () => { startBoot(); });
  } else {
    startBoot();
  }
})();
