// ops-auth-engineering.js
// Rollen (Gast / Mitarbeiter / Admin), Admin-Login, Lizenzplan & Feature-Gates
// Speziell für das Engineering-Portal (Basic / Pro / Ultimate)

(() => {
  // ========================
  // Layout-Settings (Live-Kacheln)
  // ========================
  const LAYOUT_STORAGE_KEY = "opsdeck360:layout:engineering:v1";
  let layoutState = null;
  let liveToggleBtn = null;

  function loadLayout() {
    if (layoutState) return layoutState;
    const defaults = {
      sections: {
        liveTiles: true, // Eigene Live-Kacheln sichtbar?
      },
    };
    try {
      const raw = localStorage.getItem(LAYOUT_STORAGE_KEY);
      if (!raw) {
        layoutState = JSON.parse(JSON.stringify(defaults));
      } else {
        const parsed = JSON.parse(raw);
        layoutState = {
          sections: Object.assign(
            {},
            defaults.sections,
            (parsed && parsed.sections) || {}
          ),
        };
      }
    } catch (err) {
      console.warn("[layout] Konnte Layout nicht laden, verwende Defaults:", err);
      layoutState = JSON.parse(JSON.stringify(defaults));
    }
    return layoutState;
  }

  function saveLayout() {
    try {
      localStorage.setItem(LAYOUT_STORAGE_KEY, JSON.stringify(layoutState));
    } catch (err) {
      console.warn("[layout] Konnte Layout nicht speichern:", err);
    }
  }

  function applyLayout() {
    const cfg = loadLayout();
    const show = !!cfg.sections.liveTiles;

    // Karte "Eigene Live-Kacheln" immer sichtbar lassen,
    // nur den Inhalt (Grid) ein-/ausblenden
    const livePanel = document.getElementById("live-tiles");
    if (livePanel) {
      const grid =
        document.getElementById("liveTileGrid") ||
        livePanel.querySelector("#liveTileGrid");
      if (grid) {
        grid.style.display = show ? "" : "none";
      }
    }

    if (liveToggleBtn) {
      liveToggleBtn.textContent = show
        ? "Live-Kacheln ausblenden"
        : "Live-Kacheln anzeigen";
      liveToggleBtn.setAttribute("aria-pressed", show ? "false" : "true");
    }
  }

  function setupLiveTilesToggle() {
    const livePanel = document.getElementById("live-tiles");
    if (!livePanel) return;

    const toolbar = livePanel.querySelector(".card-header .toolbar");
    if (!toolbar) return;

    let btn = toolbar.querySelector('[data-layout-toggle="liveTiles"]');
    if (!btn) {
      btn = document.createElement("button");
      btn.type = "button";
      btn.className = "btn btn-ghost";
      btn.dataset.layoutToggle = "liveTiles";
      btn.id = "liveTilesToggle";
      btn.title = "Eigene Live-Kacheln ein-/ausblenden";
      toolbar.appendChild(btn);

      btn.addEventListener("click", () => {
        const cfg = loadLayout();
        cfg.sections.liveTiles = !cfg.sections.liveTiles;
        saveLayout();
        applyLayout();
      });
    }

    liveToggleBtn = btn;
  }

  // ========================
  // Rollen / Pläne / Admin
  // ========================
  const KEY_ROLE = "opsdeck.role";
  const KEY_PLAN = "opsdeck.plan";
  const KEY_ADMIN_HASH = "opsdeck.admin_hash";

  let sessionAdminOK = false;

  // Rechte-Matrix (nur Oberfläche)
  const PERMISSIONS = {
    // Dokumente
    "doc.delete": ({ role }) => role === "admin",
    "doc.deleteAll": ({ role }) => role === "admin",

    // Aufgaben
    "tasks.reset": ({ role }) => role === "admin",
    "tasks.import": ({ role }) => role === "admin" || role === "user",
    "tasks.export": ({ role }) => role !== "viewer",

    // CSV-Werkbank
    "csv.import": ({ role, plan }) =>
      (plan === "pro" || plan === "ultimate") && role !== "viewer",
    "csv.clear": ({ role }) => role === "admin",

    // Eigene Schnellzugriff-Kacheln
    "tiles.add": ({ role, plan }) => role === "admin" && plan === "ultimate",
    "tiles.delete": ({ role, plan }) => role === "admin" && plan === "ultimate",
  };

  const OpsDeck = {
    // viewer = Gast, user = Mitarbeiter, admin = Admin
    role: "viewer",
    // basic = Instandhalter, pro = Techniker, ultimate = Engineering
    plan: "basic",
    _els: {},

    // ========================
    // Initialisierung
    // ========================
    async init() {
      // gespeicherte Rolle/Plan laden
      const savedRole = localStorage.getItem(KEY_ROLE);
      if (["viewer", "user", "admin"].includes(savedRole)) {
        this.role = savedRole;
      }
      const savedPlan = localStorage.getItem(KEY_PLAN);
      if (["basic", "pro", "ultimate"].includes(savedPlan)) {
        this.plan = savedPlan;
      }

      // Header-Elemente
      this._els.roleSel = document.getElementById("opsRoleSelect");
      this._els.roleLabel = document.getElementById("opsRoleLabel");
      this._els.planSel = document.getElementById("opsPlanSelect");
      this._els.planLabel = document.getElementById("opsPlanLabel");

      // Konfiguration
      this._els.planLabelCfg = document.getElementById("opsPlanLabelConfig");
      this._els.changePwBtn = document.getElementById("opsChangeAdminPw");

      // Admin-Login-Dialog
      this._els.loginDialog = document.getElementById("adminLoginDialog");
      this._els.loginPassword =
        this._els.loginDialog &&
        this._els.loginDialog.querySelector("#adminLoginPassword");

      // Rollenwechsel im Header
      if (this._els.roleSel) {
        this._els.roleSel.addEventListener("change", async (e) => {
          const requested = e.target.value;

          if (requested === "admin" && this.role !== "admin") {
            const ok = await this.ensureAdminAuth();
            if (!ok) {
              e.target.value = this.role;
              return;
            }
          }

          this.setRole(requested);
        });
      }

      // Planwechsel (nur Admin)
      if (this._els.planSel) {
        this._els.planSel.addEventListener("change", (e) => {
          const val = e.target.value;
          if (this.role !== "admin") {
            e.target.value = this.plan;
            alert("Nur Admin darf den Lizenzplan ändern.");
            return;
          }
          this.setPlan(val);
        });
      }

      // Admin-Passwort ändern (Fallback per prompt)
      if (this._els.changePwBtn) {
        this._els.changePwBtn.addEventListener("click", async () => {
          if (this.role !== "admin") {
            alert("Nur Admin darf das Admin-Passwort ändern.");
            return;
          }
          const changed = await this.showAdminPasswordDialog();
          if (changed) {
            sessionAdminOK = false;
            alert("Admin-Passwort wurde geändert.");
          }
        });
      }

      await this.updateUI();
    },

    // ========================
    // UI aktualisieren
    // ========================
    async updateUI() {
      // Dropdown-Werte & einmalige Engineering-Labels
      if (this._els.roleSel) {
        this._els.roleSel.value = this.role;

        if (!this._els.roleSel.dataset.engineeringLabelsApplied) {
          const rs = this._els.roleSel;
          Array.from(rs.options).forEach((opt) => {
            if (opt.value === "viewer") opt.textContent = "Gast / Lesen";
            if (opt.value === "user") opt.textContent = "Mitarbeiter";
            if (opt.value === "admin") opt.textContent = "Systemadmin";
          });
          this._els.roleSel.dataset.engineeringLabelsApplied = "1";
        }
      }

      if (this._els.planSel) {
        this._els.planSel.value = this.plan;

        if (!this._els.planSel.dataset.engineeringLabelsApplied) {
          const ps = this._els.planSel;
          Array.from(ps.options).forEach((opt) => {
            if (opt.value === "basic")
              opt.textContent = "Instandhalter (Basic)";
            if (opt.value === "pro") opt.textContent = "Techniker (Pro)";
            if (opt.value === "ultimate")
              opt.textContent = "Engineering (Ultimate)";
          });
          this._els.planSel.dataset.engineeringLabelsApplied = "1";
        }
      }

      // Rolle-Label
      if (this._els.roleLabel) {
        let roleNice = "Gast";
        if (this.role === "admin") {
          roleNice = "Admin (Engineering)";
        } else if (this.role === "user") {
          if (this.plan === "basic") roleNice = "Instandhalter";
          else if (this.plan === "pro") roleNice = "Techniker";
          else roleNice = "Engineering";
        } else {
          roleNice = "Gast";
        }
        this._els.roleLabel.textContent = `Rolle: ${roleNice}`;
      }

      // Plan-Label
      const planNice =
        this.plan === "basic"
          ? "Instandhalter (Basic)"
          : this.plan === "pro"
          ? "Techniker (Pro)"
          : "Engineering (Ultimate)";

      if (this._els.planLabel) {
        this._els.planLabel.textContent = `Plan: ${planNice}`;
      }
      if (this._els.planLabelCfg) {
        this._els.planLabelCfg.textContent = `Plan: ${planNice}`;
      }

      // Für CSS / Debug
      if (document.body) {
        document.body.dataset.role = this.role;
        document.body.dataset.plan = this.plan;
      }

      // Feature-Gates
      this.applyFeaturesToDom();

      // Live-Kachel-Layout
      try {
        setupLiveTilesToggle();
        applyLayout();
      } catch (err) {
        console.warn(
          "[ops-auth-engineering] Layout-Update fehlgeschlagen:",
          err
        );
      }
    },

    // ========================
    // Rolle / Plan setzen
    // ========================
    setRole(role) {
      if (!["viewer", "user", "admin"].includes(role)) role = "viewer";
      this.role = role;
      try {
        localStorage.setItem(KEY_ROLE, role);
      } catch {}
      this.updateUI();
    },

    setPlan(plan) {
      if (!["basic", "pro", "ultimate"].includes(plan)) plan = "basic";
      this.plan = plan;
      try {
        localStorage.setItem(KEY_PLAN, plan);
      } catch {}
      this.updateUI();
    },

    isAdmin() {
      return this.role === "admin";
    },

    // ========================
    // Admin-Login / Passwort
    // ========================
    async ensureAdminAuth() {
      if (sessionAdminOK && this.role === "admin") return true;
      const ok = await this.showAdminLoginDialog();
      if (!ok) return false;
      sessionAdminOK = true;
      return true;
    },

    async checkAdminPassword(input) {
      const storedHash = localStorage.getItem(KEY_ADMIN_HASH);
      if (!storedHash) {
        // Default, solange kein Passwort gesetzt ist
        return input === "admin";
      }
      const inHash = await sha256hex(input);
      return inHash === storedHash;
    },

    // Plan-Funktionen (Engineering-Portal)
    // Mapping:
    //  - Basic    -> Instandhalter
    //  - Pro      -> Techniker
    //  - Ultimate -> Engineering
    hasFeature(name) {
      switch (name) {
        case "docs":
          // Dokumente & FileHub nur Ultimate
          return this.plan === "ultimate";

        case "tasks":
          // Aufgaben sollen in allen Plänen sichtbar sein (Basic/Pro/Ultimate)
          return true;

        case "csv":
          // CSV-Werkbank ab Pro
          return this.plan === "pro" || this.plan === "ultimate";

        case "quickFixed":
          // OpsDeck360 • Schnellzugriff (feste Kacheln) ab Pro
          return this.plan === "pro" || this.plan === "ultimate";

        case "quickCustom":
          // Eigene Schnellzugriff-Kacheln nur Ultimate
          return this.plan === "ultimate";

        case "liveTiles":
          // Eigene Live-Kacheln nur Ultimate
          return this.plan === "ultimate";

        case "planist":
          // Plan/Ist-Demo ab Pro
          return this.plan === "pro" || this.plan === "ultimate";

        case "config":
          // Konfiguration nur im Ultimate-Plan sichtbar
          return this.plan === "ultimate";

        case "backup":
          // Backup nur im Ultimate-Plan (Admin-Check extra)
          return this.plan === "ultimate";

        default:
          return false;
      }
    },

    // Rechte für Module
    can(action) {
      const rule = PERMISSIONS[action];
      if (!rule) return false;
      const ctx = { role: this.role, plan: this.plan };
      return !!rule(ctx);
    },

    // ========================
    // Feature-Gates im DOM
    // ========================
    applyFeaturesToDom() {
      const show = (el, allowed) => {
        if (!el) return;
        el.style.display = allowed ? "" : "none";
      };

      // --- Dokumente & Schnellzugriff auseinanderziehen ---
      let docsHubSection = null;
      let quickSection = null;

      const docsCandidates = Array.from(
        document.querySelectorAll('section[data-feature="docs"], section#documents')
      );
      docsCandidates.forEach((sec) => {
        if (!(sec instanceof HTMLElement)) return;
        if (sec.querySelector("#docDropzone") || sec.querySelector("#docList")) {
          docsHubSection = sec;
        } else if (sec.querySelector("#fixedTiles")) {
          quickSection = sec;
        }
      });

      if (docsHubSection) docsHubSection.dataset.feature = "docs";
      if (quickSection) quickSection.dataset.feature = "quickFixed";

      // --- Navigation & Hero-Buttons ---
      const navDocs = document.querySelector('a[data-target="documents"]');
      const navCsv = document.querySelector('a[data-target="csv"]');

      const heroDocs = document.querySelector('.hero-actions a[href="#documents"]');
      const heroTasks = document.querySelector('.hero-actions a[href="#tasks"]');
      const heroCsv = document.querySelector('.hero-actions a[href="#csv"]');

      const anyDocsFeature =
        this.hasFeature("docs") || this.hasFeature("quickFixed");

      if (navDocs) navDocs.style.display = anyDocsFeature ? "" : "none";
      if (heroDocs) heroDocs.style.display = anyDocsFeature ? "" : "none";

      if (navCsv) navCsv.style.display = this.hasFeature("csv") ? "" : "none";
      if (heroCsv) heroCsv.style.display = this.hasFeature("csv") ? "" : "none";
      if (heroTasks) heroTasks.style.display = this.hasFeature("tasks") ? "" : "none";

      // --- Aufgaben ---
      document
        .querySelectorAll('[data-feature="tasks"]')
        .forEach((el) => show(el, this.hasFeature("tasks")));

      // --- CSV-Werkbank ---
      document
        .querySelectorAll('[data-feature="csv"]')
        .forEach((el) => show(el, this.hasFeature("csv")));

      // --- Dokumente & FileHub (Ultimate) ---
      if (docsHubSection) {
        show(docsHubSection, this.hasFeature("docs"));
      }

      // --- OpsDeck360 • Schnellzugriff (feste Kacheln) ---
      if (quickSection) {
        show(quickSection, this.hasFeature("quickFixed"));
      }

      // --- Eigene Schnellzugriff-Kacheln ---
      const customTilesSection = (() => {
        const c = document.getElementById("customTiles");
        return c ? c.closest("section") : null;
      })();
      if (customTilesSection) {
        show(customTilesSection, this.hasFeature("quickCustom"));
      }

      // +Kachel-Button (nur Ultimate + Admin)
      const tileAddBtn = document.getElementById("addTileBtn");
      if (tileAddBtn) {
        const allowAdd =
          this.hasFeature("quickCustom") && this.can("tiles.add");
        tileAddBtn.disabled = !allowAdd;
      }

      // --- Eigene Live-Kacheln ---
      const liveTilesSection = document.getElementById("live-tiles");
      if (liveTilesSection) {
        show(liveTilesSection, this.hasFeature("liveTiles"));
      }

      // --- Plan / Ist Demo ---
      const planIstPanel = document.getElementById("planIstPanel");
      if (planIstPanel) {
        show(planIstPanel, this.hasFeature("planist"));
      }

      // --- Konfiguration & Backup (nur Ultimate + Admin) ---
      const cfg = document.getElementById("opsConfig");
      if (cfg) {
        const allowCfg = this.isAdmin() && this.hasFeature("config");
        show(cfg, allowCfg);
      }

      const allowBackup = this.isAdmin() && this.hasFeature("backup");
      const backupExport = document.getElementById("backupExportBtn");
      const backupImport = document.getElementById("backupImportInput");
      if (backupExport) backupExport.style.display = allowBackup ? "" : "none";
      if (backupImport) backupImport.style.display = allowBackup ? "" : "none";

      // --- Dokumente-Buttons ---
      const delAll = document.getElementById("docDeleteAll");
      if (delAll) {
        const canDelAll = this.can("doc.deleteAll");
        delAll.style.display = canDelAll ? "" : "none";
        delAll.disabled = !canDelAll;
      }

      // --- Aufgaben-Buttons ---
      const tasksReset = document.getElementById("tasksReset");
      if (tasksReset) {
        tasksReset.disabled = !this.can("tasks.reset");
      }
      const tasksCsvInput = document.getElementById("tasksCsvInput");
      if (tasksCsvInput) {
        tasksCsvInput.disabled = !this.can("tasks.import");
      }
      const tasksExport = document.getElementById("tasksExport");
      if (tasksExport) {
        tasksExport.disabled = !this.can("tasks.export");
      }

      // --- CSV-Buttons ---
      const csvFileInput = document.getElementById("csvFileInput");
      if (csvFileInput) {
        csvFileInput.disabled = !this.can("csv.import");
      }
      const csvClear = document.getElementById("csvClear");
      if (csvClear) {
        csvClear.disabled = !this.can("csv.clear");
      }
    },

    // ---------- Admin-Login-Dialog ----------
    showAdminLoginDialog() {
      const dlg = this._els.loginDialog;
      const pwInput = this._els.loginPassword;

      // Fallback: prompt(), falls dialog nicht existiert
      if (!dlg || !pwInput || typeof dlg.showModal !== "function") {
        const input = window.prompt("Admin-Passwort eingeben:");
        if (input == null) return Promise.resolve(false);
        return this.checkAdminPassword(input).then((ok) => {
          if (!ok) alert("Falsches Passwort.");
          return ok;
        });
      }

      return new Promise((resolve) => {
        const onClose = async () => {
          dlg.removeEventListener("close", onClose);
          const action = dlg.returnValue;
          const pwd = pwInput.value || "";
          pwInput.value = "";

          if (action !== "ok") {
            resolve(false);
            return;
          }

          const ok = await this.checkAdminPassword(pwd);
          if (!ok) {
            alert("Falsches Passwort.");
            resolve(false);
            return;
          }

          resolve(true);
        };

        dlg.addEventListener("close", onClose, { once: true });
        dlg.showModal();
        pwInput.focus();
      });
    },

    // ---------- Admin-Passwort ändern (reines Prompt-UI) ----------
    showAdminPasswordDialog() {
      return (async () => {
        const current = window.prompt("Aktuelles Admin-Passwort:");
        if (current == null) return false;
        const ok = await this.checkAdminPassword(current);
        if (!ok) {
          alert("Falsches aktuelles Passwort.");
          return false;
        }

        const next = window.prompt("Neues Admin-Passwort:");
        if (!next) {
          alert("Abgebrochen.");
          return false;
        }
        const next2 = window.prompt("Neues Passwort wiederholen:");
        if (next2 !== next) {
          alert("Passwörter stimmen nicht überein.");
          return false;
        }

        const hash = await sha256hex(next);
        localStorage.setItem(KEY_ADMIN_HASH, hash);
        return true;
      })();
    },
  };

  window.OpsDeck = OpsDeck;

  // ========================
  // Passwort-Hash & Factory-Reset
  // ========================
  async function sha256hex(text) {
    const data = new TextEncoder().encode(text);
    const buf = await crypto.subtle.digest("SHA-256", data);
    return Array.from(new Uint8Array(buf))
      .map((b) => b.toString(16).padStart(2, "0"))
      .join("");
  }

  // Factory-Reset (löscht lokale Daten auf diesem Gerät/Browser)
  async function onFactoryResetClick() {
    const ok1 = confirm(
      "⚠️ Wirklich ALLES lokal löschen? (Dokumente, Aufgaben, CSV, Kacheln, Einstellungen)"
    );
    if (!ok1) return;

    const phrase = prompt('Zur Bestätigung bitte "LÖSCHEN" eingeben:');
    if (!phrase || !/^LÖSCHEN|LOESCHEN$/i.test(phrase.trim())) return;

    // Button sperren
    const btn = document.getElementById("factoryResetBtn");
    if (btn) {
      btn.disabled = true;
      btn.textContent = "Lösche…";
    }

    try {
      await factoryResetAll();
      alert("Alles gelöscht. Die Seite wird neu geladen.");
      location.reload();
    } catch (err) {
      console.error(err);
      alert("Fehler beim Löschen. Details in der Konsole.");
      if (btn) {
        btn.disabled = false;
        btn.textContent = "Factory-Reset";
      }
    }
  }

  async function factoryResetAll() {
    try {
      localStorage.clear();
    } catch {}
    try {
      sessionStorage.clear();
    } catch {}

    // IndexedDB möglichst komplett leeren
    try {
      if (indexedDB && typeof indexedDB.databases === "function") {
        const dbs = await indexedDB.databases();
        for (const db of dbs) {
          if (db && db.name) {
            await new Promise((res) => {
              const req = indexedDB.deleteDatabase(db.name);
              req.onsuccess = req.onerror = req.onblocked = () => res();
            });
          }
        }
      } else {
        const KNOWN_DBS = [
          "opsdeck-files",
          "opsdeck_docs",
          "opsdeck-tasks",
          "opsdeck_csv",
          "opsdeck-tiles",
          "opsdeck-settings",
        ];
        for (const name of KNOWN_DBS) {
          await new Promise((res) => {
            const req = indexedDB.deleteDatabase(name);
            req.onsuccess = req.onerror = req.onblocked = () => res();
          });
        }
      }
    } catch (e) {
      console.warn("IndexedDB konnte nicht komplett gelöscht werden:", e);
    }

    // Cache Storage
    try {
      if (self.caches && typeof caches.keys === "function") {
        const keys = await caches.keys();
        await Promise.all(keys.map((k) => caches.delete(k)));
      }
    } catch (e) {
      console.warn("Caches konnten nicht komplett gelöscht werden:", e);
    }
  }

  // ========================
  // DOM ready
  // ========================
  document.addEventListener("DOMContentLoaded", () => {
    OpsDeck.init();

    const resetBtn = document.getElementById("factoryResetBtn");
    if (resetBtn) {
      resetBtn.addEventListener("click", onFactoryResetClick);
    }
  });
})();
