/* ops-belegliste-addon.js
   Add-on: Belegliste für Wareneingang/Warenausgang.

   - Liest Belege aus localStorage: opsdeck_lager_receipts_v1
   - Öffnen per Button oder Event: window.dispatchEvent(new CustomEvent('opsReceiptsOpenDialog'))
   - Zeigt Liste + Detailansicht + Druck.

   Keine Abhängigkeit von Core-IIFE, nur DOM + localStorage.
*/
(() => {
  "use strict";

  const STORAGE_ARTICLES = "opsdeck_lager_v1";
  const STORAGE_BOOKINGS = "opsdeck_lager_bookings_v1";
  const STORAGE_RECEIPTS = "opsdeck_lager_receipts_v1";

  // File Store (IndexedDB) – gleicher Name wie im Core (falls vorhanden)
  const FILES_DB_NAME  = "opsdeck_lager_files_v1";
  const FILES_DB_STORE = "files";
  let __filesDbPromise = null;

  // ---- Sprache ---------------------------------------------------------
  function getLang() {
    try {
      if (typeof window.getOpsLang === "function") return window.getOpsLang();
    } catch (_) {}
    const l = document.documentElement.getAttribute("lang") || "de";
    return String(l).toLowerCase().startsWith("en") ? "en" : "de";
  }
  function t(de, en) {
    return getLang() === "en" ? en : de;
  }

function uiAlertCompat(de, en, titleDe, titleEn) {
  if (window.OPS_UI && window.OPS_UI.alert) return window.OPS_UI.alert(de, en, titleDe, titleEn);
  alert(t(de, en));
}

async function uiConfirmCompat(de, en, titleDe, titleEn) {
  if (window.OPS_UI && window.OPS_UI.confirm) return await window.OPS_UI.confirm(de, en, titleDe, titleEn);
  return confirm(t(de, en));
}

async function uiPromptCompat(de, en, def, titleDe, titleEn, labelDe, labelEn) {
  if (window.OPS_UI && window.OPS_UI.prompt) {
    return await window.OPS_UI.prompt(de, en, def, titleDe, titleEn, "OK", "OK", "Abbrechen", "Cancel", labelDe, labelEn);
  }
  return prompt(t(de, en), def);
}

  // ---- Storage helpers -------------------------------------------------
  function safeJSONParse(s, fallback) {
    try {
      const v = JSON.parse(s);
      return v == null ? fallback : v;
    } catch (_) {
      return fallback;
    }
  }
  function loadArticles() {
    return safeJSONParse(localStorage.getItem(STORAGE_ARTICLES) || "[]", []);
  }
  function loadBookings() {
    return safeJSONParse(localStorage.getItem(STORAGE_BOOKINGS) || "[]", []);
  }
  function loadReceipts() {
    return safeJSONParse(localStorage.getItem(STORAGE_RECEIPTS) || "[]", []);
  }


function saveArticles(arr) {
  try {
    localStorage.setItem(STORAGE_ARTICLES, JSON.stringify(Array.isArray(arr) ? arr : []));
  } catch (e) {
    console.warn("[Belegliste] saveArticles failed:", e);
  }
}
function saveBookings(arr) {
  try {
    localStorage.setItem(STORAGE_BOOKINGS, JSON.stringify(Array.isArray(arr) ? arr : []));
  } catch (e) {
    console.warn("[Belegliste] saveBookings failed:", e);
  }
}
function saveReceipts(arr) {
  try {
    localStorage.setItem(STORAGE_RECEIPTS, JSON.stringify(Array.isArray(arr) ? arr : []));
  } catch (e) {
    console.warn("[Belegliste] saveReceipts failed:", e);
  }
}

  // ---- IndexedDB: Datei öffnen ----------------------------------------
  function openFilesDb() {
    if (!("indexedDB" in window)) return Promise.resolve(null);
    if (__filesDbPromise) return __filesDbPromise;

    __filesDbPromise = new Promise((resolve) => {
      try {
        const req = indexedDB.open(FILES_DB_NAME, 1);
        req.onupgradeneeded = () => {
          try {
            const db = req.result;
            if (!db.objectStoreNames.contains(FILES_DB_STORE)) {
              db.createObjectStore(FILES_DB_STORE, { keyPath: "id" });
            }
          } catch (_) {}
        };
        req.onsuccess = () => resolve(req.result);
        req.onerror = () => resolve(null);
      } catch (_) {
        resolve(null);
      }
    });

    return __filesDbPromise;
  }

  async function getFileFromDb(fileId) {
    const db = await openFilesDb();
    if (!db || !fileId) return null;

    return await new Promise((resolve) => {
      try {
        const tx = db.transaction(FILES_DB_STORE, "readonly");
        const req = tx.objectStore(FILES_DB_STORE).get(String(fileId));
        req.onsuccess = () => resolve(req.result || null);
        req.onerror = () => resolve(null);
      } catch (_) {
        resolve(null);
      }
    });
  }

  async function openStoredFile(fileId) {
    try {
      const rec = await getFileFromDb(fileId);
      if (!rec || !rec.blob) {
        alert(t(
          "Datei nicht gefunden (evtl. Browser-Daten gelöscht).",
          "File not found (browser data may have been cleared)."
        ));
        return;
      }
      const url = URL.createObjectURL(rec.blob);
      const a = document.createElement("a");
      a.href = url;
      a.target = "_blank";
      a.rel = "noopener";
      document.body.appendChild(a);
      a.click();
      a.remove();
      setTimeout(() => URL.revokeObjectURL(url), 30000);
    } catch (e) {
      console.warn("[Belegliste] openStoredFile failed:", e);
    }
  }

  // ---- UI helpers ------------------------------------------------------
  function ensureStyleTag(styleId, cssText) {
    try {
      if (document.getElementById(styleId)) return;
      const st = document.createElement("style");
      st.id = styleId;
      st.textContent = cssText;
      document.head.appendChild(st);
    } catch (_) {}
  }

  function fmtDateTime(iso) {
    try {
      const d = new Date(iso);
      if (Number.isNaN(d.getTime())) return String(iso || "");
      const pad = (n) => String(n).padStart(2, "0");
      return `${pad(d.getDate())}.${pad(d.getMonth() + 1)}.${d.getFullYear()} ${pad(d.getHours())}:${pad(d.getMinutes())}`;
    } catch (_) {
      return String(iso || "");
    }
  }

  function receiptKind(r) {
    const k = String(r && (r.kind || r.type || "") || "").trim().toLowerCase();
    if (k) return k;
    if (r && r.deliveryNoteNo) return "wareneingang";
    if (r && r.docNo) return "warenausgang";
    return "beleg";
  }

  function receiptNo(r) {
    return String((r && (r.deliveryNoteNo || r.docNo || r.id || "")) || "");
  }

  function receiptPartner(r) {
    const k = receiptKind(r);
    if (k === "wareneingang") return String(r.supplier || "");
    if (k === "warenausgang") return String(r.recipient || "");
    if (k === "storno") return String(r.partner || r.refPartner || r.supplier || r.recipient || "");
    return "";
  }

  function kindLabel(k) {
    if (k === "wareneingang") return t("Wareneingang", "Goods receipt");
    if (k === "warenausgang") return t("Warenausgang", "Goods issue");
    if (k === "storno") return t("Storno", "Reversal");
    return t("Beleg", "Receipt");
  }

  // ---- Dialog ----------------------------------------------------------
  function ensureDialog() {
    let dlg = document.getElementById("invReceiptsDialog");
    if (dlg) return dlg;

    ensureStyleTag(
      "invReceiptsAddonStyle",
      `
      #invReceiptsDialog{ padding:0; border:0; }
      #invReceiptsDialog::backdrop{ background: rgba(0,0,0,.35); }
      #invReceiptsDialog .inv-rec-card{ max-width:1000px; width:min(1000px,96vw); padding:16px 18px 18px; }
      #invReceiptsDialog .inv-rec-head{ display:flex; align-items:flex-start; justify-content:space-between; gap:12px; }
      #invReceiptsDialog .inv-rec-head h3{ margin:0; font-size:18px; line-height:1.2; }
      #invReceiptsDialog .inv-rec-toolbar{ display:flex; gap:10px; flex-wrap:wrap; margin-top:12px; }
      #invReceiptsDialog .inv-rec-toolbar label{ display:flex; flex-direction:column; gap:6px; font-size:12px; }
      #invReceiptsDialog .inv-rec-toolbar input,
      #invReceiptsDialog .inv-rec-toolbar select{ padding:8px 10px; min-width:180px; }
      #invReceiptsDialog .inv-rec-grid{ display:grid; grid-template-columns: 1.2fr .8fr; gap:14px; margin-top:12px; }
      #invReceiptsDialog .inv-rec-list{ max-height:60vh; overflow:auto; border:1px solid rgba(255,255,255,.12); border-radius:12px; }
      #invReceiptsDialog table{ width:100%; border-collapse:collapse; }
      #invReceiptsDialog th, #invReceiptsDialog td{ padding:10px 10px; border-bottom:1px solid rgba(255,255,255,.08); font-size:12px; vertical-align:top; }
      #invReceiptsDialog thead th{ position:sticky; top:0; background: rgba(2,6,23,0.85); backdrop-filter: blur(6px); z-index:1; text-align:left; }
      #invReceiptsDialog tr[data-row]{ cursor:pointer; }
      #invReceiptsDialog tr[data-row]:hover{ background: rgba(255,255,255,.04); }
      #invReceiptsDialog .inv-rec-detail{ border:1px solid rgba(255,255,255,.12); border-radius:12px; padding:12px; max-height:60vh; overflow:auto; }
      #invReceiptsDialog .inv-rec-detail h4{ margin:0 0 8px; font-size:14px; }
      #invReceiptsDialog .inv-rec-meta{ font-size:12px; opacity:.9; display:grid; grid-template-columns:1fr 1fr; gap:8px 12px; margin-bottom:10px; }
      #invReceiptsDialog .inv-rec-meta div{ white-space:nowrap; overflow:hidden; text-overflow:ellipsis; }
      #invReceiptsDialog .inv-rec-lines{ margin-top:10px; }
      #invReceiptsDialog .inv-rec-lines table td{ font-size:12px; }
      #invReceiptsDialog .inv-rec-files{ display:flex; flex-wrap:wrap; gap:8px; margin-top:10px; }
      #invReceiptsDialog .inv-rec-actions{ display:flex; justify-content:flex-end; gap:10px; margin-top:12px; }
      @media (max-width: 900px){
        #invReceiptsDialog .inv-rec-grid{ grid-template-columns: 1fr; }
      }
    `
    );

    dlg = document.createElement("dialog");
    dlg.id = "invReceiptsDialog";
    dlg.className = "dialog";

    dlg.innerHTML = `
      <div class="card inv-rec-card">
        <div class="inv-rec-head">
          <div>
            <h3>🧾 ${t("Belegliste", "Receipts")}</h3>
            <p class="tiny" style="margin:6px 0 0; opacity:.85;">${t("Wareneingang & Warenausgang (lokal im Browser)", "Goods receipts & issues (stored locally in the browser)")}</p>
          </div>
          <button type="button" id="invReceiptsCloseX" class="btn btn-ghost" aria-label="close" title="${t("Schließen", "Close")}">✕</button>
        </div>

        <div class="inv-rec-toolbar">
          <label>
            <span>${t("Typ", "Type")}</span>
            <select id="invReceiptsFilterKind">
              <option value="">${t("Alle", "All")}</option>
              <option value="wareneingang">${t("Wareneingang", "Goods receipt")}</option>
              <option value="warenausgang">${t("Warenausgang", "Goods issue")}</option>
              <option value="storno">↩️ ${t("Storno", "Reversal")}</option>
            </select>
          </label>

          <label>
            <span>${t("Suche", "Search")}</span>
            <input id="invReceiptsSearch" type="text" placeholder="${t("Beleg-Nr., Lieferant, Empfänger...", "Doc no., supplier, recipient...")}" />
          </label>

          <label>
            <span>${t("Sortierung", "Sort")}</span>
            <select id="invReceiptsSort">
              <option value="new">${t("Neueste zuerst", "Newest first")}</option>
              <option value="old">${t("Älteste zuerst", "Oldest first")}</option>
            </select>
          </label>
        </div>

        <div class="inv-rec-grid">
          <div class="inv-rec-list">
            <table>
              <thead>
                <tr>
                  <th>${t("Zeit", "Time")}</th>
                  <th>${t("Typ", "Type")}</th>
                  <th>${t("Nr.", "No.")}</th>
                  <th>${t("Partner", "Partner")}</th>
                  <th>${t("Pos.", "Lines")}</th>
                  <th>${t("Bearb.", "By")}</th>
                  <th>📎</th>
                </tr>
              </thead>
              <tbody id="invReceiptsTbody"></tbody>
            </table>
          </div>

          <div class="inv-rec-detail" id="invReceiptsDetail">
            <h4>${t("Beleg auswählen", "Select a receipt")}</h4>
            <div class="tiny" style="opacity:.85;">${t("Klicke links auf einen Eintrag.", "Click an entry on the left.")}</div>
          </div>
        </div>

        <div class="inv-rec-actions">
          <button type="button" class="btn btn-ghost" id="invReceiptsExportListCsv">⬇️ CSV (${t("Liste", "List")})</button>
          <button type="button" class="btn btn-ghost" id="invReceiptsRefresh">🔄 ${t("Aktualisieren", "Refresh")}</button>
          <button type="button" class="btn" id="invReceiptsClose">${t("Schließen", "Close")}</button>
        </div>
      </div>
    `;

    document.body.appendChild(dlg);

    // bind close
    const close = () => {
      try { dlg.close(); } catch (_) { dlg.removeAttribute("open"); }
    };
    dlg.querySelector("#invReceiptsClose")?.addEventListener("click", close);
    dlg.querySelector("#invReceiptsCloseX")?.addEventListener("click", close);

    // export list CSV
    dlg.querySelector("#invReceiptsExportListCsv")?.addEventListener("click", () => exportListCsv(dlg));

    // refresh button
    dlg.querySelector("#invReceiptsRefresh")?.addEventListener("click", () => renderList(dlg));

    // filter/search/sort
    const fKind = dlg.querySelector("#invReceiptsFilterKind");
    const fSearch = dlg.querySelector("#invReceiptsSearch");
    const fSort = dlg.querySelector("#invReceiptsSort");
    if (fKind) fKind.addEventListener("change", () => renderList(dlg));
    if (fSort) fSort.addEventListener("change", () => renderList(dlg));
    if (fSearch) {
      let to = null;
      fSearch.addEventListener("input", () => {
        clearTimeout(to);
        to = setTimeout(() => renderList(dlg), 120);
      });
    }

    return dlg;
  }

  function normalizeReceipt(r) {
    const kind = receiptKind(r);
    // Backwards compatibility: WE receipts had no kind
    if (!r.kind) r.kind = kind;
    // unify fields
    if (kind === "wareneingang") {
      r.docNo = r.deliveryNoteNo || r.docNo;
    }
    if (kind === "warenausgang") {
      r.deliveryNoteNo = r.deliveryNoteNo || "";
    }
    return r;
  }

  function renderList(dlg) {
    if (!dlg) return;
    const tbody = dlg.querySelector("#invReceiptsTbody");
    const detail = dlg.querySelector("#invReceiptsDetail");
    if (!tbody || !detail) return;

    const kindFilter = String(dlg.querySelector("#invReceiptsFilterKind")?.value || "").trim().toLowerCase();
    const q = String(dlg.querySelector("#invReceiptsSearch")?.value || "").trim().toLowerCase();
    const sort = String(dlg.querySelector("#invReceiptsSort")?.value || "new");

    let receipts = loadReceipts().map(normalizeReceipt);

    // filter
    if (kindFilter) receipts = receipts.filter((r) => receiptKind(r) === kindFilter);
    if (q) {
      receipts = receipts.filter((r) => {
        const hay = [
          receiptNo(r),
          receiptPartner(r),
          r.orderNo || "",
          r.note || "",
          r.reason || "",
          r.bookedBy || "",
          r.ts || ""
        ].join(" ").toLowerCase();
        return hay.includes(q);
      });
    }

    // sort
    receipts.sort((a, b) => {
      const ta = new Date(a.ts || 0).getTime();
      const tb = new Date(b.ts || 0).getTime();
      return sort === "old" ? ta - tb : tb - ta;
    });

    // cache for export
    dlg.__opsReceiptsView = receipts;

    tbody.innerHTML = "";

    if (!receipts.length) {
      dlg.__opsReceiptsView = [];
      tbody.innerHTML = `<tr><td colspan="7" style="opacity:.8; padding:14px;">${t("Keine Belege gefunden.", "No receipts found.")}</td></tr>`;
      detail.innerHTML = `<h4>${t("Keine Auswahl", "No selection")}</h4><div class="tiny" style="opacity:.85;">${t("Es gibt aktuell keine passenden Belege.", "There are currently no matching receipts.")}</div>`;
      return;
    }

    receipts.forEach((r, idx) => {
      const k = receiptKind(r);
      const no = receiptNo(r);
      const partner = receiptPartner(r);
      const by = String(r.bookedBy || "");
      const lines = Array.isArray(r.lines) ? r.lines.length : 0;
      const files = Array.isArray(r.attachments) ? r.attachments.length : 0;
      const tr = document.createElement("tr");
      tr.setAttribute("data-row", "1");
      tr.dataset.rid = String(r.id || idx);
      tr.innerHTML = `
        <td>${fmtDateTime(r.ts)}</td>
        <td>${kindLabel(k)}</td>
        <td><strong>${escapeHtml(no)}</strong>${r.stornoed ? " ↩️" : ""}</td>
        <td>${escapeHtml(partner)}</td>
        <td>${lines}</td>
        <td>${escapeHtml(by)}</td>
        <td>${files ? "📎 " + files : ""}</td>
      `;
      tr.addEventListener("click", () => renderDetail(dlg, r));
      tbody.appendChild(tr);
    });

    // auto open first
    renderDetail(dlg, receipts[0]);
  }

  function escapeHtml(s) {
    return String(s || "")
      .replace(/&/g, "&amp;")
      .replace(/</g, "&lt;")
      .replace(/>/g, "&gt;")
      .replace(/\"/g, "&quot;")
      .replace(/'/g, "&#039;");
  }

  function findArticle(articles, id) {
    return articles.find((a) => String(a.id) === String(id)) || null;
  }

  function renderDetail(dlg, receipt) {
    const detail = dlg.querySelector("#invReceiptsDetail");
    if (!detail) return;

    const r = normalizeReceipt(Object.assign({}, receipt));
    const k = receiptKind(r);
    const no = receiptNo(r);
    const partner = receiptPartner(r);
    const lines = Array.isArray(r.lines) ? r.lines : [];
    const files = Array.isArray(r.attachments) ? r.attachments : [];

    const articles = loadArticles();

    const metaRows = [];
    metaRows.push(`<div><strong>${t("Typ", "Type")}</strong>: ${escapeHtml(kindLabel(k))}</div>`);
    metaRows.push(`<div><strong>${t("Zeit", "Time")}</strong>: ${escapeHtml(fmtDateTime(r.ts))}</div>`);
    metaRows.push(`<div><strong>${t("Nr.", "No.")}</strong>: ${escapeHtml(no)}</div>`);

    if (k === "wareneingang") {
      metaRows.push(`<div><strong>${t("Lieferant", "Supplier")}</strong>: ${escapeHtml(String(r.supplier || ""))}</div>`);
      if (r.orderNo) metaRows.push(`<div><strong>${t("Auftrag/PO", "Order/PO")}</strong>: ${escapeHtml(String(r.orderNo))}</div>`);
      if (r.deliveryNoteDate) metaRows.push(`<div><strong>${t("LS-Datum", "DN date")}</strong>: ${escapeHtml(String(r.deliveryNoteDate))}</div>`);
    }
    if (k === "warenausgang") {
      if (partner) metaRows.push(`<div><strong>${t("Empfänger", "Recipient")}</strong>: ${escapeHtml(String(partner))}</div>`);
      if (r.issueDate) metaRows.push(`<div><strong>${t("Datum", "Date")}</strong>: ${escapeHtml(String(r.issueDate))}</div>`);
    }

    metaRows.push(`<div><strong>${t("Bearbeiter", "By")}</strong>: ${escapeHtml(String(r.bookedBy || ""))}</div>`);
    if (r.reason) metaRows.push(`<div><strong>${t("Grund", "Reason")}</strong>: ${escapeHtml(String(r.reason))}</div>`);
    if (r.note) metaRows.push(`<div style="grid-column:1/-1;"><strong>${t("Notiz", "Note")}</strong>: ${escapeHtml(String(r.note))}</div>`);

    // Storno-Infos
    if (k === "storno") {
      const refKind = String(r.refKind || "");
      const refNo = String(r.refNo || "");
      const refTxt = [refKind ? kindLabel(refKind) : "", refNo].filter(Boolean).join(" ");
      if (refTxt) metaRows.push(`<div style="grid-column:1/-1;"><strong>${t("Storno zu", "Reversal for")}</strong>: ${escapeHtml(refTxt)}</div>`);
      if (r.refReceiptId) metaRows.push(`<div style="grid-column:1/-1;"><strong>${t("Original-ID", "Original ID")}</strong>: ${escapeHtml(String(r.refReceiptId))}</div>`);
    } else if (r.stornoed) {
      metaRows.push(`<div style="grid-column:1/-1;"><strong>${t("Status", "Status")}</strong>: ↩️ ${t("storniert", "reversed")}</div>`);
      if (r.stornoReceiptId) metaRows.push(`<div style="grid-column:1/-1;"><strong>${t("Storno-Beleg", "Reversal receipt")}</strong>: ${escapeHtml(String(r.stornoReceiptId))}</div>`);
    }

    const lineRows = lines.map((l) => {
      const art = findArticle(articles, l.articleId);
      const nr = (art && (art.artikelnummer || art.interneArtikelnummer)) ? (art.artikelnummer || art.interneArtikelnummer) : "";
      const name = (art && (art.bezeichnung || art.hersteller)) ? (art.bezeichnung || art.hersteller) : "";
      const qty = l.amount != null ? l.amount : (l.qty != null ? l.qty : "");
      const unitPrice = (l.unitPrice != null && String(l.unitPrice).trim()) ? String(l.unitPrice) : "";
      return `
        <tr>
          <td>${escapeHtml(nr)}</td>
          <td>${escapeHtml(name)}</td>
          <td style="text-align:right; white-space:nowrap;">${escapeHtml(qty)}</td>
          <td style="text-align:right; white-space:nowrap;">${escapeHtml(unitPrice)}</td>
        </tr>
      `;
    }).join("");

    const fileBtns = files.map((a) => {
      const id = a && a.id ? String(a.id) : "";
      const name = a && a.name ? String(a.name) : t("Datei", "File");
      if (!id) return "";
      return `<button type="button" class="btn btn-ghost" data-file="${escapeHtml(id)}">📎 ${escapeHtml(name)}</button>`;
    }).join("");

    // cache selected receipt
    try { dlg.__selectedReceipt = r; } catch (_) {}

    const isStornoable = (k === "wareneingang" || k === "warenausgang");
    const stornoDisabled = !!r.stornoed;
    const stornoBtnHtml = isStornoable
      ? `<button type="button" class="btn btn-ghost" id="invReceiptsStornoBtn"${stornoDisabled ? " disabled" : ""}>↩️ ${t("Stornieren", "Reverse")}</button>`
      : "";

detail.innerHTML = `
      <h4>${escapeHtml(kindLabel(k))}: ${escapeHtml(no)}</h4>
      <div class="inv-rec-meta">${metaRows.join("")}</div>

      <div class="inv-rec-actions" style="justify-content:flex-start; flex-wrap:wrap;">
        <button type="button" class="btn" id="invReceiptsPrintBtn">🖨️ ${t("Drucken", "Print")}</button>
        <button type="button" class="btn btn-ghost" id="invReceiptsExportReceiptPdfBtn">⬇️ PDF</button>
        <button type="button" class="btn btn-ghost" id="invReceiptsExportReceiptCsvBtn">⬇️ CSV</button>
        ${stornoBtnHtml}
        <button type="button" class="btn btn-ghost" id="invReceiptsDeleteBtn">🗑️ ${t("Entfernen", "Remove")}</button>
      </div>

      <div class="inv-rec-lines">
        <table>
          <thead>
            <tr>
              <th>${t("Artikel", "Item")}</th>
              <th>${t("Bezeichnung", "Name")}</th>
              <th style="text-align:right;">${t("Menge", "Qty")}</th>
              <th style="text-align:right;">${t("Preis", "Price")}</th>
            </tr>
          </thead>
          <tbody>
            ${lineRows || `<tr><td colspan="4" style="opacity:.8; padding:10px;">${t("Keine Positionen", "No lines")}</td></tr>`}
          </tbody>
        </table>
      </div>

      <div class="inv-rec-files">${fileBtns}</div>
    `;

    // bind print
    detail.querySelector("#invReceiptsPrintBtn")?.addEventListener("click", () => {
      printReceipt(r);
    });

    // export receipt
    detail.querySelector("#invReceiptsExportReceiptCsvBtn")?.addEventListener("click", () => {
      exportReceiptCsv(r);
    });
    detail.querySelector("#invReceiptsExportReceiptPdfBtn")?.addEventListener("click", () => {
      exportReceiptPdf(r);
    });

    // storno / delete
    detail.querySelector("#invReceiptsStornoBtn")?.addEventListener("click", () => {
      stornoReceipt(dlg, r);
    });
 detail.querySelector("#invReceiptsDeleteBtn")?.addEventListener("click", async () => {
  await removeReceipt(dlg, r);
});


    // bind attachments
    Array.from(detail.querySelectorAll("button[data-file]"))
      .forEach((btn) => {
        btn.addEventListener("click", () => {
          const fid = btn.getAttribute("data-file");
          if (fid) openStoredFile(fid);
        });
      });
  }

  function printReceipt(r) {
    try {
      const k = receiptKind(r);
      const title = (k === "wareneingang") ? t("Wareneingang", "Goods receipt") : (k === "warenausgang" ? t("Warenausgang", "Goods issue") : (k === "storno" ? t("Storno", "Reversal") : t("Beleg", "Receipt")));
      const no = receiptNo(r);
      const partner = receiptPartner(r);
      const lines = Array.isArray(r.lines) ? r.lines : [];
      const articles = loadArticles();

      const rows = lines.map((l) => {
        const art = findArticle(articles, l.articleId);
        const nr = (art && (art.artikelnummer || art.interneArtikelnummer)) ? (art.artikelnummer || art.interneArtikelnummer) : "";
        const name = (art && (art.bezeichnung || art.hersteller)) ? (art.bezeichnung || art.hersteller) : "";
        const qty = l.amount != null ? l.amount : (l.qty != null ? l.qty : "");
        const unitPrice = (l.unitPrice != null && String(l.unitPrice).trim()) ? String(l.unitPrice) : "";
        return `<tr><td>${escapeHtml(nr)}</td><td>${escapeHtml(name)}</td><td style="text-align:right;">${escapeHtml(qty)}</td><td style="text-align:right;">${escapeHtml(unitPrice)}</td></tr>`;
      }).join("");

      const meta = [];
      meta.push(`<div><strong>${escapeHtml(title)}</strong></div>`);
      meta.push(`<div>${t("Nr.", "No.")}: <strong>${escapeHtml(no)}</strong></div>`);
      meta.push(`<div>${t("Zeit", "Time")}: ${escapeHtml(fmtDateTime(r.ts))}</div>`);
      if (k === "wareneingang") {
        if (r.supplier) meta.push(`<div>${t("Lieferant", "Supplier")}: ${escapeHtml(String(r.supplier))}</div>`);
        if (r.orderNo) meta.push(`<div>${t("Auftrag/PO", "Order/PO")}: ${escapeHtml(String(r.orderNo))}</div>`);
        if (r.deliveryNoteDate) meta.push(`<div>${t("LS-Datum", "DN date")}: ${escapeHtml(String(r.deliveryNoteDate))}</div>`);
      }
      if (k === "warenausgang") {
        if (partner) meta.push(`<div>${t("Empfänger", "Recipient")}: ${escapeHtml(String(partner))}</div>`);
        if (r.issueDate) meta.push(`<div>${t("Datum", "Date")}: ${escapeHtml(String(r.issueDate))}</div>`);
      }
      if (r.bookedBy) meta.push(`<div>${t("Bearbeiter", "By")}: ${escapeHtml(String(r.bookedBy))}</div>`);
      if (r.reason) meta.push(`<div>${t("Grund", "Reason")}: ${escapeHtml(String(r.reason))}</div>`);
      if (r.note) meta.push(`<div>${t("Notiz", "Note")}: ${escapeHtml(String(r.note))}</div>`);

      const html = `
        <!doctype html>
        <html><head><meta charset="utf-8"><title>${escapeHtml(title)} ${escapeHtml(no)}</title>
        <style>
          body{ font-family: system-ui, -apple-system, Segoe UI, Roboto, Arial, sans-serif; padding:20px; }
          h1{ font-size:18px; margin:0 0 8px; }
          .meta{ font-size:12px; opacity:.95; display:grid; grid-template-columns:1fr 1fr; gap:6px 16px; margin:10px 0 14px; }
          table{ width:100%; border-collapse:collapse; font-size:12px; }
          th,td{ border:1px solid #ddd; padding:8px; }
          th{ background:#f3f3f3; text-align:left; }
        </style>
        </head><body>
          <h1>${escapeHtml(title)} – ${escapeHtml(no)}</h1>
          <div class="meta">${meta.map((m) => `<div>${m}</div>`).join("")}</div>
          <table>
            <thead><tr><th>${t("Artikel", "Item")}</th><th>${t("Bezeichnung", "Name")}</th><th style="text-align:right;">${t("Menge", "Qty")}</th><th style="text-align:right;">${t("Preis", "Price")}</th></tr></thead>
            <tbody>${rows || `<tr><td colspan="4">${t("Keine Positionen", "No lines")}</td></tr>`}</tbody>
          </table>
          <script>window.onload=()=>{ window.print(); };</script>
        </body></html>
      `;

      const w = window.open("", "_blank");
      if (!w) {
        alert(t("Popup blockiert – bitte Popups erlauben.", "Popup blocked – please allow popups."));
        return;
      }
      w.document.open();
      w.document.write(html);
      w.document.close();
    } catch (e) {
      console.warn("[Belegliste] printReceipt failed:", e);
    }
  }


  // ---- Export / Storno / Remove ---------------------------------------
  function nowIso() { return new Date().toISOString(); }

  function sleep(ms) { return new Promise((r) => setTimeout(r, ms)); }

  async function waitForDialogOpen(id, timeoutMs = 4000) {
    const start = Date.now();
    while (Date.now() - start < timeoutMs) {
      const dlg = document.getElementById(id);
      if (dlg && dlg.open) return dlg;
      await sleep(25);
    }
    return null;
  }

  async function waitForDialogClose(id, timeoutMs = 8000) {
    const start = Date.now();
    while (Date.now() - start < timeoutMs) {
      const dlg = document.getElementById(id);
      if (dlg && !dlg.open) return true;
      await sleep(25);
    }
    return false;
  }

  function dispatchInput(el) {
    if (!el) return;
    el.dispatchEvent(new Event("input", { bubbles: true }));
    el.dispatchEvent(new Event("change", { bubbles: true }));
  }

  function pickAllValue(sel) {
    if (!sel || !sel.options || !sel.options.length) return null;
    const opts = Array.from(sel.options);
    const preferred = opts.find((o) => o.value === "" || o.value === "all");
    return (preferred ? preferred.value : opts[0].value);
  }

  function snapshotFilters() {
    return {
      search: document.getElementById("invSearch")?.value ?? null,
      lager: document.getElementById("invFilterLager")?.value ?? null,
      kat: document.getElementById("invFilterKategorie")?.value ?? null,
      bestand: document.getElementById("invFilterBestand")?.value ?? null,
      sperre: document.getElementById("invFilterSperre")?.value ?? null,
    };
  }

  function setFiltersAll() {
    const search = document.getElementById("invSearch");
    if (search) {
      search.value = "";
      dispatchInput(search);
    }

    const lager = document.getElementById("invFilterLager");
    const kat = document.getElementById("invFilterKategorie");
    const bestand = document.getElementById("invFilterBestand");
    const sperre = document.getElementById("invFilterSperre");

    for (const sel of [lager, kat, bestand, sperre]) {
      if (!sel) continue;
      const v = pickAllValue(sel);
      if (v != null) {
        sel.value = v;
        sel.dispatchEvent(new Event("change", { bubbles: true }));
      }
    }
  }

  function restoreFilters(snap) {
    if (!snap) return;
    const search = document.getElementById("invSearch");
    if (search && snap.search != null) {
      search.value = snap.search;
      dispatchInput(search);
    }

    const mapping = [
      ["invFilterLager", snap.lager],
      ["invFilterKategorie", snap.kat],
      ["invFilterBestand", snap.bestand],
      ["invFilterSperre", snap.sperre],
    ];

    for (const [id, val] of mapping) {
      const sel = document.getElementById(id);
      if (sel && val != null) {
        sel.value = val;
        sel.dispatchEvent(new Event("change", { bubbles: true }));
      }
    }
  }

  function downloadText(filename, mime, text) {
    try {
      const blob = new Blob([text], { type: mime || "text/plain;charset=utf-8" });
      const url = URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = filename || "export.txt";
      document.body.appendChild(a);
      a.click();
      a.remove();
      setTimeout(() => URL.revokeObjectURL(url), 30000);
    } catch (e) {
      console.warn("[Belegliste] downloadText failed:", e);
    }
  }

  function csvEsc(v, sep = ";") {
    const s = String(v ?? "");
    const needs = s.includes('"') || s.includes("\n") || s.includes("\r") || s.includes(sep);
    const out = s.replace(/"/g, '""');
    return needs ? `"${out}"` : out;
  }

  function exportListCsv(dlg) {
    try {
      const receipts = (dlg && Array.isArray(dlg.__opsReceiptsView)) ? dlg.__opsReceiptsView : loadReceipts().map(normalizeReceipt);
      const sep = ";";
      const rows = [];
      rows.push("sep=" + sep);
      rows.push([
        t("Zeit", "Time"),
        t("Typ", "Type"),
        t("Nr.", "No."),
        t("Partner", "Partner"),
        t("Pos.", "Lines"),
        t("Bearb.", "By"),
        t("Grund", "Reason"),
        t("Notiz", "Note"),
        "id",
        t("Storniert", "Reversed"),
        t("Referenz", "Reference"),
      ].join(sep));

      receipts.forEach((r) => {
        const k = receiptKind(r);
        const ref = k === "storno" ? [r.refKind ? kindLabel(String(r.refKind)) : "", r.refNo || ""].filter(Boolean).join(" ") : "";
        rows.push([
          csvEsc(fmtDateTime(r.ts), sep),
          csvEsc(kindLabel(k), sep),
          csvEsc(receiptNo(r), sep),
          csvEsc(receiptPartner(r), sep),
          csvEsc((Array.isArray(r.lines) ? r.lines.length : 0), sep),
          csvEsc(r.bookedBy || "", sep),
          csvEsc(r.reason || "", sep),
          csvEsc(r.note || "", sep),
          csvEsc(r.id || "", sep),
          csvEsc(r.stornoed ? "1" : "", sep),
          csvEsc(ref, sep),
        ].join(sep));
      });

      const fn = `belegliste_${new Date().toISOString().slice(0,10)}.csv`;
      downloadText(fn, "text/csv;charset=utf-8", rows.join("\n"));
    } catch (e) {
      console.warn("[Belegliste] exportListCsv failed:", e);
      uiAlertCompat("CSV-Export fehlgeschlagen.", "CSV export failed.", "⚠️ Fehler", "⚠️ Error");
    }
  }

  function exportReceiptCsv(r) {
    try {
      const k = receiptKind(r);
      const sep = ";";
      const rows = [];
      rows.push("sep=" + sep);

      rows.push([t("Feld", "Field"), t("Wert", "Value")].join(sep));
      rows.push([csvEsc(t("Typ", "Type"), sep), csvEsc(kindLabel(k), sep)].join(sep));
      rows.push([csvEsc(t("Nr.", "No."), sep), csvEsc(receiptNo(r), sep)].join(sep));
      rows.push([csvEsc(t("Zeit", "Time"), sep), csvEsc(fmtDateTime(r.ts), sep)].join(sep));
      if (receiptPartner(r)) rows.push([csvEsc(t("Partner", "Partner"), sep), csvEsc(receiptPartner(r), sep)].join(sep));
      if (r.bookedBy) rows.push([csvEsc(t("Bearbeiter", "By"), sep), csvEsc(r.bookedBy, sep)].join(sep));
      if (r.reason) rows.push([csvEsc(t("Grund", "Reason"), sep), csvEsc(r.reason, sep)].join(sep));
      if (r.note) rows.push([csvEsc(t("Notiz", "Note"), sep), csvEsc(r.note, sep)].join(sep));
      if (k === "storno") {
        const refTxt = [r.refKind ? kindLabel(String(r.refKind)) : "", r.refNo || ""].filter(Boolean).join(" ");
        if (refTxt) rows.push([csvEsc(t("Storno zu", "Reversal for"), sep), csvEsc(refTxt, sep)].join(sep));
      }
      rows.push("");

      rows.push([
        t("Artikel", "Item"),
        t("Bezeichnung", "Name"),
        t("Menge", "Qty"),
        t("Preis", "Price"),
      ].join(sep));

      const articles = loadArticles();
      const lines = Array.isArray(r.lines) ? r.lines : [];
      lines.forEach((l) => {
        const art = findArticle(articles, l.articleId);
        const nr = (art && (art.artikelnummer || art.interneArtikelnummer)) ? (art.artikelnummer || art.interneArtikelnummer) : "";
        const name = (art && (art.bezeichnung || art.hersteller)) ? (art.bezeichnung || art.hersteller) : "";
        const qty = l.amount != null ? l.amount : (l.qty != null ? l.qty : "");
        const unitPrice = (l.unitPrice != null && String(l.unitPrice).trim()) ? String(l.unitPrice) : "";
        rows.push([
          csvEsc(nr, sep),
          csvEsc(name, sep),
          csvEsc(qty, sep),
          csvEsc(unitPrice, sep),
        ].join(sep));
      });

      const base = `${receiptKind(r)}_${receiptNo(r) || r.id || "beleg"}`.replace(/[^a-z0-9_\-]+/gi, "_");
      downloadText(`${base}.csv`, "text/csv;charset=utf-8", rows.join("\n"));
    } catch (e) {
      console.warn("[Belegliste] exportReceiptCsv failed:", e);
      uiAlertCompat("CSV-Export fehlgeschlagen.", "CSV export failed.", "⚠️ Fehler", "⚠️ Error");
    }
  }

  function exportReceiptPdf(r) {
    // Browser-PDF: über den Druckdialog "Als PDF speichern"
    printReceipt(r);
  }

	async function removeReceipt(dlg, r) {
    try {
      const k = receiptKind(r);
      const no = receiptNo(r);
      const msg = [
        `${t("Beleg entfernen?", "Remove receipt?")}`,
        "",
        `${kindLabel(k)}: ${no}`,
        "",
        t("Hinweis: Das entfernt nur den Eintrag aus der Belegliste. Bestandsbuchungen bleiben unverändert.", "Note: This only removes the entry from the receipts list. Stock bookings remain unchanged."),
      ].join("\n");

     const ok = await uiConfirmCompat(
  msg,
  msg,
  "⚠️ Bestätigen",
  "⚠️ Confirm"
);
if (!ok) return;


      const receipts = loadReceipts().map(normalizeReceipt);

      const rid = String(r.id || "");
      const kept = receipts.filter((x) => String(x.id || "") !== rid);

      // Falls ein anderer Beleg auf diesen Storno-Beleg verweist: Verweis entfernen
      kept.forEach((x) => {
        if (String(x.stornoReceiptId || "") === rid) {
          x.stornoed = false;
          delete x.stornoReceiptId;
          delete x.stornoTs;
        }
      });

      saveReceipts(kept);
      renderList(dlg);
    } catch (e) {
      console.warn("[Belegliste] removeReceipt failed:", e);
      alert(t("Entfernen fehlgeschlagen.", "Remove failed."));
    }
  }

  async function postStornoLine({ articleId, qty, actionValue, employee, noteText, stornoReceipt }) {
    // ensure row exists
    const actionSelect = document.querySelector(`select.inv-action-select[data-id="${CSS.escape(String(articleId))}"]`);
    if (!actionSelect) {
      return { ok: false, error: t("Artikel in Tabelle nicht gefunden (Filter?)", "Item not found in table (filters?)") };
    }

    // open booking dialog via core by triggering action select
    window.__opsWeBypass = true;
    window.__opsWoBypass = true;

    actionSelect.value = actionValue;
    actionSelect.dispatchEvent(new Event("change", { bubbles: true }));

    window.__opsWeBypass = false;
    window.__opsWoBypass = false;

    const dlg = await waitForDialogOpen("invBookingDialog", 4000);
    if (!dlg) {
      return { ok: false, error: t("Buchungsdialog konnte nicht geöffnet werden.", "Booking dialog could not be opened.") };
    }

    // fill booking fields
    const amountEl = document.getElementById("invBookingAmount");
    const employeeEl = document.getElementById("invBookingEmployee");
    const noteEl = document.getElementById("invBookingNote");
    const reasonEl = document.getElementById("invBookingReason");

    if (amountEl) amountEl.value = String(qty);
    if (employeeEl) employeeEl.value = employee || "";
    if (noteEl) noteEl.value = noteText || "";

    // keep storno reason as "korrektur" (works for in/out, and avoids WE LS-check)
    if (reasonEl) {
      const has = Array.from(reasonEl.options || []).some((o) => o.value === "korrektur");
      if (has) {
        reasonEl.value = "korrektur";
        reasonEl.dispatchEvent(new Event("change", { bubbles: true }));
      }
    }

    // submit form
    const form = document.getElementById("invBookingForm");
    const before = loadBookings();
    const beforeLen = before.length;

    if (form && typeof form.requestSubmit === "function") form.requestSubmit();
    else if (form) form.dispatchEvent(new Event("submit", { bubbles: true, cancelable: true }));

    const closed = await waitForDialogClose("invBookingDialog", 8000);
    if (!closed) return { ok: false, error: t("Buchung konnte nicht abgeschlossen werden.", "Booking could not be completed.") };

    const after = loadBookings();
    const added = after.slice(beforeLen);
    if (!added.length) {
      return { ok: false, error: t("Keine Buchung erzeugt – evtl. wurde abgebrochen.", "No booking created – may have been cancelled.") };
    }

    const addedIds = [];
    for (let i = after.length - added.length; i < after.length; i++) {
      const b = after[i];
      if (!b || typeof b !== "object") continue;

      b.receiptId = stornoReceipt.id;
      b.stornoRefReceiptId = stornoReceipt.refReceiptId;
      b.stornoRefNo = stornoReceipt.refNo;
      b.stornoRefKind = stornoReceipt.refKind;
      b.receiptReason = stornoReceipt.reason;
      b.receiptBookedBy = stornoReceipt.bookedBy;

      if (b.id != null) addedIds.push(b.id);
    }
    saveBookings(after);

    return { ok: true, bookingIds: addedIds };
  }

  async function stornoReceipt(dlg, receipt) {
    const r = normalizeReceipt(Object.assign({}, receipt));
    const k = receiptKind(r);
    const no = receiptNo(r);

    if (!(k === "wareneingang" || k === "warenausgang")) {
    uiAlertCompat("Dieser Beleg kann nicht storniert werden.", "This receipt cannot be reversed.", "ℹ️ Hinweis", "ℹ️ Info");
      return;
    }
    if (r.stornoed) {
     uiAlertCompat("Dieser Beleg wurde bereits storniert.", "This receipt has already been reversed.", "ℹ️ Hinweis", "ℹ️ Info");
      return;
    }

  const ok = await uiConfirmCompat(
  `${t("Beleg stornieren?", "Reverse receipt?")}\n\n${kindLabel(k)}: ${no}\n\n` +
  t("Es werden Gegenbuchungen erzeugt und der Bestand entsprechend zurückgedreht.", "Counter-bookings will be created and stock will be reverted accordingly."),
  `${t("Beleg stornieren?", "Reverse receipt?")}\n\n${kindLabel(k)}: ${no}\n\n` +
  t("Es werden Gegenbuchungen erzeugt und der Bestand entsprechend zurückgedreht.", "Counter-bookings will be created and stock will be reverted accordingly."),
  "⚠️ Storno",
  "⚠️ Reversal"
);
if (!ok) return;


    const bookedByRaw = await uiPromptCompat(
  "Name/Kürzel für Storno:",
  "Name/initials for reversal:",
  String(r.bookedBy || ""),
  "✍️ Storno",
  "✍️ Reversal",
  "Name/Kürzel",
  "Name/initials"
);
const bookedBy = String(bookedByRaw || "").trim();


    const defaultNote = `${t("Storno zu", "Reversal for")} ${kindLabel(k)} ${no}`;
   const noteRaw = await uiPromptCompat(
  "Notiz (optional):",
  "Note (optional):",
  defaultNote,
  "📝 Notiz",
  "📝 Note",
  "Notiz",
  "Note"
);
const note = String(noteRaw ?? "").trim();


    const sid = `st_${new Date().toISOString().slice(0,10)}_${Date.now().toString(36)}`;
    const storno = {
      id: sid,
      kind: "storno",
      ts: nowIso(),
      docNo: `STO-${no}`,
      refReceiptId: String(r.id || ""),
      refKind: k,
      refNo: no,
      refPartner: receiptPartner(r),
      partner: receiptPartner(r),
      bookedBy: bookedBy,
      reason: "Storno",
      note: note,
      lines: Array.isArray(r.lines) ? r.lines.map((l) => ({ articleId: l.articleId, amount: l.amount, unitPrice: l.unitPrice })) : [],
      bookingIds: [],
    };

    if (!storno.lines.length) {
      alert(t("Keine Positionen gefunden – Storno nicht möglich.", "No line items found – cannot reverse."));
      return;
    }

    // ensure rows are visible (filters)
    const snap = snapshotFilters();
    setFiltersAll();
    await sleep(30);

    try {
      const actionValue = (k === "wareneingang") ? "goods-out" : "book-in"; // WE -> raus; WA -> rein
      for (let i = 0; i < storno.lines.length; i++) {
        const l = storno.lines[i];
        const noteText = [
          `STORNO`,
          `${kindLabel(k)} ${no}`,
          note ? note : ""
        ].filter(Boolean).join(" | ");

        const res = await postStornoLine({
          articleId: l.articleId,
          qty: l.amount,
          actionValue,
          employee: bookedBy,
          noteText,
          stornoReceipt: storno,
        });

        if (!res.ok) throw new Error(res.error || "storno failed");
        if (res.bookingIds && res.bookingIds.length) storno.bookingIds.push(...res.bookingIds);
      }

      // update receipts: mark original stornoed + add storno receipt
      const receipts = loadReceipts().map(normalizeReceipt);
      const rid = String(r.id || "");
      const idx = receipts.findIndex((x) => String(x.id || "") === rid);
      if (idx >= 0) {
        receipts[idx].stornoed = true;
        receipts[idx].stornoReceiptId = storno.id;
        receipts[idx].stornoTs = storno.ts;
      }
      receipts.push(storno);
      saveReceipts(receipts);

      renderList(dlg);

     uiAlertCompat("Storno erfolgreich gebucht.", "Reversal posted successfully.", "✅ Fertig", "✅ Done");
    } catch (e) {
      console.error("[Belegliste] storno failed:", e);
   if (window.OPS_UI && window.OPS_UI.alert) {
  window.OPS_UI.alert(
    "Storno fehlgeschlagen: " + String(e && e.message ? e.message : e),
    "Reversal failed: " + String(e && e.message ? e.message : e),
    "⚠️ Fehler",
    "⚠️ Error"
  );
} else {
  alert(t("Storno fehlgeschlagen: ", "Reversal failed: ") + String(e && e.message ? e.message : e));
}

    } finally {
      restoreFilters(snap);
    }
  }

  function openDialog() {
    const dlg = ensureDialog();
    renderList(dlg);
    try {
      dlg.showModal();
    } catch (_) {
      dlg.setAttribute("open", "");
    }
  }

   // ---- Boot ------------------------------------------------------------
  function boot() {
    try {
      if (!window.__opsReceiptsAddonBound) {
        window.__opsReceiptsAddonBound = true;

        // open via event
        window.addEventListener("opsReceiptsOpenDialog", () => {
          openDialog();
        });
      }

      window.__opsReceiptsAddon = { version: "v1", ts: Date.now() };
    } catch (e) {
      console.error("[Belegliste] boot failed:", e);
    }
  }

  if (document.readyState === "loading") {
  document.addEventListener("DOMContentLoaded", boot, { once: true });
  } else {
    boot();
  }
})();

