/* ops-demo-bootstrap.js
   Demo mode:
   - No persistent saving (localStorage is in-memory)
   - Mocked /api endpoints (login/me/rbac/files) so everything is clickable without a backend
   - Reset on every reload/start
*/
(() => {
  "use strict";
  if (window.__opsDemoBootstrapLoaded) return;
  window.__opsDemoBootstrapLoaded = true;

  window.OPS_DEMO = true;

  // ---- Helpers ----------------------------------------------------------
  const ALIVE_TOKEN = "__alive";
  const TRANSPARENT_GIF =
    "data:image/gif;base64,R0lGODlhAQABAIAAAAAAAP///ywAAAAAAQABAAACAUwAOw==";

  function strUrl(u) {
    if (typeof u === "string") return u;
    if (u && typeof u === "object") {
      if (typeof u.url === "string") return u.url;
      try { return String(u); } catch (_) {}
    }
    return String(u || "");
  }

  function isAlive(u) {
    const s = strUrl(u);
    return s.includes(ALIVE_TOKEN);
  }

  // ---- In-memory localStorage (reset on reload) -------------------------
  const store = new Map();
  const ls = window.localStorage;

  const toKey = (k) => String(k);
  const toVal = (v) => String(v);

  function keysArr() { return Array.from(store.keys()); }

  try {
    ls.getItem = (k) => store.has(toKey(k)) ? store.get(toKey(k)) : null;
    ls.setItem = (k, v) => { store.set(toKey(k), toVal(v)); };
    ls.removeItem = (k) => { store.delete(toKey(k)); };
    ls.clear = () => { store.clear(); };
    ls.key = (i) => keysArr()[Number(i)] ?? null;
    Object.defineProperty(ls, "length", { get: () => store.size });
  } catch (e) {
    // If the browser disallows patching, we still do best-effort cleanup
    try { window.localStorage.clear(); } catch (_) {}
  }

  // ---- STOP __alive 404 spam (Image/XHR/fetch) --------------------------
  // 1) Image ping: new Image().src = "/?__alive?ts=..."
  try {
    const desc = Object.getOwnPropertyDescriptor(HTMLImageElement.prototype, "src");
    if (desc && typeof desc.set === "function") {
      Object.defineProperty(HTMLImageElement.prototype, "src", {
        configurable: true,
        enumerable: desc.enumerable,
        get: desc.get,
        set: function(v) {
          if (isAlive(v)) return desc.set.call(this, TRANSPARENT_GIF);
          return desc.set.call(this, v);
        }
      });
    }
  } catch (_) {}

  // 2) XHR ping (falls verwendet)
  try {
    const XHROpen = XMLHttpRequest.prototype.open;
    XMLHttpRequest.prototype.open = function(method, url, ...rest) {
      if (isAlive(url)) {
        // statt 404 -> "/" (liefert 200 und erzeugt kein Error-Spam)
        url = "/";
      }
      return XHROpen.call(this, method, url, ...rest);
    };
  } catch (_) {}

  // ---- Mock API ---------------------------------------------------------
  const realFetch = window.fetch ? window.fetch.bind(window) : null;

  const DEMO_PERMS = [
    "article.create",
    "goods_in.create",
    "goods_out.create",
    "history.view",
    "inventory.count",
    "inventory.delete_all",
    "inventory.export",
    "inventory.import",
    "inventory.print",
    "receipts.view",
    "roles.manage",
    "ui.columns"
  ];

  const demoState = {
    loggedIn: true,
    user: {
      username: "demo",
      display_name: "Demo User",
      roles: ["admin"],
      permissions: DEMO_PERMS
    },
    rbac: {
      roles: [
        { key: "admin", name: "Admin" },
        { key: "user", name: "User" }
      ],
      users: [
        { username: "demo", display_name: "Demo User", roles: ["admin"] }
      ],
      matrix: {
        "admin": DEMO_PERMS.reduce((acc, p) => (acc[p]=true, acc), {}),
        "user":  DEMO_PERMS.reduce((acc, p) => (acc[p]= (p !== "roles.manage"), acc), {})
      }
    },
    files: new Map() // id -> { blob, name, type, size, ts, kind }
  };

  function jsonResponse(obj, status=200, extraHeaders={}) {
    return new Response(JSON.stringify(obj), {
      status,
      headers: Object.assign({ "content-type": "application/json" }, extraHeaders)
    });
  }
  function textResponse(text, status=200, extraHeaders={}) {
    return new Response(text, {
      status,
      headers: Object.assign({ "content-type": "text/plain; charset=utf-8" }, extraHeaders)
    });
  }

  async function handleApi(url, opts) {
    const u = strUrl(url);
    const method = (opts && opts.method ? String(opts.method) : "GET").toUpperCase();

    // ✅ fetch("__alive...") abfangen
    if (isAlive(u)) {
      return new Response(null, {
        status: 204,
        headers: { "cache-control": "no-store" }
      });
    }

    // normalize to path
    let path = u;
    try {
      const uu = new URL(u, window.location.href);
      path = uu.pathname + (uu.search || "");
    } catch (_) {}

    // auch nach Normalisierung nochmal abfangen (z.B. "/?__alive?ts=...")
    if (path.includes(ALIVE_TOKEN)) {
      return new Response(null, {
        status: 204,
        headers: { "cache-control": "no-store" }
      });
    }

    // --- Auth & Me ---
    if (path === "/api/me" || path === "/api/me/") {
      if (!demoState.loggedIn) return jsonResponse({ error: "unauthorized" }, 401);
      return jsonResponse(demoState.user, 200);
    }

    if (path === "/api/auth/login" || path === "/api/auth/login/") {
      demoState.loggedIn = true;
      return jsonResponse({ ok: true }, 200);
    }

    if (path === "/api/auth/logout" || path === "/api/auth/logout/") {
      demoState.loggedIn = false;
      return jsonResponse({ ok: true }, 200);
    }

    // --- RBAC (admin dialogs) ---
    if (path === "/api/rbac/roles") return jsonResponse(demoState.rbac.roles, 200);
    if (path === "/api/rbac/users") return jsonResponse(demoState.rbac.users, 200);
    if (path === "/api/rbac/matrix") return jsonResponse(demoState.rbac.matrix, 200);

    if (path === "/api/rbac/roles" && method === "POST") {
      let body = {};
      try { body = JSON.parse(opts.body || "{}"); } catch (_) {}
      if (body && body.key) demoState.rbac.roles.push({ key: String(body.key), name: String(body.name || body.key) });
      return jsonResponse({ ok: true }, 200);
    }

    if (path === "/api/rbac/users" && method === "POST") {
      let body = {};
      try { body = JSON.parse(opts.body || "{}"); } catch (_) {}
      if (body && body.username) demoState.rbac.users.push({
        username: String(body.username),
        display_name: String(body.display_name || body.username),
        roles: Array.isArray(body.roles) ? body.roles.map(String) : []
      });
      return jsonResponse({ ok: true }, 200);
    }

    if (path === "/api/rbac/matrix" && method === "POST") {
      let body = {};
      try { body = JSON.parse(opts.body || "{}"); } catch (_) {}
      if (body && body.roleKey && body.permKey) {
        const r = String(body.roleKey), p = String(body.permKey);
        demoState.rbac.matrix[r] = demoState.rbac.matrix[r] || {};
        demoState.rbac.matrix[r][p] = !!body.value;
      }
      return jsonResponse({ ok: true }, 200);
    }

    // --- File upload used by Lagerverwaltung ---
    if (path === "/api/files/put" && method === "POST") {
      try {
        const fd = opts && opts.body instanceof FormData ? opts.body : null;
        if (!fd) return jsonResponse({ ok: false, error: "expected formdata" }, 400);

        const id = String(fd.get("id") || ("file_" + Math.random().toString(36).slice(2)));
        const kind = String(fd.get("kind") || "");
        const ts = String(fd.get("ts") || "");
        const name = String(fd.get("name") || "file");
        const type = String(fd.get("type") || "");
        const size = Number(fd.get("size") || 0) || 0;
        const file = fd.get("file");

        if (file && typeof file.arrayBuffer === "function") {
          demoState.files.set(id, { blob: file, name, type, size, ts, kind });
          return jsonResponse({ ok: true, id }, 200);
        }
        return jsonResponse({ ok: false, error: "no file" }, 400);
      } catch (e) {
        return jsonResponse({ ok: false, error: String(e && e.message ? e.message : e) }, 500);
      }
    }

    // --- Admin backup (not available in demo) ---
    if (path === "/api/admin/backup") {
      return textResponse("DEMO: Backup is disabled.", 501);
    }

    // Block any other /api call to avoid hanging on a missing backend
    if (path.startsWith("/api/")) {
      return jsonResponse({ error: "DEMO_NO_BACKEND", path }, 404);
    }

    return null;
  }

  if (realFetch) {
    window.fetch = async (url, opts) => {
      const mocked = await handleApi(url, opts || {});
      if (mocked) return mocked;
      return realFetch(url, opts);
    };
  }

})();
