// ops-demo-dashboard-viz-addon.js
// Demo-only visual enhancements for KPI cards (no core logic changes).
// - Mini chart in "Artikel" card
// - Click on KPI cards toggles stock-status filter
// - Mini monthly movement chart in "Materialwert" card (updates when booking changes stock)
// Safe: only reads DOM and triggers existing filter dropdown.
(() => {
  "use strict";
  if (window.__opsDemoDashVizLoaded) return;
  window.__opsDemoDashVizLoaded = true;

  const LS_KEY = "ops_demo_metrics_v1";

  const $ = (sel, root = document) => root.querySelector(sel);
  const $$ = (sel, root = document) => Array.from(root.querySelectorAll(sel));

  function toNum(v) {
    const s = String(v ?? "").replace(/\./g, "").replace(",", ".").trim();
    const n = Number(s);
    return Number.isFinite(n) ? n : 0;
  }

  function monthKey(d = new Date()) {
    const y = d.getFullYear();
    const m = String(d.getMonth() + 1).padStart(2, "0");
    return `${y}-${m}`;
  }

  function loadMetrics() {
    try {
      const raw = localStorage.getItem(LS_KEY);
      if (!raw) return { months: {} };
      const obj = JSON.parse(raw);
      if (!obj || typeof obj !== "object") return { months: {} };
      if (!obj.months) obj.months = {};
      return obj;
    } catch (_) {
      return { months: {} };
    }
  }

  function saveMetrics(obj) {
    try { localStorage.setItem(LS_KEY, JSON.stringify(obj)); } catch (_) {}
  }

  function ensureCss() {
    if ($("#opsDemoDashVizCss")) return;
    const st = document.createElement("style");
    st.id = "opsDemoDashVizCss";
    st.textContent = `
      .kpi-card{ position: relative; }
      .kpi-card.ops-kpi-clickable{ cursor:pointer; }
      .kpi-card.ops-kpi-clickable:hover{ outline: 1px solid rgba(56,189,248,0.35); }
      .ops-kpi-mini{
        position:absolute;
        right:12px;
        bottom:12px;
        width:110px;
        height:44px;
        opacity:.92;
        pointer-events:none;
      }
      .ops-kpi-mini svg{ width:100%; height:100%; display:block; }
      .ops-kpi-hint{
        position:absolute;
        left:12px;
        bottom:10px;
        font-size:11px;
        opacity:.65;
      }
      .ops-kpi-active{
        outline: 1px solid rgba(56,189,248,0.55) !important;
        box-shadow: 0 0 0 2px rgba(56,189,248,0.12) inset;
      }
    `;
    document.head.appendChild(st);
  }

  function getColIndexMap() {
    const ths = $$("#invTable thead th[data-col]");
    const map = {};
    ths.forEach((th, i) => {
      const k = th.getAttribute("data-col");
      if (k) map[k] = i + 1; // +1 because first checkbox column has no data-col
    });
    return map;
  }

  function getRowsSnapshot() {
    const body = $("#invTableBody") || $("#invTable tbody");
    if (!body) return null;

    const col = getColIndexMap();
    const idxArtikel = col.artikelnummer ?? null;
    const idxBestand = col.bestand ?? null;
    const idxMin = col.mindestbestand ?? null;
    if (idxArtikel == null || idxBestand == null || idxMin == null) return null;

    const rows = $$("#invTable tbody tr");
    const snap = {};
    let total = 0, low = 0, zero = 0, ok = 0;

    for (const tr of rows) {
      const tds = tr.querySelectorAll("td");
      if (!tds || tds.length < Math.max(idxArtikel, idxBestand, idxMin)) continue;

      const artikel = (tds[idxArtikel - 1]?.textContent || "").trim();
      const bestand = toNum(tds[idxBestand - 1]?.textContent || "0");
      const min = toNum(tds[idxMin - 1]?.textContent || "0");
      const key = artikel || `row_${total}`;

      snap[key] = bestand;

      total++;
      if (bestand <= 0) { zero++; continue; }
      if (min > 0 && bestand < min) { low++; continue; }
      ok++;
    }

    return { snap, total, low, zero, ok };
  }

  function renderArtikelMini(stats) {
    const host = $("#opsKpiMiniArtikel") || (() => {
      const card = $("#invSummaryCount")?.closest(".kpi-card");
      if (!card) return null;
      const div = document.createElement("div");
      div.className = "ops-kpi-mini";
      div.id = "opsKpiMiniArtikel";
      card.appendChild(div);

      const hint = document.createElement("div");
      hint.className = "ops-kpi-hint";
      hint.id = "opsKpiHintArtikel";
      hint.textContent = "Status‑Mix";
      card.appendChild(hint);
      return div;
    })();
    if (!host) return;

    const total = Math.max(1, stats.total);
    const pZero = stats.zero / total;
    const pLow = stats.low / total;
    const pOk = Math.max(0, 1 - pZero - pLow);

    // stacked bar
    const w = 110, h = 44, pad = 3;
    const bw = w - pad * 2;
    const bh = 12;
    const x0 = pad, y0 = h - pad - bh;

    const seg1 = Math.round(bw * pOk);
    const seg2 = Math.round(bw * pLow);
    const seg3 = bw - seg1 - seg2;

    host.innerHTML = `
      <svg viewBox="0 0 ${w} ${h}" aria-hidden="true">
        <rect x="${x0}" y="${y0}" width="${bw}" height="${bh}" rx="6" ry="6" fill="rgba(255,255,255,0.08)"></rect>
        <rect x="${x0}" y="${y0}" width="${seg1}" height="${bh}" rx="6" ry="6" fill="rgba(34,197,94,0.55)"></rect>
        <rect x="${x0 + seg1}" y="${y0}" width="${seg2}" height="${bh}" fill="rgba(245,158,11,0.55)"></rect>
        <rect x="${x0 + seg1 + seg2}" y="${y0}" width="${seg3}" height="${bh}" rx="6" ry="6" fill="rgba(239,68,68,0.55)"></rect>

        <text x="${x0}" y="${14}" font-size="10" fill="rgba(226,232,240,0.85)">OK ${stats.ok}</text>
        <text x="${x0}" y="${26}" font-size="10" fill="rgba(226,232,240,0.85)">↓Min ${stats.low}</text>
        <text x="${x0}" y="${38}" font-size="10" fill="rgba(226,232,240,0.85)">0 ${stats.zero}</text>
      </svg>
    `;
  }

  function ensureMaterialwertMini() {
    const card = $("#invSummaryValue")?.closest(".kpi-card");
    if (!card) return null;
    let host = $("#opsKpiMiniValue");
    if (!host) {
      host = document.createElement("div");
      host.className = "ops-kpi-mini";
      host.id = "opsKpiMiniValue";
      card.appendChild(host);

      const hint = document.createElement("div");
      hint.className = "ops-kpi-hint";
      hint.id = "opsKpiHintValue";
      hint.textContent = "Bewegung 6M";
      card.appendChild(hint);
    }
    return host;
  }

  function renderMovementMini(metrics) {
    const host = ensureMaterialwertMini();
    if (!host) return;

    const months = Object.keys(metrics.months || {}).sort().slice(-6);
    const data = months.map(k => metrics.months[k] || { in:0, out:0, stock:0 });

    const w = 110, h = 44, pad = 4;
    const mid = 26;
    const maxMove = Math.max(1, ...data.map(d => Math.max(d.in||0, d.out||0)));

    const barW = 12;
    const gap = 6;
    const totalW = data.length * barW + (data.length - 1) * gap;
    let x = (w - totalW) / 2;

    const bars = data.map((d, i) => {
      const inH  = Math.round(14 * (d.in  / maxMove));
      const outH = Math.round(14 * (d.out / maxMove));
      const bx = x + i*(barW+gap);
      return `
        <rect x="${bx}" y="${mid - inH}" width="${barW}" height="${inH}" rx="3" fill="rgba(56,189,248,0.55)"></rect>
        <rect x="${bx}" y="${mid}" width="${barW}" height="${outH}" rx="3" fill="rgba(244,63,94,0.55)"></rect>
      `;
    }).join("");

    host.innerHTML = `
      <svg viewBox="0 0 ${w} ${h}" aria-hidden="true">
        <line x1="6" y1="${mid}" x2="${w-6}" y2="${mid}" stroke="rgba(148,163,184,0.35)" stroke-width="1"></line>
        ${bars}
      </svg>
    `;
  }

  function setStatusFilter(value) {
    const sel = $("#invFilterBestand") || $("#invFilterStatus");
    if (!sel) return;
    sel.value = value || "";
    sel.dispatchEvent(new Event("change", { bubbles: true }));
  }

  function wireKpiClicks() {
    const countCard = $("#invSummaryCount")?.closest(".kpi-card");
    const lowCard   = $("#invSummaryLow")?.closest(".kpi-card");
    const zeroCard  = $("#invSummaryZero")?.closest(".kpi-card");
    const valCard   = $("#invSummaryValue")?.closest(".kpi-card");

    const sel = $("#invFilterBestand") || $("#invFilterStatus");

    const active = () => (sel ? String(sel.value || "") : "");

    function setActiveClass() {
      const v = active();
      [countCard, lowCard, zeroCard].forEach(c => c && c.classList.remove("ops-kpi-active"));
      if (v === "unterMin") lowCard && lowCard.classList.add("ops-kpi-active");
      else if (v === "null" || v === "zero") zeroCard && zeroCard.classList.add("ops-kpi-active");
      else if (!v) countCard && countCard.classList.add("ops-kpi-active");
    }

    if (countCard) {
      countCard.classList.add("ops-kpi-clickable");
      countCard.addEventListener("click", () => setStatusFilter(""));
    }
    if (lowCard) {
      lowCard.classList.add("ops-kpi-clickable");
      lowCard.addEventListener("click", () => setStatusFilter("unterMin"));
    }
    if (zeroCard) {
      zeroCard.classList.add("ops-kpi-clickable");
      zeroCard.addEventListener("click", () => setStatusFilter("null"));
    }
    if (valCard) {
      valCard.classList.add("ops-kpi-clickable");
      // optional: click shows info about chart
      valCard.title = "Demo: Bewegung der letzten 6 Monate (Zugang/Entnahme)";
    }

    if (sel) sel.addEventListener("change", setActiveClass);
    setActiveClass();
  }

  // Detect stock changes (e.g. bookings) and update metrics
  let lastSnap = null;

  function updateMetricsFromSnapshot(stats) {
    const cur = stats.snap;
    const now = new Date();
    const mk = monthKey(now);
    const metrics = loadMetrics();
    if (!metrics.months[mk]) metrics.months[mk] = { in: 0, out: 0, stock: 0 };

    // compute deltas vs last snapshot
    let inSum = 0, outSum = 0;
    if (lastSnap) {
      const keys = new Set([...Object.keys(cur), ...Object.keys(lastSnap)]);
      for (const k of keys) {
        const prev = Number(lastSnap[k] ?? 0);
        const next = Number(cur[k] ?? 0);
        const d = next - prev;
        if (d > 0) inSum += d;
        else if (d < 0) outSum += Math.abs(d);
      }
    }

    const stockSum = Object.values(cur).reduce((a, b) => a + Number(b || 0), 0);
    metrics.months[mk].in = (metrics.months[mk].in || 0) + inSum;
    metrics.months[mk].out = (metrics.months[mk].out || 0) + outSum;
    metrics.months[mk].stock = stockSum;

    // keep only last 12 months
    const keysSorted = Object.keys(metrics.months).sort();
    if (keysSorted.length > 12) {
      keysSorted.slice(0, keysSorted.length - 12).forEach(k => delete metrics.months[k]);
    }

    saveMetrics(metrics);
    renderMovementMini(metrics);

    lastSnap = cur;
  }

  function tick() {
    const stats = getRowsSnapshot();
    if (!stats) return;
    renderArtikelMini(stats);
    updateMetricsFromSnapshot(stats);
  }

  function init() {
    ensureCss();
    wireKpiClicks();
    tick();

    // Observe table changes to update charts when bookings happen
    const tbody = $("#invTable tbody");
    if (tbody && !tbody.__opsObs) {
      tbody.__opsObs = true;
      const mo = new MutationObserver(() => tick());
      mo.observe(tbody, { childList: true, subtree: true, characterData: true });
    }

    // Also refresh occasionally (safe)
    if (!window.__opsDashVizTimer) {
      window.__opsDashVizTimer = setInterval(tick, 5000);
    }
  }

  if (document.readyState === "loading") document.addEventListener("DOMContentLoaded", init);
  else init();
})();
