// ops-demo-dashboard-viz-addon.v2.js
// Demo-only visual enhancements for KPI cards (no core logic changes).
// - "Artikel" card: mini distribution chart by Lager (Zentrallager / Außenlager / Q-Lager / Reparatur ...)
// - KPI cards clickable: toggles stock-status filter (unter Mindest / Bestand=0 / alle)
// - "Materialwert" card: mini movement chart (last 6 months: in/out) based on detected table stock deltas
// - Adds safe scrollbar-stability to the table container (does NOT change data)
// Safe: only reads DOM and triggers existing filter dropdown.
(() => {
  "use strict";
  if (window.__opsDemoDashVizV2Loaded) return;
  window.__opsDemoDashVizV2Loaded = true;

  const LS_KEY = "ops_demo_metrics_v1";

  const $  = (sel, root=document) => root.querySelector(sel);
  const $$ = (sel, root=document) => Array.from(root.querySelectorAll(sel));

  function toNum(v) {
    const s = String(v ?? "").replace(/\./g, "").replace(",", ".").trim();
    const n = Number(s);
    return Number.isFinite(n) ? n : 0;
  }
  function monthKey(d=new Date()){
    const y = d.getFullYear();
    const m = String(d.getMonth()+1).padStart(2,"0");
    return `${y}-${m}`;
  }

  function loadMetrics() {
    try {
      const raw = localStorage.getItem(LS_KEY);
      if (!raw) return { months: {} };
      const obj = JSON.parse(raw);
      if (!obj || typeof obj !== "object") return { months: {} };
      if (!obj.months) obj.months = {};
      return obj;
    } catch(_) { return { months: {} }; }
  }
  function saveMetrics(obj) {
    try { localStorage.setItem(LS_KEY, JSON.stringify(obj)); } catch(_) {}
  }

  function ensureCss() {
    if ($("#opsDemoDashVizCssV2")) return;
    const st = document.createElement("style");
    st.id = "opsDemoDashVizCssV2";
    st.textContent = `
      /* Make space for mini charts so nothing overlaps */
      .kpi-card.kpi-boosted{
        padding-right: 140px !important;
        min-height: 92px;
      }
      .kpi-card{ position: relative; }
      .kpi-card.ops-kpi-clickable{ cursor:pointer; }
      .kpi-card.ops-kpi-clickable:hover{ outline: 1px solid rgba(56,189,248,0.35); }
      .kpi-card.ops-kpi-active{
        outline: 1px solid rgba(56,189,248,0.55) !important;
        box-shadow: 0 0 0 2px rgba(56,189,248,0.12) inset;
      }

      .ops-kpi-mini{
        position:absolute;
        right:12px;
        top:12px;
        width:120px;
        height:56px;
        opacity:.94;
        pointer-events:none;
      }
      .ops-kpi-mini svg{ width:100%; height:100%; display:block; }

      /* subtle label under chart */
      .ops-kpi-mini-label{
        position:absolute;
        right:12px;
        top:70px;
        font-size: 11px;
        opacity:.65;
        pointer-events:none;
      }

      /* keep scrollbars stable (prevents "jumping" and helps visibility) */
      .table-wrap{
        overflow: auto !important;
        scrollbar-gutter: stable both-edges;
      }
    `;
    document.head.appendChild(st);
  }

  // Map <th data-col="..."> to column index in tbody
  function getColIndexMap() {
    const ths = $$("#invTable thead th");
    const map = {};
    ths.forEach((th, i) => {
      const k = th.getAttribute("data-col");
      if (k) map[k] = i + 1; // 1-based
    });
    return map;
  }

  function getRowsSnapshot() {
    const body = $("#invTableBody") || $("#invTable tbody");
    if (!body) return null;

    const col = getColIndexMap();
    const idxArtikel = col.artikelnummer ?? null;
    const idxBestand = col.bestand ?? null;
    const idxMin     = col.mindestbestand ?? null;
    const idxLager    = col.lager ?? null;

    if (idxArtikel == null || idxBestand == null || idxMin == null) return null;

    const rows = $$("#invTable tbody tr");
    const snap = {};
    let total = 0, low = 0, zero = 0, ok = 0;

    const lagerCounts = {}; // string -> count
    for (const tr of rows) {
      const tds = tr.querySelectorAll("td");
      const need = Math.max(idxArtikel, idxBestand, idxMin, idxLager || 0);
      if (!tds || tds.length < need) continue;

      const artikel = (tds[idxArtikel - 1]?.textContent || "").trim();
      const bestand = toNum(tds[idxBestand - 1]?.textContent || "0");
      const min = toNum(tds[idxMin - 1]?.textContent || "0");
      const key = artikel || `row_${total}`;

      snap[key] = bestand;

      total++;
      if (bestand <= 0) { zero++; }
      else if (min > 0 && bestand < min) { low++; }
      else ok++;

      if (idxLager != null) {
        const raw = (tds[idxLager - 1]?.textContent || "").trim();
        const l = raw || "—";
        lagerCounts[l] = (lagerCounts[l] || 0) + 1;
      }
    }

    return { snap, total, low, zero, ok, lagerCounts };
  }

  function ensureMiniHost(card, id, labelText) {
    if (!card) return null;
    card.classList.add("kpi-boosted");
    let host = document.getElementById(id);
    if (!host) {
      host = document.createElement("div");
      host.className = "ops-kpi-mini";
      host.id = id;
      card.appendChild(host);

      const lab = document.createElement("div");
      lab.className = "ops-kpi-mini-label";
      lab.id = id + "_label";
      lab.textContent = labelText || "";
      card.appendChild(lab);
    } else {
      const lab = document.getElementById(id + "_label");
      if (lab && labelText != null) lab.textContent = labelText;
    }
    return host;
  }

  function renderArtikelMiniByLager(stats) {
    const card = $("#invSummaryCount")?.closest(".kpi-card");
    const host = ensureMiniHost(card, "opsKpiMiniArtikel", "Lager‑Mix");
    if (!host) return;

    const entries = Object.entries(stats.lagerCounts || {}).sort((a,b)=>b[1]-a[1]);
    const top = entries.slice(0, 4);
    const rest = entries.slice(4).reduce((a,[,c])=>a+c,0);

    const total = Math.max(1, stats.total);
    const items = top.map(([k,c]) => [k, c, c/total]);
    if (rest > 0) items.push(["Rest", rest, rest/total]);

    // Build stacked bar with 5 segments max
    const w=120, h=56, pad=4;
    const bw = w - pad*2;
    const bh = 12;
    const x0=pad, y0=h-pad-bh;

    // palette (kept subtle, theme-like)
    const fills = [
      "rgba(56,189,248,0.55)",
      "rgba(168,85,247,0.50)",
      "rgba(34,197,94,0.50)",
      "rgba(245,158,11,0.50)",
      "rgba(148,163,184,0.35)"
    ];

    let x = x0;
    const segs = items.map(([name,count,p],i)=>{
      const ww = (i === items.length-1) ? (x0+bw - x) : Math.max(3, Math.round(bw*p));
      const fill = fills[i % fills.length];
      const r = (i===0 || i===items.length-1) ? 6 : 0;
      const rect = `<rect x="${x}" y="${y0}" width="${ww}" height="${bh}" rx="${r}" ry="${r}" fill="${fill}"></rect>`;
      x += ww;
      return rect;
    }).join("");

    // show up to 3 labels
    const labelLines = items.slice(0,3).map(([name,count],i)=>{
      const yy = 14 + i*13;
      const nm = String(name).length > 14 ? String(name).slice(0,13)+"…" : name;
      return `<text x="${x0}" y="${yy}" font-size="10" fill="rgba(226,232,240,0.88)">${nm} ${count}</text>`;
    }).join("");

    host.innerHTML = `
      <svg viewBox="0 0 ${w} ${h}" aria-hidden="true">
        ${labelLines}
        <rect x="${x0}" y="${y0}" width="${bw}" height="${bh}" rx="6" ry="6" fill="rgba(255,255,255,0.07)"></rect>
        ${segs}
      </svg>
    `;
  }

  function ensureMaterialwertMini() {
    const card = $("#invSummaryValue")?.closest(".kpi-card");
    const host = ensureMiniHost(card, "opsKpiMiniValue", "Bewegung 6M");
    return host;
  }

  function renderMovementMini(metrics) {
    const host = ensureMaterialwertMini();
    if (!host) return;

    const months = Object.keys(metrics.months || {}).sort().slice(-6);
    const data = months.map(k => metrics.months[k] || { in:0, out:0, stock:0 });

    const w = 120, h = 56;
    const mid = 32;
    const maxMove = Math.max(1, ...data.map(d => Math.max(d.in||0, d.out||0)));

    const barW = 12;
    const gap = 6;
    const totalW = data.length * barW + (data.length - 1) * gap;
    const startX = (w - totalW) / 2;

    const bars = data.map((d, i) => {
      const inH  = Math.round(16 * (d.in  / maxMove));
      const outH = Math.round(16 * (d.out / maxMove));
      const x = startX + i*(barW+gap);
      return `
        <rect x="${x}" y="${mid - inH}" width="${barW}" height="${inH}" rx="3" fill="rgba(56,189,248,0.55)"></rect>
        <rect x="${x}" y="${mid}" width="${barW}" height="${outH}" rx="3" fill="rgba(244,63,94,0.52)"></rect>
      `;
    }).join("");

    host.innerHTML = `
      <svg viewBox="0 0 ${w} ${h}" aria-hidden="true">
        <line x1="6" y1="${mid}" x2="${w-6}" y2="${mid}" stroke="rgba(148,163,184,0.35)" stroke-width="1"></line>
        ${bars}
      </svg>
    `;
  }

  function setStatusFilter(value) {
    const sel = $("#invFilterBestand") || $("#invFilterStatus");
    if (!sel) return;
    sel.value = value || "";
    sel.dispatchEvent(new Event("change", { bubbles: true }));
  }

  function wireKpiClicks() {
    const countCard = $("#invSummaryCount")?.closest(".kpi-card");
    const lowCard   = $("#invSummaryLow")?.closest(".kpi-card");
    const zeroCard  = $("#invSummaryZero")?.closest(".kpi-card");

    const sel = $("#invFilterBestand") || $("#invFilterStatus");
    const active = () => (sel ? String(sel.value || "") : "");

    function refreshActive() {
      const v = active();
      [countCard, lowCard, zeroCard].forEach(c => c && c.classList.remove("ops-kpi-active"));
      if (v === "unterMin") lowCard && lowCard.classList.add("ops-kpi-active");
      else if (v === "null" || v === "zero") zeroCard && zeroCard.classList.add("ops-kpi-active");
      else if (!v) countCard && countCard.classList.add("ops-kpi-active");
    }

    if (countCard) {
      countCard.classList.add("ops-kpi-clickable");
      countCard.addEventListener("click", () => setStatusFilter(""));
    }
    if (lowCard) {
      lowCard.classList.add("ops-kpi-clickable");
      lowCard.addEventListener("click", () => setStatusFilter("unterMin"));
    }
    if (zeroCard) {
      zeroCard.classList.add("ops-kpi-clickable");
      zeroCard.addEventListener("click", () => setStatusFilter("null"));
    }

    if (sel) sel.addEventListener("change", refreshActive);
    refreshActive();
  }

  // Detect stock changes (bookings) and update metrics
  let lastSnap = null;

  function updateMetricsFromSnapshot(stats) {
    const cur = stats.snap;
    const mk = monthKey(new Date());
    const metrics = loadMetrics();
    if (!metrics.months[mk]) metrics.months[mk] = { in: 0, out: 0, stock: 0 };

    let inSum = 0, outSum = 0;
    if (lastSnap) {
      const keys = new Set([...Object.keys(cur), ...Object.keys(lastSnap)]);
      for (const k of keys) {
        const prev = Number(lastSnap[k] ?? 0);
        const next = Number(cur[k] ?? 0);
        const d = next - prev;
        if (d > 0) inSum += d;
        else if (d < 0) outSum += Math.abs(d);
      }
    }

    const stockSum = Object.values(cur).reduce((a,b)=>a+Number(b||0),0);
    metrics.months[mk].in = (metrics.months[mk].in || 0) + inSum;
    metrics.months[mk].out = (metrics.months[mk].out || 0) + outSum;
    metrics.months[mk].stock = stockSum;

    const keysSorted = Object.keys(metrics.months).sort();
    if (keysSorted.length > 12) {
      keysSorted.slice(0, keysSorted.length - 12).forEach(k => delete metrics.months[k]);
    }

    saveMetrics(metrics);
    renderMovementMini(metrics);
    lastSnap = cur;
  }

  function tick() {
    const stats = getRowsSnapshot();
    if (!stats) return;
    renderArtikelMiniByLager(stats);
    updateMetricsFromSnapshot(stats);
  }

  function init() {
    ensureCss();
    wireKpiClicks();
    tick();

    const tbody = $("#invTable tbody");
    if (tbody && !tbody.__opsObsViz) {
      tbody.__opsObsViz = true;
      const mo = new MutationObserver(() => tick());
      mo.observe(tbody, { childList: true, subtree: true, characterData: true });
    }

    if (!window.__opsDashVizTimerV2) {
      window.__opsDashVizTimerV2 = setInterval(tick, 6000);
    }
  }

  if (document.readyState === "loading") document.addEventListener("DOMContentLoaded", init);
  else init();
})();
