// ops-demo-dashboard-viz-addon.v6.js
// Demo-only visual enhancements for KPI cards (no core logic changes).
// Fixes:
// - Forces internal table scrollbars even in narrow window mode (overrides a max-width media rule)
// - KPI mini charts: proper padding so texts stay aligned
// - Restores small captions (Status-Mix / Bewegung 6M)

(() => {
  "use strict";
  if (window.__opsDemoDashVizV5Loaded) return;
  window.__opsDemoDashVizV5Loaded = true;

  const LS_KEY = "ops_demo_metrics_v1";

  const $  = (sel, root=document) => root.querySelector(sel);
  const $$ = (sel, root=document) => Array.from(root.querySelectorAll(sel))
  // Language helper (DE/EN)
  function opsLang() {
    try { if (typeof window.getOpsLang === "function") return String(window.getOpsLang()).toLowerCase().startsWith("en") ? "en" : "de"; } catch(_) {}
    const l = document.documentElement.getAttribute("lang") || "de";
    return String(l).toLowerCase().startsWith("en") ? "en" : "de";
  }
  function t(deText, enText){ return opsLang() === "en" ? enText : deText; }

;

  function toNum(v) {
    const s = String(v ?? "").replace(/\./g, "").replace(",", ".").trim();
    const n = Number(s);
    return Number.isFinite(n) ? n : 0;
  }
  function monthKey(d=new Date()){
    const y = d.getFullYear();
    const m = String(d.getMonth()+1).padStart(2,"0");
    return `${y}-${m}`;
  }

  function loadMetrics() {
    try {
      const raw = localStorage.getItem(LS_KEY);
      if (!raw) return { months: {} };
      const obj = JSON.parse(raw);
      if (!obj || typeof obj !== "object") return { months: {} };
      if (!obj.months) obj.months = {};
      return obj;
    } catch(_) { return { months: {} }; }
  }
  function saveMetrics(obj) {
    try { localStorage.setItem(LS_KEY, JSON.stringify(obj)); } catch(_) {}
  }

  function cleanupOldNodes() {
    // Remove leftover hints/labels from previous add-on versions
    [
      "#opsKpiHintArtikel", "#opsKpiHintValue",
      "#opsKpiMiniArtikel_label", "#opsKpiMiniValue_label",
      ".ops-kpi-mini-label", ".ops-kpi-hint"
    ].forEach(sel => $$(sel).forEach(n => { try { n.remove(); } catch(_) { n.style.display="none"; } }));

    // If an old mini host exists with old content, we keep the container but will overwrite its innerHTML anyway.
  }

  function ensureCss() {
    if ($("#opsDemoDashVizCssV6")) return;
    const st = document.createElement("style");
    st.id = "opsDemoDashVizCssV6";
    st.textContent = `
      /* Give KPI cards space for the mini chart on the right */
      .kpi-card.kpi-boosted{
        padding-right: 160px !important;
        min-height: 96px;
      }
      .kpi-card{ position: relative; }

      /* Make subtitles align nicely */
      .kpi-card .kpi-sub{
        margin-top: 6px;
        line-height: 1.15;
      }

      /* Mini charts: right side, vertically centered */
      /* Cards that get a mini chart reserve space on the right so texts never overlap */
      .kpi-card.ops-kpi-has-mini{
        padding-right: 172px;
      }

      .ops-kpi-mini{
        position:absolute;
        right:12px;
        top:50%;
        transform: translateY(-50%);
        width:146px;
        height:64px;
        opacity:.95;
        pointer-events:none;
      }

      .ops-kpi-mini-caption{
        position:absolute;
        right:14px;
        bottom:10px;
        font-size: 11px;
        line-height: 1.1;
        color: rgba(148,163,184,0.90);
        letter-spacing: .02em;
        pointer-events:none;
        user-select:none;
        text-align:right;
      }
      .ops-kpi-mini svg{ width:100%; height:100%; display:block; }

      /* Click affordance */
      .kpi-card.ops-kpi-clickable{ cursor:pointer; }
      .kpi-card.ops-kpi-clickable:hover{ outline: 1px solid rgba(56,189,248,0.35); }
      .kpi-card.ops-kpi-active{
        outline: 1px solid rgba(56,189,248,0.55) !important;
        box-shadow: 0 0 0 2px rgba(56,189,248,0.12) inset;
      }

      /* Table viewport: inline style was height:250px -> override for demo comfort.
         Do not touch fullscreen/large modes if your app adds these classes. */
      #lagerverwaltung .table-wrap:not(.inv-table-large):not(.inv-table-in-fs){
        height: 650px !important;
        max-height: 60vh !important;
        overflow-x: auto !important;
        overflow-y: scroll !important; /* always show vertical bar */
        scrollbar-gutter: stable both-edges;
      }

      /* Force the table to keep its natural width -> horizontal scrollbar shows all columns */
      #lagerverwaltung .table-wrap table.data-table{
        width: max-content !important;
        min-width: 1600px !important;
      }
    `;
    document.head.appendChild(st);
  }

  // Map <th data-col="..."> to 1-based td index
  function getColIndexMap() {
    const ths = $$("#invTable thead th");
    const map = {};
    ths.forEach((th, i) => {
      const k = th.getAttribute("data-col");
      if (k) map[k] = i + 1;
    });
    return map;
  }

  function getRowsSnapshot() {
    const body = $("#invTableBody") || $("#invTable tbody");
    if (!body) return null;

    const col = getColIndexMap();
    const idxArtikel = col.artikelnummer ?? null;
    const idxBestand = col.bestand ?? null;
    const idxMin     = col.mindestbestand ?? null;
    const idxLager    = col.lager ?? null;

    if (idxArtikel == null || idxBestand == null || idxMin == null) return null;

    const rows = $$("#invTable tbody tr");
    const snap = {};
    let total = 0;

    const lagerCounts = {};
    for (const tr of rows) {
      const tds = tr.querySelectorAll("td");
      const need = Math.max(idxArtikel, idxBestand, idxMin, idxLager || 0);
      if (!tds || tds.length < need) continue;

      const artikel = (tds[idxArtikel - 1]?.textContent || "").trim();
      const bestand = toNum(tds[idxBestand - 1]?.textContent || "0");
      const key = artikel || `row_${total}`;
      snap[key] = bestand;
      total++;

      if (idxLager != null) {
        const raw = (tds[idxLager - 1]?.textContent || "").trim();
        const l = raw || "—";
        lagerCounts[l] = (lagerCounts[l] || 0) + 1;
      }
    }

    return { snap, total, lagerCounts };
  }

  // Hard-fix for scrollbars in normal (non-fullscreen) mode.
  // Some responsive CSS sets the table-wrap height to "auto" on small screens;
  // then the "scrollbar in table" disappears. We enforce a viewport height and
  // keep horizontal scrolling enabled so all columns are reachable.
  function forceTableViewport() {
    try {
      const wrap = document.querySelector("#lagerverwaltung .table-wrap");
      if (!wrap) return;
      // Don't touch the fullscreen overlay mode (handled by its own CSS)
      if (wrap.classList.contains("inv-table-in-fs")) return;

      // Enforce scroll viewport (inline wins over everything)
      wrap.style.height = "650px";
      wrap.style.maxHeight = "60vh";
      wrap.style.overflow = "auto";
      wrap.style.overflowY = "scroll"; // show vertical bar
      wrap.style.overflowX = "auto";
      try { wrap.style.scrollbarGutter = "stable both-edges"; } catch(_) {}

      // Ensure the table doesn't shrink to 100% -> enable horizontal scroll
      const tbl = document.getElementById("invTable");
      if (tbl) {
        tbl.style.width = "max-content";
        tbl.style.minWidth = "1600px";
      }
    } catch (e) {}
  }

  function ensureViewportHooks() {
    if (window.__opsDemoDashVizViewportHookV6) return;
    window.__opsDemoDashVizViewportHookV6 = true;
    window.addEventListener("resize", () => {
      // small debounce
      clearTimeout(window.__opsDemoDashVizViewportTmoV6);
      window.__opsDemoDashVizViewportTmoV6 = setTimeout(forceTableViewport, 120);
    });
    // Also re-apply shortly after load (fonts/layout may settle late)
    setTimeout(forceTableViewport, 250);
    setTimeout(forceTableViewport, 900);
  }


  function ensureMiniHost(card, id) {
    if (!card) return null;
    card.classList.add("kpi-boosted");
    let host = document.getElementById(id);
    if (!host) {
      host = document.createElement("div");
      host.className = "ops-kpi-mini";
      host.id = id;
      card.appendChild(host);
    }
    return host;
  }

  function ensureMiniCaption(card, id, text) {
    if (!card) return null;
    let el = document.getElementById(id);
    if (!el) {
      el = document.createElement("div");
      el.id = id;
      el.className = "ops-kpi-mini-caption";
      card.appendChild(el);
    }
    el.textContent = text || "";
    return el;
  }


  function renderArtikelMiniByLager(stats) {
    const card = $("#invSummaryCount")?.closest(".kpi-card");
    const host = ensureMiniHost(card, "opsKpiMiniArtikel");
    if (!host) return;

    const entries = Object.entries(stats.lagerCounts || {}).sort((a,b)=>b[1]-a[1]);
    const top = entries.slice(0, 4);
    const rest = entries.slice(4).reduce((a,[,c])=>a+c,0);

    const total = Math.max(1, stats.total);
    const items = top.map(([k,c]) => ({ name:k, count:c, p:c/total }));
    if (rest > 0) items.push({ name:"Rest", count:rest, p:rest/total });

    const w=146, h=64, pad=6;
    const bw = w - pad*2;
    const bh = 14;
    const x0=pad, y0=h-pad-bh;

    const fills = [
      "rgba(56,189,248,0.55)",
      "rgba(168,85,247,0.50)",
      "rgba(34,197,94,0.50)",
      "rgba(245,158,11,0.50)",
      "rgba(148,163,184,0.35)"
    ];

    let x = x0;
    const segs = items.map((it,i)=>{
      const ww = (i === items.length-1) ? (x0+bw - x) : Math.max(3, Math.round(bw*it.p));
      const fill = fills[i % fills.length];
      const r = (i===0 || i===items.length-1) ? 7 : 0;
      const rect = `<rect x="${x}" y="${y0}" width="${ww}" height="${bh}" rx="${r}" ry="${r}" fill="${fill}"></rect>`;
      x += ww;
      return rect;
    }).join("");

    // small legend lines (2 lines max) above the bar
    const legend = items.slice(0,2).map((it,i)=>{
      const yy = 16 + i*14;
      const nm = String(it.name).length > 12 ? String(it.name).slice(0,11)+"…" : it.name;
      return `<text x="${x0}" y="${yy}" font-size="10" fill="rgba(226,232,240,0.88)">${nm} ${it.count}</text>`;
    }).join("");

    host.innerHTML = `
      <svg viewBox="0 0 ${w} ${h}" aria-hidden="true">
        ${legend}
        <rect x="${x0}" y="${y0}" width="${bw}" height="${bh}" rx="7" ry="7" fill="rgba(255,255,255,0.07)"></rect>
        ${segs}
      </svg>
    `;
    ensureMiniCaption(card, "opsKpiMiniArtikelCaption", t("Status‑Mix", "Stock mix"));
    host.title = items.map(it => `${it.name}: ${it.count}`).join(" • ");
  }

  function renderMovementMini(metrics) {
    const card = $("#invSummaryValue")?.closest(".kpi-card");
    const host = ensureMiniHost(card, "opsKpiMiniValue");
    if (!host) return;

    const months = Object.keys(metrics.months || {}).sort().slice(-6);
    const data = months.map(k => metrics.months[k] || { in:0, out:0, stock:0 });

    const w=146, h=64, pad=6;
    const mid=34;
    const maxMove = Math.max(1, ...data.map(d => Math.max(d.in||0, d.out||0)));

    const barW=12, gap=7;
    const totalW = data.length*barW + (data.length-1)*gap;
    const startX = (w-totalW)/2;

    const bars = data.map((d,i)=>{
      const inH = Math.round(18*(d.in/maxMove));
      const outH = Math.round(18*(d.out/maxMove));
      const x = startX + i*(barW+gap);
      return `
        <rect x="${x}" y="${mid-inH}" width="${barW}" height="${inH}" rx="3" fill="rgba(56,189,248,0.55)"></rect>
        <rect x="${x}" y="${mid}" width="${barW}" height="${outH}" rx="3" fill="rgba(244,63,94,0.52)"></rect>
      `;
    }).join("");

    host.innerHTML = `
      <svg viewBox="0 0 ${w} ${h}" aria-hidden="true">
        <line x1="${pad}" y1="${mid}" x2="${w-pad}" y2="${mid}" stroke="rgba(148,163,184,0.35)" stroke-width="1"></line>
        ${bars}
      </svg>
    `;
    ensureMiniCaption(card, "opsKpiMiniValueCaption", t("Bewegung 6M", "Movement 6M"));
    host.title = t("Bewegung (Zugang/Entnahme) – Demo", "Movement (in/out) – demo");
  }

  function setStatusFilter(value) {
    const sel = $("#invFilterBestand") || $("#invFilterStatus");
    if (!sel) return;
    sel.value = value || "";
    sel.dispatchEvent(new Event("change", { bubbles: true }));
  }

  function wireKpiClicks() {
    const countCard = $("#invSummaryCount")?.closest(".kpi-card");
    const lowCard   = $("#invSummaryLow")?.closest(".kpi-card");
    const zeroCard  = $("#invSummaryZero")?.closest(".kpi-card");

    const sel = $("#invFilterBestand") || $("#invFilterStatus");
    const active = () => (sel ? String(sel.value || "") : "");

    function refreshActive() {
      const v = active();
      [countCard, lowCard, zeroCard].forEach(c => c && c.classList.remove("ops-kpi-active"));
      if (v === "unterMin") lowCard && lowCard.classList.add("ops-kpi-active");
      else if (v === "null" || v === "zero") zeroCard && zeroCard.classList.add("ops-kpi-active");
      else if (!v) countCard && countCard.classList.add("ops-kpi-active");
    }

    if (countCard) {
      countCard.classList.add("ops-kpi-clickable");
      countCard.addEventListener("click", () => setStatusFilter(""));
    }
    if (lowCard) {
      lowCard.classList.add("ops-kpi-clickable");
      lowCard.addEventListener("click", () => setStatusFilter("unterMin"));
    }
    if (zeroCard) {
      zeroCard.classList.add("ops-kpi-clickable");
      zeroCard.addEventListener("click", () => setStatusFilter("null"));
    }
    if (sel) sel.addEventListener("change", refreshActive);
    refreshActive();
  }

  // Detect stock changes and accumulate monthly in/out
  let lastSnap = null;

  function updateMetricsFromSnapshot(stats) {
    const cur = stats.snap;
    const mk = monthKey(new Date());
    const metrics = loadMetrics();
    if (!metrics.months[mk]) metrics.months[mk] = { in: 0, out: 0, stock: 0 };

    let inSum=0, outSum=0;
    if (lastSnap) {
      const keys = new Set([...Object.keys(cur), ...Object.keys(lastSnap)]);
      for (const k of keys) {
        const prev = Number(lastSnap[k] ?? 0);
        const next = Number(cur[k] ?? 0);
        const d = next - prev;
        if (d > 0) inSum += d;
        else if (d < 0) outSum += Math.abs(d);
      }
    }

    const stockSum = Object.values(cur).reduce((a,b)=>a+Number(b||0),0);
    metrics.months[mk].in = (metrics.months[mk].in || 0) + inSum;
    metrics.months[mk].out = (metrics.months[mk].out || 0) + outSum;
    metrics.months[mk].stock = stockSum;

    const keysSorted = Object.keys(metrics.months).sort();
    if (keysSorted.length > 12) keysSorted.slice(0, keysSorted.length-12).forEach(k => delete metrics.months[k]);

    saveMetrics(metrics);
    renderMovementMini(metrics);
    lastSnap = cur;
  }

  function tick() {
    const stats = getRowsSnapshot();
    if (!stats) return;
    renderArtikelMiniByLager(stats);
    updateMetricsFromSnapshot(stats);
  }

  function init() {
    cleanupOldNodes();
    ensureCss();
    ensureViewportHooks();
    forceTableViewport();
    wireKpiClicks();
    tick();

    const tbody = $("#invTable tbody");
    if (tbody && !tbody.__opsObsVizV6) {
      tbody.__opsObsVizV6 = true;
      const mo = new MutationObserver(() => tick());
      mo.observe(tbody, { childList: true, subtree: true, characterData: true });
    }

    if (!window.__opsDashVizTimerV6) {
      window.__opsDashVizTimerV6 = setInterval(tick, 7000);
    }
  }

  if (document.readyState === "loading") document.addEventListener("DOMContentLoaded", init);
  else init();
})();
