// ops-demo-table-scroll-fix.js
// Fixes for DEMO layout:
// 1) Ensure the inventory table has vertical + horizontal scrollbars in normal view (not only fullscreen)
// 2) Prevent the table from being "squeezed" so the last column (Lagerort) is fully reachable
//
// Safe: UI-only (styles), no business logic touched.
(() => {
  "use strict";
  if (window.__opsDemoTableScrollFixLoaded) return;
  window.__opsDemoTableScrollFixLoaded = true;

  function ensureStyle() {
    if (document.getElementById("opsDemoTableScrollFixCss")) return;
    const st = document.createElement("style");
    st.id = "opsDemoTableScrollFixCss";
    st.textContent = `
      /* Make sure the table area scrolls inside the card */
      #lagerverwaltung .table-wrap{
        height: 650px !important;
        max-height: 60vh !important;
        overflow-y: scroll !important;
        overflow-x: auto !important;
        scrollbar-gutter: stable both-edges;
      }

      /* Critical: allow horizontal scrolling by letting the table be as wide as its content */
      #lagerverwaltung table#invTable.data-table{
        width: max-content !important;
        min-width: 100% !important;
        table-layout: auto !important;
      }

      /* Give the last column a little extra breathing room so it never looks cut off */
      #lagerverwaltung table#invTable.data-table th:last-child,
      #lagerverwaltung table#invTable.data-table td:last-child{
        padding-right: 22px !important;
      }
    `;
    document.head.appendChild(st);
  }

  function applyInlineSafety() {
    const wrap = document.querySelector("#lagerverwaltung .table-wrap");
    const tbl  = document.getElementById("invTable");
    if (!wrap || !tbl) return;

    // Remove hardcoded inline height from HTML ("height:250px") if present
    try {
      // This only clears the height/maxHeight/overflow parts, keeps other inline styling intact.
      wrap.style.height = "650px";
      wrap.style.maxHeight = "60vh";
      wrap.style.overflowY = "scroll";
      wrap.style.overflowX = "auto";
    } catch (_) {}

    try {
      tbl.style.width = "max-content";
      tbl.style.minWidth = "100%";
      tbl.style.tableLayout = "auto";
    } catch (_) {}
  }

  function init() {
    ensureStyle();
    applyInlineSafety();

    // In case other scripts re-render/replace the table area:
    const root = document.getElementById("lagerverwaltung");
    if (!root || root.__opsDemoTableScrollFixObs) return;
    root.__opsDemoTableScrollFixObs = true;

    const mo = new MutationObserver(() => {
      // cheap re-apply
      applyInlineSafety();
    });
    mo.observe(root, { childList: true, subtree: true });
  }

  if (document.readyState === "loading") {
    document.addEventListener("DOMContentLoaded", init);
  } else {
    init();
  }
  document.addEventListener("DOMContentLoaded", function () {
  const wrapper = document.getElementById("scrollFixWrapper");
  if (!wrapper) return;

  // Nur wenn genug Zeilen vorhanden sind
  const table = document.getElementById("invTable");
  if (!table) return;

  const rows = table.querySelectorAll("tbody tr");
  if (rows.length > 10) {
    wrapper.style.overflowY = "auto";
    wrapper.style.overflowX = "auto";
    wrapper.style.maxHeight = "60vh";
    wrapper.style.height = "650px";
    wrapper.style.border = "1px solid rgba(148, 163, 184, 0.4)";
    wrapper.style.borderRadius = "6px";
    wrapper.style.padding = "2px";
    wrapper.style.background = "rgba(15, 23, 42, 0.85)";
    wrapper.style.scrollbarGutter = "stable both-edges";
    wrapper.style.pointerEvents = "auto";
    wrapper.style.boxSizing = "border-box";
  }
});

})();
