/* ops-lagerverwaltung-rbac.js (Offline Server RBAC)

   Real RBAC (no prototype): permissions/roles/users are stored in SQLite on the local Python server.
   - Reads effective permissions from /api/me
   - Admin dialogs (same style) for:
     - User ↔ roles assignment
     - Role ↔ permission matrix
     - ZIP backup download (DB snapshot + optional files)

   Requirements:
   - ops-offline-client.js loaded BEFORE this file (provides OpsOffline + login + sync)
*/

(() => {
  'use strict';

  if (window.__opsLvRbacServerLoaded) return;
  window.__opsLvRbacServerLoaded = true;

  // --- i18n -----------------------------------------------------------
  const getLang = () => {
    try {
      if (typeof window.getOpsLang === 'function') return window.getOpsLang();
    } catch (_) {}
    const l = String(document.documentElement.lang || 'de').toLowerCase();
    return l.startsWith('en') ? 'en' : 'de';
  };

  const t = (de, en) => {
    try {
      if (typeof window.tLang === 'function') return window.tLang(de, en);
    } catch (_) {}
    return getLang() === 'en' ? (en ?? de) : (de ?? en);
  };

  // --- style (match your dialogs) ------------------------------------
  function ensureDialogTheme() {
    if (document.getElementById('opsLvServerRbacTheme')) return;
    const style = document.createElement('style');
    style.id = 'opsLvServerRbacTheme';
    style.textContent = `
      dialog.dialog{ border:none; padding:0; background:transparent; }
      dialog.dialog::backdrop{ background: rgba(2,6,23,0.60); backdrop-filter: blur(3px); }
      dialog.dialog .card{
        background: linear-gradient(180deg, rgba(30,58,138,.96), rgba(15,23,42,.98));
        color: rgba(226,232,240,.98);
        border: 1px solid rgba(59,130,246,.35);
        border-radius: 16px;
        box-shadow: 0 18px 44px rgba(0,0,0,.45);
      }
      dialog.dialog .card .row{ display:flex; gap:10px; flex-wrap:wrap; align-items:center; }
      dialog.dialog .card input[type="password"],
      dialog.dialog .card input[type="text"],
      dialog.dialog .card input[type="search"]{
        background: rgba(2,6,23,.35);
        color: rgba(226,232,240,.98);
        border: 1px solid rgba(148,163,184,.35);
        border-radius: 12px;
        padding: 8px 12px;
        outline: none;
      }
      dialog.dialog .card input:focus{
        border-color: rgba(56,189,248,.55);
        box-shadow: 0 0 0 2px rgba(56,189,248,.18);
      }
      dialog.dialog .card .muted{ opacity:.9; }
      .ops-admin-table{ width:100%; border-collapse:collapse; font-size:13px; }
      .ops-admin-table th,.ops-admin-table td{ padding:8px 10px; border-bottom:1px solid rgba(148,163,184,.22); vertical-align:top; }
      .ops-admin-table th{ position:sticky; top:0; background: rgba(15,23,42,.92); z-index:1; text-align:left; }
      .ops-admin-table tbody tr:hover td{ background: rgba(2,6,23,.18); }
      .ops-chip{ display:inline-flex; align-items:center; gap:6px; padding:4px 10px; border-radius:999px; border:1px solid rgba(148,163,184,.28); background: rgba(2,6,23,.22); }
      .ops-chip small{ opacity:.85; }
      .ops-roles-wrap{ display:flex; flex-wrap:wrap; gap:8px; }
      .ops-role-tag{ padding:3px 10px; border-radius:999px; border:1px solid rgba(148,163,184,.25); background: rgba(2,6,23,.18); font-size:12px; }
      .ops-mini{ font-size:12px; opacity:.88; }
      .ops-danger{ color:#ffb4b4; }
    `;
    document.head.appendChild(style);
  }

  function openDialog(dlg) {
    try { dlg.showModal(); return; } catch (_) {}
    try { dlg.setAttribute('open', ''); } catch (_) {}
  }
  function closeDialog(dlg) {
    try { dlg.close(); } catch (_) {}
    try { dlg.removeAttribute('open'); } catch (_) {}
  }

  async function uiAlert(de, en, titleDe, titleEn) {
    try {
      if (window.OPS_UI && typeof window.OPS_UI.alert === 'function') {
        return await window.OPS_UI.alert(de, en, titleDe || 'ℹ️', titleEn || 'ℹ️');
      }
    } catch (_) {}
    alert((getLang()==='en' ? (titleEn||'Info') : (titleDe||'Info')) + '\n\n' + t(de,en));
  }

  // --- API ------------------------------------------------------------
  function api() {
    if (!window.OpsOffline || typeof window.OpsOffline.apiFetch !== 'function') {
      throw new Error('OpsOffline.apiFetch missing (did you load ops-offline-client.js?)');
    }
    return window.OpsOffline;
  }

  const STATE = {
    me: null,
    lastMeFetch: 0,
    busy: false
  };

  async function refreshMe(force=false) {
    const now = Date.now();
    if (!force && STATE.me && (now - STATE.lastMeFetch) < 1000) return STATE.me;
    const u = await api().me();
    STATE.me = u;
    STATE.lastMeFetch = now;
    return u;
  }

  function permsSet() {
    const p = (STATE.me && STATE.me.permissions) ? STATE.me.permissions : [];
    return new Set(Array.isArray(p) ? p : []);
  }

  function can(permKey) {
    if (!permKey) return true;
    const s = permsSet();
    return s.has(String(permKey));
  }

  function applyUi() {
    const nodes = document.querySelectorAll('[data-perm]');
    nodes.forEach((el) => {
      if (!el || el.nodeType !== 1) return;
      const perm = el.getAttribute('data-perm');
      const ok = can(perm);
      if (!ok) {
        try { el.disabled = true; } catch (_) {}
        try { el.setAttribute('aria-disabled','true'); } catch (_) {}
        try { el.style.pointerEvents = 'none'; } catch (_) {}
        try { el.style.opacity = '0.55'; } catch (_) {}
      } else {
        try { el.disabled = false; } catch (_) {}
        try { el.removeAttribute('aria-disabled'); } catch (_) {}
        try { el.style.pointerEvents = 'auto'; } catch (_) {}
        try { el.style.opacity = ''; } catch (_) {}
      }
    });
  }

  // Expose for other modules
  window.OpsRBAC = window.OpsRBAC || {};
  window.OpsRBAC.can = can;
  window.OpsRBAC.applyUi = applyUi;
  window.OpsRBAC.refresh = () => refreshMe(true).then(() => applyUi());

  // --- Header role pill ------------------------------------------------
  function findHeaderRight() {
    return (
      document.querySelector('#header-inner .header-right') ||
      document.querySelector('.app-header .header-right') ||
      null
    );
  }

  function roleLabelForKey(key) {
    const k = String(key||'');
    const map = {
      mitarbeiter: [t('Mitarbeiter','Employee')],
      wareneingang: [t('Wareneingang','Goods receipt')],
      einkauf: [t('Einkauf','Purchasing')],
      beschaffung: [t('Beschaffung','Procurement')],
      inventur: [t('Inventurbeauftragter','Stocktake officer')],
      admin: [t('Admin','Admin')],
      systemadmin: [t('Systemadmin','System admin')]
    };
    return (map[k] && map[k][0]) ? map[k][0] : k;
  }

  function ensureRolePill() {
    const right = findHeaderRight();
    if (!right) return;
    let pill = document.getElementById('opsLvRolePill');
    if (!pill) {
      pill = document.createElement('div');
      pill.id = 'opsLvRolePill';
      pill.className = 'header-pill header-pill-label';
      pill.style.maxWidth = '420px';
      pill.style.overflow = 'hidden';
      pill.style.textOverflow = 'ellipsis';
      pill.style.whiteSpace = 'nowrap';
      right.appendChild(pill);
    }
    const roles = (STATE.me && Array.isArray(STATE.me.roles)) ? STATE.me.roles : [];
    const roleTxt = roles.length ? roles.map(roleLabelForKey).join(', ') : t('—','—');
    pill.textContent = t('Rolle: ','Role: ') + roleTxt;
  }

  // --- Admin dialogs ---------------------------------------------------
  function ensureAdminPanelDialog() {
    ensureDialogTheme();
    let dlg = document.getElementById('opsAdminPanelDialog');
    if (dlg) return dlg;

    dlg = document.createElement('dialog');
    dlg.id = 'opsAdminPanelDialog';
    dlg.className = 'dialog';

    dlg.innerHTML = `
      <div class="card" style="min-width:760px; max-width:980px; padding:16px;">
        <div class="row" style="justify-content:space-between;">
          <h3 style="display:flex;align-items:center;gap:10px; margin:0;">
            <span>🔒</span>
            <span>${t('Admin','Admin')}</span>
          </h3>
          <button type="button" class="btn btn-ghost" id="opsAdminCloseX">✕</button>
        </div>

        <div class="muted" id="opsAdminInfo" style="margin:8px 0 12px 0;"></div>

        <div class="row" style="justify-content:flex-start;">
          <button type="button" class="btn btn-ghost" id="opsAdminUsersBtn">👥 ${t('Benutzer & Rollen','Users & roles')}</button>
          <button type="button" class="btn btn-ghost" id="opsAdminPermsBtn">🔐 ${t('Berechtigungen','Permissions')}</button>
          <button type="button" class="btn btn-primary" id="opsAdminBackupBtn">🧰 ${t('Backup (ZIP)','Backup (ZIP)')}</button>
          <button type="button" class="btn btn-ghost" id="opsAdminLogoutBtn">🚪 ${t('Logout','Logout')}</button>
        </div>

        <div class="ops-mini" style="margin-top:10px; opacity:.9;">
          ${t('Backup enthält Datenbank (SQLite) und Anhänge (falls vorhanden).','Backup contains the SQLite database and attachments (if present).')}
          <br/>
          ${t('Zusätzlich wird eine Kopie im Ordner "backend\\data\\backups" gespeichert.','A copy is also stored in the folder "backend\\data\\backups".')}
        </div>

        <div class="row" style="justify-content:flex-end; margin-top:14px;">
          <button type="button" class="btn" id="opsAdminCloseBtn">${t('Schliessen','Close')}</button>
        </div>
      </div>
    `;

    document.body.appendChild(dlg);

    dlg.querySelector('#opsAdminCloseX')?.addEventListener('click', () => closeDialog(dlg));
    dlg.querySelector('#opsAdminCloseBtn')?.addEventListener('click', () => closeDialog(dlg));

    dlg.querySelector('#opsAdminLogoutBtn')?.addEventListener('click', async () => {
      try {
        await api().logout();
      } catch (_) {}
      closeDialog(dlg);
      location.reload();
    });

    dlg.querySelector('#opsAdminUsersBtn')?.addEventListener('click', async () => {
      const u = await refreshMe(true);
      if (!u || !can('roles.manage')) {
        await uiAlert('Keine Berechtigung.','No permission.','⛔','⛔');
        return;
      }
      const ud = ensureUsersDialog();
      await renderUsersDialog(ud);
      openDialog(ud);
    });

    dlg.querySelector('#opsAdminPermsBtn')?.addEventListener('click', async () => {
      const u = await refreshMe(true);
      if (!u || !can('roles.manage')) {
        await uiAlert('Keine Berechtigung.','No permission.','⛔','⛔');
        return;
      }
      const pd = ensurePermsDialog();
      await renderPermsDialog(pd);
      openDialog(pd);
    });

    dlg.querySelector('#opsAdminBackupBtn')?.addEventListener('click', async () => {
      const u = await refreshMe(true);
      if (!u || !can('roles.manage')) {
        await uiAlert('Keine Berechtigung.','No permission.','⛔','⛔');
        return;
      }
      await downloadBackupZip(dlg.querySelector('#opsAdminBackupBtn'));
    });

    return dlg;
  }

  function setAdminInfo(dlg) {
    const info = dlg.querySelector('#opsAdminInfo');
    const u = STATE.me;
    if (!info) return;
    if (!u) {
      info.textContent = t('Nicht angemeldet.','Not logged in.');
      return;
    }
    const roles = Array.isArray(u.roles) ? u.roles : [];
    const roleTxt = roles.length ? roles.map(roleLabelForKey).join(', ') : t('—','—');
    info.textContent = t('Angemeldet als: ','Logged in as: ') + (u.display_name || u.username) + ' • ' + t('Rolle(n): ','Role(s): ') + roleTxt;
  }

  async function downloadBackupZip(btn) {
    const label0 = btn ? btn.textContent : '';
    try {
      if (btn) { btn.disabled = true; btn.textContent = t('Backup wird erstellt…','Creating backup…'); }

      const res = await fetch('/api/admin/backup', { method: 'GET', credentials: 'include' });
      if (!res.ok) {
        let detail = '';
        try {
          const j = await res.json();
          detail = j && j.detail ? String(j.detail) : '';
        } catch (_) {}
        if (res.status === 401) {
          await uiAlert('Bitte erneut einloggen.','Please log in again.','🔒','🔒');
        } else if (res.status === 403) {
          await uiAlert('Keine Berechtigung für Backup.','No permission for backup.','⛔','⛔');
        } else {
          await uiAlert('Backup konnte nicht erstellt werden.','Backup could not be created.','⚠️','⚠️');
        }
        console.warn('[backup] failed', res.status, detail);
        return;
      }

      const blob = await res.blob();
      const cd = res.headers.get('content-disposition') || '';
      let filename = 'opsdeck_backup.zip';
      const m = /filename\*=UTF-8''([^;]+)|filename="?([^";]+)"?/i.exec(cd);
      if (m) {
        filename = decodeURIComponent(m[1] || m[2] || filename);
      }

      const url = URL.createObjectURL(blob);
      const a = document.createElement('a');
      a.href = url;
      a.download = filename;
      a.style.display = 'none';
      document.body.appendChild(a);
      a.click();
      setTimeout(() => {
        try { URL.revokeObjectURL(url); } catch (_) {}
        try { a.remove(); } catch (_) {}
      }, 2000);

      await uiAlert(
        'Backup wurde heruntergeladen. Zusätzlich liegt eine Kopie im Ordner "backend\\data\\backups".',
        'Backup downloaded. A copy is also stored in the folder "backend\\data\\backups".',
        '✅ Backup',
        '✅ Backup'
      );
    } catch (e) {
      console.error(e);
      await uiAlert('Backup fehlgeschlagen.','Backup failed.','⚠️','⚠️');
    } finally {
      if (btn) { btn.disabled = false; btn.textContent = label0 || ('🧰 ' + t('Backup (ZIP)','Backup (ZIP)')); }
    }
  }

  // --- Users dialog ----------------------------------------------------
  function ensureUsersDialog() {
    ensureDialogTheme();
    let dlg = document.getElementById('opsUsersDialog');
    if (dlg) return dlg;

    dlg = document.createElement('dialog');
    dlg.id = 'opsUsersDialog';
    dlg.className = 'dialog';

    dlg.innerHTML = `
      <div class="card" style="min-width:920px; max-width:1100px; padding:16px;">
        <div class="row" style="justify-content:space-between;">
          <h3 style="display:flex;align-items:center;gap:10px;margin:0;">
            <span>👥</span>
            <span>${t('Benutzer & Rollen','Users & roles')}</span>
          </h3>
          <button type="button" class="btn btn-ghost" id="opsUsersCloseX">✕</button>
        </div>

        <div class="row" style="justify-content:space-between; margin:10px 0 10px;">
          <input id="opsUsersSearch" type="search" placeholder="${t('Suche (Name oder Benutzername)','Search (name or username)')}" style="flex:1; min-width:240px;" />
          <button type="button" class="btn btn-ghost" id="opsUsersReload">↻ ${t('Neu laden','Reload')}</button>
        </div>

        <div style="max-height:55vh; overflow:auto; border:1px solid rgba(148,163,184,.22); border-radius:12px;">
          <table class="ops-admin-table">
            <thead>
              <tr>
                <th style="min-width:180px;">${t('Benutzer','User')}</th>
                <th style="min-width:260px;">${t('Rollen','Roles')}</th>
                <th style="min-width:160px;">${t('Aktion','Action')}</th>
              </tr>
            </thead>
            <tbody id="opsUsersBody"></tbody>
          </table>
        </div>

        <div style="margin-top:14px; padding-top:12px; border-top:1px solid rgba(148,163,184,.22);">
          <h4 style="margin:0 0 10px;">${t('Neuen Benutzer anlegen','Create user')}</h4>
          <div class="row">
            <input id="opsNewUserUsername" type="text" placeholder="${t('Benutzername','Username')}" style="min-width:180px;" />
            <input id="opsNewUserDisplay" type="text" placeholder="${t('Anzeigename','Display name')}" style="min-width:220px; flex:1;" />
            <input id="opsNewUserPass" type="password" placeholder="${t('Passwort','Password')}" style="min-width:180px;" />
          </div>
          <div class="row" style="margin-top:10px;">
            <div id="opsNewUserRoles" class="ops-roles-wrap"></div>
            <div style="flex:1;"></div>
            <button type="button" class="btn btn-primary" id="opsCreateUserBtn">➕ ${t('Anlegen','Create')}</button>
          </div>
          <div id="opsUsersMsg" class="ops-mini" style="margin-top:8px;"></div>
        </div>

        <div class="row" style="justify-content:flex-end; margin-top:14px;">
          <button type="button" class="btn" id="opsUsersCloseBtn">${t('Schliessen','Close')}</button>
        </div>
      </div>
    `;

    document.body.appendChild(dlg);

    dlg.querySelector('#opsUsersCloseX')?.addEventListener('click', () => closeDialog(dlg));
    dlg.querySelector('#opsUsersCloseBtn')?.addEventListener('click', () => closeDialog(dlg));
    dlg.querySelector('#opsUsersReload')?.addEventListener('click', async () => { await renderUsersDialog(dlg); });
    dlg.querySelector('#opsUsersSearch')?.addEventListener('input', () => filterUsersTable(dlg));
    dlg.querySelector('#opsCreateUserBtn')?.addEventListener('click', async () => { await createUserFromDialog(dlg); });

    return dlg;
  }

  function setUsersMsg(dlg, msg, isErr=false) {
    const el = dlg.querySelector('#opsUsersMsg');
    if (!el) return;
    el.textContent = msg || '';
    el.className = 'ops-mini' + (isErr ? ' ops-danger' : '');
  }

  function filterUsersTable(dlg) {
    const q = String(dlg.querySelector('#opsUsersSearch')?.value || '').trim().toLowerCase();
    const rows = dlg.querySelectorAll('#opsUsersBody tr');
    rows.forEach((tr) => {
      const hay = String(tr.dataset.hay || '').toLowerCase();
      tr.style.display = (!q || hay.includes(q)) ? '' : 'none';
    });
  }

  async function renderUsersDialog(dlg) {
    setUsersMsg(dlg, '');
    const [rolesRes, usersRes] = await Promise.all([
      api().apiFetch('/rbac/roles'),
      api().apiFetch('/rbac/users')
    ]);

    const roles = (rolesRes && rolesRes.roles) ? rolesRes.roles : [];
    const users = (usersRes && usersRes.users) ? usersRes.users : [];

    // create user roles checkboxes
    const newRolesWrap = dlg.querySelector('#opsNewUserRoles');
    newRolesWrap.innerHTML = '';
    roles.forEach((r) => {
      const id = 'newrole_' + r.key;
      const lab = document.createElement('label');
      lab.className = 'ops-role-tag';
      lab.style.cursor = 'pointer';
      lab.innerHTML = `<input type="checkbox" id="${id}" data-role="${r.key}" style="margin-right:6px;" />${getLang()==='en' ? (r.name_en||r.key) : (r.name_de||r.key)}`;
      newRolesWrap.appendChild(lab);
    });

    const body = dlg.querySelector('#opsUsersBody');
    body.innerHTML = '';

    users.forEach((u) => {
      const tr = document.createElement('tr');
      tr.dataset.hay = (u.username||'') + ' ' + (u.display_name||'');

      const tdUser = document.createElement('td');
      tdUser.innerHTML = `<div style="font-weight:600;">${escapeHtml(u.display_name||u.username)}</div><div class="ops-mini" style="opacity:.85;">@${escapeHtml(u.username||'')}</div>`;

      const tdRoles = document.createElement('td');
      const wrap = document.createElement('div');
      wrap.className = 'ops-roles-wrap';
      const current = new Set(Array.isArray(u.roles) ? u.roles : []);
      roles.forEach((r) => {
        const lab = document.createElement('label');
        lab.className = 'ops-role-tag';
        lab.style.cursor = 'pointer';
        const cb = document.createElement('input');
        cb.type = 'checkbox';
        cb.dataset.role = r.key;
        cb.checked = current.has(r.key);
        cb.style.marginRight = '6px';
        lab.appendChild(cb);
        lab.appendChild(document.createTextNode(getLang()==='en' ? (r.name_en||r.key) : (r.name_de||r.key)));
        wrap.appendChild(lab);
      });
      tdRoles.appendChild(wrap);

      const tdAct = document.createElement('td');
      tdAct.style.whiteSpace = 'nowrap';
      const saveBtn = document.createElement('button');
      saveBtn.type = 'button';
      saveBtn.className = 'btn btn-primary';
      saveBtn.textContent = t('Speichern','Save');
      saveBtn.addEventListener('click', async () => {
        const selected = Array.from(wrap.querySelectorAll('input[type=checkbox]'))
          .filter(cb => cb.checked)
          .map(cb => cb.dataset.role);
        saveBtn.disabled = true;
        const old = saveBtn.textContent;
        saveBtn.textContent = t('Speichert…','Saving…');
        try {
          await api().apiFetch(`/rbac/users/${u.id}/roles`, { method:'PUT', body:{ roles: selected } });
          setUsersMsg(dlg, t('Rollen gespeichert.','Roles saved.'));
          await refreshMe(true);
          applyUi();
          ensureRolePill();
        } catch (e) {
          console.error(e);
          setUsersMsg(dlg, t('Konnte nicht speichern.','Could not save.'), true);
        } finally {
          saveBtn.disabled = false;
          saveBtn.textContent = old;
        }
      });
      tdAct.appendChild(saveBtn);

      tr.appendChild(tdUser);
      tr.appendChild(tdRoles);
      tr.appendChild(tdAct);
      body.appendChild(tr);
    });

    filterUsersTable(dlg);
  }

  async function createUserFromDialog(dlg) {
    const username = String(dlg.querySelector('#opsNewUserUsername')?.value || '').trim();
    const display = String(dlg.querySelector('#opsNewUserDisplay')?.value || '').trim();
    const pass = String(dlg.querySelector('#opsNewUserPass')?.value || '');

    const roles = Array.from(dlg.querySelectorAll('#opsNewUserRoles input[type=checkbox]'))
      .filter(cb => cb.checked)
      .map(cb => cb.dataset.role);

    if (!username || !pass) {
      setUsersMsg(dlg, t('Bitte Benutzername und Passwort angeben.','Please provide username and password.'), true);
      return;
    }

    const btn = dlg.querySelector('#opsCreateUserBtn');
    const old = btn.textContent;
    btn.disabled = true;
    btn.textContent = t('Legt an…','Creating…');

    try {
      await api().apiFetch('/rbac/users', { method:'POST', body:{ username, display_name: display, password: pass, roles } });
      setUsersMsg(dlg, t('Benutzer angelegt.','User created.'));
      dlg.querySelector('#opsNewUserUsername').value = '';
      dlg.querySelector('#opsNewUserDisplay').value = '';
      dlg.querySelector('#opsNewUserPass').value = '';
      await renderUsersDialog(dlg);
    } catch (e) {
      console.error(e);
      const msg = (e && e.message === 'username_exists') ? t('Benutzername existiert bereits.','Username already exists.') : t('Konnte Benutzer nicht anlegen.','Could not create user.');
      setUsersMsg(dlg, msg, true);
    } finally {
      btn.disabled = false;
      btn.textContent = old;
    }
  }

  // --- Permissions matrix dialog --------------------------------------
  function ensurePermsDialog() {
    ensureDialogTheme();
    let dlg = document.getElementById('opsPermsDialog');
    if (dlg) return dlg;

    dlg = document.createElement('dialog');
    dlg.id = 'opsPermsDialog';
    dlg.className = 'dialog';

    dlg.innerHTML = `
      <div class="card" style="min-width:920px; max-width:1200px; padding:16px;">
        <div class="row" style="justify-content:space-between;">
          <h3 style="display:flex;align-items:center;gap:10px;margin:0;">
            <span>🔐</span>
            <span>${t('Berechtigungs-Matrix','Permission matrix')}</span>
          </h3>
          <button type="button" class="btn btn-ghost" id="opsPermsCloseX">✕</button>
        </div>

        <div class="muted" style="margin:8px 0 12px 0;">
          ${t('Änderungen gelten systemweit (SQLite).','Changes apply system-wide (SQLite).')}
        </div>

        <div style="max-height:62vh; overflow:auto; border:1px solid rgba(148,163,184,.22); border-radius:12px;">
          <table class="ops-admin-table">
            <thead><tr id="opsPermsHead"></tr></thead>
            <tbody id="opsPermsBody"></tbody>
          </table>
        </div>

        <div class="row" style="justify-content:flex-end; margin-top:14px;">
          <button type="button" class="btn btn-ghost" id="opsPermsReload">↻ ${t('Neu laden','Reload')}</button>
          <button type="button" class="btn btn-primary" id="opsPermsSave">${t('Speichern','Save')}</button>
        </div>

        <div class="row" style="justify-content:flex-end; margin-top:10px;">
          <button type="button" class="btn" id="opsPermsCloseBtn">${t('Schliessen','Close')}</button>
        </div>
      </div>
    `;

    document.body.appendChild(dlg);

    dlg.querySelector('#opsPermsCloseX')?.addEventListener('click', () => closeDialog(dlg));
    dlg.querySelector('#opsPermsCloseBtn')?.addEventListener('click', () => closeDialog(dlg));
    dlg.querySelector('#opsPermsReload')?.addEventListener('click', async () => { await renderPermsDialog(dlg); });
    dlg.querySelector('#opsPermsSave')?.addEventListener('click', async () => { await savePermsDialog(dlg); });

    return dlg;
  }

  async function renderPermsDialog(dlg) {
    const payload = await api().apiFetch('/rbac/matrix');
    const roles = payload.roles || [];
    const perms = payload.permissions || [];
    const matrix = payload.matrix || {};

    dlg.__roles = roles;
    dlg.__perms = perms;
    dlg.__matrix = JSON.parse(JSON.stringify(matrix));

    const head = dlg.querySelector('#opsPermsHead');
    const body = dlg.querySelector('#opsPermsBody');
    head.innerHTML = '';
    body.innerHTML = '';

    const th0 = document.createElement('th');
    th0.textContent = t('Aktivität','Activity');
    th0.style.minWidth = '260px';
    head.appendChild(th0);

    roles.forEach((r) => {
      const th = document.createElement('th');
      th.textContent = (getLang()==='en' ? (r.name_en||r.key) : (r.name_de||r.key));
      th.style.minWidth = '140px';
      head.appendChild(th);
    });

    perms.forEach((p) => {
      const tr = document.createElement('tr');
      const tdA = document.createElement('td');
      tdA.innerHTML = `<div style="font-weight:600;">${escapeHtml(getLang()==='en' ? (p.name_en||p.key) : (p.name_de||p.key))}</div><div class="ops-mini" style="opacity:.75;">${escapeHtml(p.key)}</div>`;
      tr.appendChild(tdA);

      roles.forEach((r) => {
        const td = document.createElement('td');
        td.style.textAlign = 'center';
        const cb = document.createElement('input');
        cb.type = 'checkbox';
        const current = dlg.__matrix[r.key] || {};
        cb.checked = !!current[p.key];

        // Safety: systemadmin always keeps roles.manage (and backend will enforce systemadmin all perms)
        if (r.key === 'systemadmin' && p.key === 'roles.manage') {
          cb.checked = true;
          cb.disabled = true;
        }

        cb.addEventListener('change', () => {
          dlg.__matrix[r.key] = dlg.__matrix[r.key] || {};
          dlg.__matrix[r.key][p.key] = !!cb.checked;
        });

        td.appendChild(cb);
        tr.appendChild(td);
      });

      body.appendChild(tr);
    });
  }

  async function savePermsDialog(dlg) {
    const btn = dlg.querySelector('#opsPermsSave');
    const old = btn.textContent;
    btn.disabled = true;
    btn.textContent = t('Speichert…','Saving…');

    try {
      await api().apiFetch('/rbac/matrix', { method:'PUT', body:{ matrix: dlg.__matrix || {} } });
      await uiAlert('Berechtigungen gespeichert.','Permissions saved.','✅','✅');
      await refreshMe(true);
      applyUi();
      ensureRolePill();
    } catch (e) {
      console.error(e);
      await uiAlert('Konnte nicht speichern.','Could not save.','⚠️','⚠️');
    } finally {
      btn.disabled = false;
      btn.textContent = old;
    }
  }

  // --- Binding buttons -------------------------------------------------
  function bindAdminButtons() {
    const adminBtn = document.getElementById('invAdminBtn');
    const permsBtn = document.getElementById('invPermsBtn');

    if (adminBtn && !adminBtn.dataset.boundServerRbac) {
      adminBtn.dataset.boundServerRbac = '1';
      adminBtn.addEventListener('click', async (e) => {
        e.preventDefault();
        e.stopPropagation();
        const u = await refreshMe(true);
        if (!u) {
          await uiAlert('Bitte einloggen.','Please log in.','🔒','🔒');
          return;
        }
        const dlg = ensureAdminPanelDialog();
        setAdminInfo(dlg);
        openDialog(dlg);
      }, true);
    }

    if (permsBtn && !permsBtn.dataset.boundServerRbac) {
      permsBtn.dataset.boundServerRbac = '1';
      permsBtn.addEventListener('click', async (e) => {
        e.preventDefault();
        e.stopPropagation();
        const u = await refreshMe(true);
        if (!u) {
          await uiAlert('Bitte einloggen.','Please log in.','🔒','🔒');
          return;
        }
        if (!can('roles.manage')) {
          await uiAlert('Keine Berechtigung.','No permission.','⛔','⛔');
          return;
        }
        const dlg = ensurePermsDialog();
        await renderPermsDialog(dlg);
        openDialog(dlg);
      }, true);
    }
  }

  // --- helpers ---------------------------------------------------------
  function escapeHtml(s) {
    return String(s || '')
      .replaceAll('&', '&amp;')
      .replaceAll('<', '&lt;')
      .replaceAll('>', '&gt;')
      .replaceAll('"', '&quot;')
      .replaceAll("'", '&#039;');
  }

  // --- boot ------------------------------------------------------------
  async function boot() {
    // Wait for offline client readiness (login + initial sync)
    if (window.__opsOfflineReady) {
      try { await window.__opsOfflineReady; } catch (_) {}
    }

    await refreshMe(true);
    ensureDialogTheme();
    ensureRolePill();
    bindAdminButtons();
    applyUi();
  }

  function bootSoon() {
    boot().catch((e) => console.error('[rbac]', e));
  }

  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', bootSoon);
  } else {
    bootSoon();
  }
})();
