/* ops-lagerverwaltung.banf.js
   Add-on für ops-lagerverwaltung.js (ohne Umbau am bestehenden System)
   - fügt oben einen Button "🧾 BANF / Bestellung" hinzu (falls nicht vorhanden)
   - erzeugt BANF aus ausgewählten Tabellenzeilen (DOM-basiert, kein Zugriff auf internes state nötig)
   - Drucklayout inkl. Logo, Datum & Unterschrift
   - Export JSON (BANF + SAP-Payload Stub)

   Hinweis: Dieses Add-on muss NACH ops-lagerverwaltung.js geladen werden (defer + Reihenfolge).
*/

(() => {
  "use strict";

  const APP = {
    company: "OpsDeck360",
    logoSrc: "icon-512.png",
    currency: "EUR"
  };

  // --- Language helpers (DE/EN) -----------------------------------------
  function opsLang() {
    try {
      if (typeof window.getOpsLang === "function") return window.getOpsLang();
    } catch (e) {}
    const l = document.documentElement.getAttribute("lang") || "de";
    return String(l).toLowerCase().startsWith("en") ? "en" : "de";
  }
  function tLang(deText, enText) {
    return opsLang() === "en" ? enText : deText;
  }

  // --- Modern info dialog (replaces window.alert) ------------------------
  const INFO_DLG_ID = "invBanfInfoDialog";
  function ensureInfoDialog() {
    let dlg = $(INFO_DLG_ID);
    if (dlg) return dlg;

    dlg = document.createElement("dialog");
    dlg.id = INFO_DLG_ID;
    dlg.className = "dialog";
    dlg.innerHTML = `
      <div class="card" style="max-width:520px;width:92vw;">
        <h3 id="invBanfInfoTitle">${esc(tLang("Hinweis", "Notice"))}</h3>
        <p id="invBanfInfoMsg" style="white-space:pre-line;margin-top:8px;"></p>
        <div style="margin-top:14px;display:flex;justify-content:flex-end;gap:8px;">
          <button type="button" class="btn btn-primary" id="invBanfInfoOk">OK</button>
        </div>
      </div>
    `;
    document.body.appendChild(dlg);
    dlg.addEventListener("click", (e) => { if (e.target === dlg) dlg.close(); });

    // Apply current language to labels/buttons
    applyBanfI18n();
    dlg.querySelector("#invBanfInfoOk")?.addEventListener("click", () => dlg.close());
    return dlg;
  }

  function showInfo(message, title) {
    const dlg = ensureInfoDialog();
    dlg.querySelector("#invBanfInfoTitle").textContent = title || tLang("Hinweis", "Notice");
    dlg.querySelector("#invBanfInfoMsg").textContent = message || "";
    dlg.querySelector("#invBanfInfoOk").textContent = "OK";
    applyBanfI18n();
    dlg.showModal();
  }


  const $ = (id) => document.getElementById(id);
  const esc = (v) => (v == null ? "" : String(v))
    .replace(/&/g, "&amp;")
    .replace(/</g, "&lt;")
    .replace(/>/g, "&gt;")
    .replace(/"/g, "&quot;")
    .replace(/'/g, "&#039;");

  const num = (v, fb = 0) => {
    if (v == null) return fb;
    if (typeof v === "number") return Number.isFinite(v) ? v : fb;
    let s = String(v).trim();
    if (!s) return fb;
    s = s.replace(/\s/g, "").replace(/[^0-9,.\-]/g, "");
    if (s.includes(",") && s.includes(".")) s = s.replace(/\./g, "").replace(",", ".");
    else if (s.includes(",")) s = s.replace(",", ".");
    const n = Number(s);
    return Number.isFinite(n) ? n : fb;
  };

  const uid = (p="banf") => `${p}_${Math.random().toString(36).slice(2)}_${Date.now().toString(36)}`;

  function ensureTopButton() {
    const root = $("lagerverwaltung");
    if (!root) return;

    // Helper: bind click exactly once
    function wire(btn) {
      if (!btn) return;
      if (btn.dataset && btn.dataset.banfBound === "1") return;
      if (btn.dataset) btn.dataset.banfBound = "1";

      btn.title = tLang("BANF / Bestellung öffnen", "Open requisition / order");
      btn.textContent = tLang("🧾 BANF / Bestellung", "🧾 Requisition / Order");

      btn.addEventListener("click", (e) => {
        e.preventDefault();
        e.stopImmediatePropagation();
        openBanfDialog();
      }, { capture: true });
    }

    // If button already exists (maybe created by ops-lagerverwaltung.js), reuse it
    let btn = $("invBanfBtn");

    // Prefer: place BANF inside the rubric grid (if present)
    const rubrics = root.querySelector(".inv-actions-rubrics");
    if (rubrics) {
      // Create or reuse rubric card
      let card = root.querySelector("#invBanfRubric");
      if (!card) {
        card = document.createElement("div");
        card.id = "invBanfRubric";
        card.className = "rubric-card";

        card.innerHTML = `
          <div class="rubric-head">
            <span class="rubric-icon">🛒</span>
            <span class="rubric-title">${esc(tLang("Einkauf", "Purchasing"))}</span>
          </div>
          <div class="rubric-body" id="invBanfRubricBody"></div>
        `;

        // Insert near the beginning (before first rubric)
        rubrics.insertBefore(card, rubrics.firstChild);
      } else {
        // If it already exists, keep title synced
        const titleEl = card.querySelector(".rubric-title");
        if (titleEl) titleEl.textContent = tLang("Einkauf", "Purchasing");
      }

      const body = card.querySelector("#invBanfRubricBody") || card.querySelector(".rubric-body");
      if (!body) return;

      // Create button if missing
      if (!btn) {
        btn = document.createElement("button");
        btn.id = "invBanfBtn";
        btn.className = "btn btn-ghost";
      } else {
        // Ensure styling if reused
        if (!btn.classList.contains("btn")) btn.classList.add("btn");
        if (!btn.classList.contains("btn-ghost")) btn.classList.add("btn-ghost");
      }

      wire(btn);

      // Move/append button into the rubric body
      if (btn.parentElement !== body) body.appendChild(btn);
      return;
    }

    // Fallback: old behavior (top button row)
    const headerRight = root.querySelector(".card-header > div:last-child");
    if (!headerRight) return;

    if (!btn) {
      btn = document.createElement("button");
      btn.id = "invBanfBtn";
      btn.className = "btn btn-ghost";
      headerRight.insertBefore(btn, headerRight.firstChild);
    }
    wire(btn);
  }

  function ensureDialog() {
    let dlg = $("invBanfDialog");
    if (dlg) return dlg;

    dlg = document.createElement("dialog");
    dlg.id = "invBanfDialog";
    dlg.className = "dialog";

    dlg.innerHTML = `
      <div class="card" style="max-width:980px;width:92vw;">
        <h3>BANF / Bestellung (Entwurf)</h3>
        <p class="tiny" style="margin-top:-6px;opacity:.85;">
          Erstellt eine BANF aus markierten Zeilen (Checkbox links). Tipp: erst filtern, dann markieren.
        </p>

        <form id="invBanfForm">
          <div class="grid-2" style="gap:8px;">
            <div>
              <label for="invBanfRequester">Anforderer</label>
              <input id="invBanfRequester" type="text" placeholder="z.B. Max Mustermann">
            </div>
            <div>
              <label for="invBanfCostCenter">Kostenstelle</label>
              <input id="invBanfCostCenter" type="text" placeholder="z.B. KST-1001">
            </div>

            <div>
              <label for="invBanfSupplier">Lieferant (optional)</label>
              <input id="invBanfSupplier" type="text" placeholder="z.B. Würth, RS, …">
            </div>
            <div>
              <label for="invBanfNeedBy">Bedarfstermin</label>
              <input id="invBanfNeedBy" type="date">
            </div>

            <div class="grid-span-2">
              <label for="invBanfNote">Bemerkung</label>
              <textarea id="invBanfNote" rows="2" placeholder="z.B. Wartung KW03, dringend…"></textarea>
            </div>
          </div>

          <div style="margin-top:12px;">
            <div class="tiny" id="invBanfLinesTitle" style="margin-bottom:6px;">Positionen</div>
            <div style="max-height: 42vh; overflow:auto; border:1px solid rgba(148,163,184,0.35); border-radius:8px;">
              <table class="data-table" style="width:100%;">
                <thead>
                  <tr>
                    <th style="width:56px;">${tLang("Menge","Qty")}</th>
                    <th id="invBanfThMaterial">${tLang("Artikelnummer","Item no.")}</th>
                    <th id="invBanfThDesc">${tLang("Bezeichnung","Description")}</th>
                    <th id="invBanfThSupplier">${tLang("Lieferant","Supplier")}</th>
                    <th style="width:80px;">${tLang("Min","Min")}</th>
                    <th style="width:80px;">${tLang("Bestand","Stock")}</th>
                    <th style="width:110px;">${tLang("Vorschlag","Suggestion")}</th>
                  </tr>
                </thead>
                <tbody id="invBanfLines"></tbody>
              </table>
            </div>
          </div>

          <div class="grid-2" style="gap:10px;margin-top:12px;">
            <button type="button" class="btn btn-ghost" id="invBanfCancel">Abbrechen</button>
            <div style="display:flex;justify-content:flex-end;gap:8px;">
              <button type="button" class="btn btn-ghost" id="invBanfPrint">🖨 BANF drucken</button>
              <button type="button" class="btn btn-ghost" id="invBanfExport">⬇️ Export (JSON/SAP)</button>
              <button type="submit" class="btn btn-primary">BANF speichern</button>
            </div>
          </div>
        </form>
      </div>
    `;

    document.body.appendChild(dlg);
    dlg.addEventListener("click", (e) => { if (e.target === dlg) dlg.close(); });

    // Apply current language to labels/buttons
    applyBanfI18n();

    // Cancel
    $("invBanfCancel")?.addEventListener("click", (e) => { e.preventDefault(); dlg.close(); });

    // Default Bedarfstermin: +7 Tage
    const d = new Date(); d.setDate(d.getDate() + 7);
    const needBy = `${d.getFullYear()}-${String(d.getMonth()+1).padStart(2,"0")}-${String(d.getDate()).padStart(2,"0")}`;
    const needEl = $("invBanfNeedBy");
    if (needEl && !needEl.value) needEl.value = needBy;

    // Submit: speichern
    $("invBanfForm")?.addEventListener("submit", (e) => {
      e.preventDefault();
      const draft = collectDraftFromDialog();
      if (!draft.lines.length) { showInfo(tLang("Bitte mindestens eine Position mit Menge > 0 angeben.", "Please add at least one line item with quantity > 0."), tLang("🧾 BANF / Bestellung", "🧾 Requisition / Order")); return; }
      const docs = loadBanfDocs();
      docs.unshift({ id: uid("banf"), ts: new Date().toISOString(), status: "draft", ...draft });
      saveBanfDocs(docs);
      dlg.close();
      showInfo(tLang("BANF gespeichert (Entwurf).", "Requisition saved (draft)."), tLang("🧾 BANF / Bestellung", "🧾 Requisition / Order"));
    });

    // Print
    $("invBanfPrint")?.addEventListener("click", (e) => {
      e.preventDefault();
      const draft = collectDraftFromDialog();
      if (!draft.lines.length) { showInfo(tLang("Keine Positionen (Menge 0).", "No line items (quantity is 0)."), tLang("🧾 BANF / Bestellung", "🧾 Requisition / Order")); return; }
      printBanf({ id: uid("banf-print"), ts: new Date().toISOString(), ...draft });
    });

    // Export
    $("invBanfExport")?.addEventListener("click", (e) => {
      e.preventDefault();
      const draft = collectDraftFromDialog();
      if (!draft.lines.length) { showInfo(tLang("Keine Positionen (Menge 0).", "No line items (quantity is 0)."), tLang("🧾 BANF / Bestellung", "🧾 Requisition / Order")); return; }
      const banfDoc = { id: uid("banf"), ts: new Date().toISOString(), status: "draft", ...draft };
      downloadJson(`banf-${new Date().toISOString().slice(0,10)}.json`, banfDoc);
      downloadJson(`banf-${new Date().toISOString().slice(0,10)}-sap-payload.json`, buildSapPayload(banfDoc));
    });

    return dlg;
  }

  function applyBanfI18n() {
    // Top button
    const btn = $("invBanfBtn");
    if (btn) {
      btn.textContent = tLang("🧾 BANF / Bestellung", "🧾 Requisition / Order");
      btn.title = tLang("BANF / Bestellung öffnen", "Open requisition / order");
    }

    const dlg = $("invBanfDialog");
    if (!dlg) return;

    // Title + help
    const h3 = dlg.querySelector("h3");
    if (h3) h3.textContent = tLang("BANF / Bestellung (Entwurf)", "Requisition / Order (Draft)");

    const help = dlg.querySelector("p.tiny");
    if (help) help.textContent = tLang(
      "Erstellt eine BANF aus markierten Zeilen (Checkbox links). Tipp: erst filtern, dann markieren.",
      "Creates a requisition from selected rows (left checkbox). Tip: filter first, then select."
    );

    // Labels
    const setLabel = (forId, deText, enText) => {
      const lab = dlg.querySelector(`label[for="${forId}"]`);
      if (lab) lab.textContent = tLang(deText, enText);
    };
    setLabel("invBanfRequester", "Anforderer", "Requester");
    setLabel("invBanfCostCenter", "Kostenstelle", "Cost center");
    setLabel("invBanfSupplier", "Lieferant (optional)", "Supplier (optional)");
    setLabel("invBanfNeedBy", "Bedarfstermin", "Need-by date");
    setLabel("invBanfNote", "Bemerkung", "Note");

    // Placeholders
    const req = $("invBanfRequester");
    if (req) req.placeholder = tLang("z.B. Max Mustermann", "e.g. Alex Example");
    const cc = $("invBanfCostCenter");
    if (cc) cc.placeholder = tLang("z.B. KST-1001", "e.g. CC-1001");
    const sup = $("invBanfSupplier");
    if (sup) sup.placeholder = tLang("z.B. Würth, RS, …", "e.g. Würth, RS, …");
    const note = $("invBanfNote");
    if (note) note.placeholder = tLang("z.B. Wartung KW03, dringend…", "e.g. Maintenance week 03, urgent…");

    // Titles / headers / buttons
    const lt = $("invBanfLinesTitle");
    if (lt) lt.textContent = tLang("Positionen", "Line items");

    // Table headers (if dialog already exists)
    const setTh = (id, deText, enText) => {
      const th = $(id);
      if (th) th.textContent = tLang(deText, enText);
    };
    setTh("invBanfThMaterial", "Artikelnummer", "Item no.");
    setTh("invBanfThDesc", "Bezeichnung", "Description");
    setTh("invBanfThSupplier", "Lieferant", "Supplier");

    const printBtn = $("invBanfPrint");
    if (printBtn) printBtn.textContent = tLang("🖨 BANF drucken", "🖨 Print requisition");
    const expBtn = $("invBanfExport");
    if (expBtn) expBtn.textContent = tLang("⬇️ Export (JSON/SAP)", "⬇️ Export (JSON/SAP)");
    const cancelBtn = $("invBanfCancel");
    if (cancelBtn) cancelBtn.textContent = tLang("Abbrechen", "Cancel");

    const submit = dlg.querySelector('button[type="submit"]');
    if (submit) submit.textContent = tLang("BANF speichern", "Save requisition");

    // Section headers
    const posHdr = Array.from(dlg.querySelectorAll(".tiny")).find(el => (el.textContent || "").trim() === "Positionen");
    if (posHdr) posHdr.textContent = tLang("Positionen", "Line items");

    // Table headers by index
    const ths = dlg.querySelectorAll("table thead th");
    const headerTexts = [
      [ "Menge", "Qty" ],
      [ "Artikelnummer", "Item no." ],
      [ "Bezeichnung", "Description" ],
      [ "Lieferant", "Supplier" ],
      [ "Min", "Min" ],
      [ "Bestand", "Stock" ],
      [ "Vorschlag", "Suggestion" ]
    ];
    ths.forEach((th, i) => {
      if (headerTexts[i]) th.textContent = tLang(headerTexts[i][0], headerTexts[i][1]);
    });

    // Buttons
    const cancel = $("invBanfCancel");
    if (cancel) cancel.textContent = tLang("Abbrechen", "Cancel");
    const print = $("invBanfPrint");
    if (print) print.textContent = tLang("🖨 BANF drucken", "🖨 Print");
    const exp = $("invBanfExport");
    if (exp) exp.textContent = tLang("⬇️ Export (JSON/SAP)", "⬇️ Export (JSON/SAP)");
    const save = dlg.querySelector('button[type="submit"]');
    if (save) save.textContent = tLang("BANF speichern", "Save draft");
  }


  function loadBanfDocs() {
    try {
      const raw = localStorage.getItem("opsdeck_banf_docs_v1");
      const arr = raw ? JSON.parse(raw) : [];
      return Array.isArray(arr) ? arr : [];
    } catch { return []; }
  }
  function saveBanfDocs(arr) {
    try { localStorage.setItem("opsdeck_banf_docs_v1", JSON.stringify(arr || [])); } catch {}
  }

  function getSelectedRows() {
    const body = $("invTableBody");
    if (!body) return [];
    const rows = Array.from(body.querySelectorAll("tr"));
    const selected = rows.filter(tr => {
      const cb = tr.querySelector("input.inv-row-select") || tr.querySelector('input[type="checkbox"]');
      return cb && cb.checked;
    });
    return selected.length ? selected : [];
  }

  function readCell(tr, colKey) {
    const td = tr.querySelector(`td[data-col="${colKey}"]`);
    return td ? (td.textContent || "").trim() : "";
  }

  function buildLinesFromSelectedRows() {
    const rows = getSelectedRows();
    if (!rows.length) return [];

    return rows.map(tr => {
      const artikelnummer = readCell(tr, "artikelnummer");
      const bezeichnung = readCell(tr, "bezeichnung");
      const lieferant = readCell(tr, "lieferant");
      const min = num(readCell(tr, "mindestbestand"), 0);
      const best = num(readCell(tr, "bestand"), 0);
      const einheit = readCell(tr, "einheit");
      const einzelpreis = num(readCell(tr, "einzelpreis"), 0); // falls leer: 0

      const vorschlag = Math.max(0, min - best);
      return {
        lineId: uid("l"),
        artikelnummer,
        bezeichnung,
        lieferant,
        mindestbestand: min,
        bestand: best,
        menge: vorschlag || 1,
        einheit,
        einzelpreis,
        waehrung: APP.currency
      };
    }).filter(l => l.artikelnummer || l.bezeichnung);
  }

  function openBanfDialog() {
    const dlg = ensureDialog();
    const tbody = $("invBanfLines");
    if (!tbody) return;

    const lines = buildLinesFromSelectedRows();
    if (!lines.length) {
      showInfo(tLang("Bitte erst mindestens eine Zeile markieren (Checkbox links) und dann erneut BANF klicken.", "Please select at least one row (left checkbox), then click BANF again."), tLang("🧾 BANF / Bestellung", "🧾 Requisition / Order"));
      return;
    }

    tbody.innerHTML = lines.map(l => `
      <tr>
        <td><input class="banf-qty" type="number" min="0" step="1" value="${esc(l.menge)}" style="width:72px;"></td>
        <td class="banf-artikel">${esc(l.artikelnummer)}</td>
        <td class="banf-bez">${esc(l.bezeichnung)}</td>
        <td class="banf-lief">${esc(l.lieferant)}</td>
        <td style="text-align:right;" class="banf-min">${esc(l.mindestbestand)}</td>
        <td style="text-align:right;" class="banf-best">${esc(l.bestand)}</td>
        <td class="tiny">${l.menge ? tLang("unter Mindestbestand", "below min stock") : ""}</td>
      </tr>
    `).join("");

    applyBanfI18n();
    dlg.showModal();
  }

  function collectDraftFromDialog() {
    const requester = ($("invBanfRequester")?.value || "").trim();
    const costCenter = ($("invBanfCostCenter")?.value || "").trim();
    const supplier = ($("invBanfSupplier")?.value || "").trim();
    const needBy = ($("invBanfNeedBy")?.value || "").trim();
    const note = ($("invBanfNote")?.value || "").trim();

    const tbody = $("invBanfLines");
    const lines = [];
    if (tbody) {
      tbody.querySelectorAll("tr").forEach((tr) => {
        const qty = num(tr.querySelector("input.banf-qty")?.value, 0);
        if (qty <= 0) return;

        lines.push({
          lineId: uid("l"),
          artikelnummer: (tr.querySelector(".banf-artikel")?.textContent || "").trim(),
          bezeichnung: (tr.querySelector(".banf-bez")?.textContent || "").trim(),
          lieferant: supplier || (tr.querySelector(".banf-lief")?.textContent || "").trim(),
          mindestbestand: num(tr.querySelector(".banf-min")?.textContent, 0),
          bestand: num(tr.querySelector(".banf-best")?.textContent, 0),
          menge: qty,
          einheit: "",
          einzelpreis: 0,
          waehrung: APP.currency
        });
      });
    }

    return { requester, costCenter, supplier, needBy, note, lines };
  }

  function downloadJson(filename, obj) {
    const json = JSON.stringify(obj, null, 2);
    const blob = new Blob([json], { type: "application/json;charset=utf-8" });
    const url = URL.createObjectURL(blob);
    const a = document.createElement("a");
    a.href = url;
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    document.body.removeChild(a);
    URL.revokeObjectURL(url);
  }

  function buildSapPayload(banfDoc) {
    // SAP Stub: bitte an eure Schnittstelle anpassen (Materialnummer / Account Assignment / Plant etc.)
    return {
      documentDate: banfDoc.ts,
      requester: banfDoc.requester || "",
      costCenter: banfDoc.costCenter || "",
      supplier: banfDoc.supplier || "",
      needBy: banfDoc.needBy || "",
      note: banfDoc.note || "",
      items: (banfDoc.lines || []).map((l, i) => ({
        itemNo: String(i + 1).padStart(5, "0"),
        material: l.artikelnummer || "",
        shortText: l.bezeichnung || "",
        quantity: Number(l.menge || 0),
        unit: l.einheit || "",
        accountAssignment: { costCenter: banfDoc.costCenter || "" },
        supplier: l.lieferant || banfDoc.supplier || ""
      }))
    };
  }

  function openPrint(html) {
    const win = window.open("", "_blank", "width=980,height=760");
    if (!win) { showInfo(tLang("Popup-Blocker aktiv – bitte Popups erlauben, um zu drucken.", "Pop-up blocker is enabled — allow pop-ups to print."), tLang("Drucken", "Printing")); return; }
    win.document.open(); win.document.write(html); win.document.close();
  }

  function printBanf(banfDoc) {
    const dt = new Date(banfDoc.ts || Date.now());
    const isEn = opsLang() === "en";
    const dateStr = dt.toLocaleDateString(isEn ? "en-GB" : "de-DE");
    const timeStr = dt.toLocaleTimeString(isEn ? "en-GB" : "de-DE", { hour:"2-digit", minute:"2-digit" });

    const rows = (banfDoc.lines || []).map((l, idx) => `
      <tr>
        <td style="text-align:right;">${idx+1}</td>
        <td>${esc(l.artikelnummer)}</td>
        <td>${esc(l.bezeichnung)}</td>
        <td>${esc(l.lieferant || "")}</td>
        <td style="text-align:right;">${esc(l.menge)}</td>
        <td>${esc(l.einheit || "")}</td>
      </tr>
    `).join("");

    const html = `<!doctype html>
<html lang="${isEn ? "en" : "de"}">
<head>
<meta charset="utf-8">
<title>${isEn ? "Requisition – Draft" : "BANF – Entwurf"}</title>
<style>
  body { font-family: system-ui,-apple-system,BlinkMacSystemFont,"Segoe UI",sans-serif; margin: 15mm; }
  .hdr { display:flex; justify-content:space-between; align-items:flex-start; border-bottom: 1px solid #444; padding-bottom: 6px; }
  .hdr-left { display:flex; gap:10px; align-items:flex-start; }
  .logo { width: 40mm; height:auto; }
  .company { font-size: 10pt; text-transform: uppercase; }
  .title { font-size: 16pt; font-weight: 700; margin-top: 2px; }
  .meta { font-size: 9pt; text-align:right; }
  h2 { margin: 10mm 0 4mm; font-size: 12pt; }
  table { width:100%; border-collapse: collapse; font-size: 10pt; }
  thead th { text-align:left; padding: 4px 6px; border-bottom: 1px solid #9ca3af; background: #e5e7eb; }
  tbody td { padding: 3px 6px; border-bottom: 1px solid #e5e7eb; vertical-align: top; }
  tbody tr:nth-child(even) td { background: #f9fafb; }
  .grid { display:grid; grid-template-columns: 1fr 1fr; gap: 10mm; margin-top: 6mm; }
  .sign { margin-top: 14mm; display:grid; grid-template-columns: 1fr 1fr; gap: 12mm; }
  .line { margin-top: 18mm; border-top: 1px solid #222; padding-top: 3mm; font-size: 10pt; }
</style>
</head>
<body>
  <div class="hdr">
    <div class="hdr-left">
      <img class="logo" src="${esc(APP.logoSrc)}" alt="Logo" onerror="this.style.display='none'" />
      <div>
        <div class="company">${esc(APP.company)}</div>
        <div class="title">${isEn ? "Requisition (Draft)" : "BANF (Entwurf)"}</div>
        <div class="meta" style="text-align:left;font-size:9pt;opacity:.85;">ID: ${esc(banfDoc.id || "-")}</div>
      </div>
    </div>
    <div class="meta">
      <div>${esc(isEn ? "Date" : "Datum")}: ${esc(dateStr)}</div>
      <div>${esc(isEn ? "Time" : "Uhrzeit")}: ${esc(timeStr)}</div>
    </div>
  </div>

  <div class="grid">
    <div>
      <h2>${isEn ? "Header data" : "Stammdaten"}</h2>
      <table>
        <tr><th>${isEn ? "Requester" : "Anforderer"}</th><td>${esc(banfDoc.requester || "–")}</td></tr>
        <tr><th>${isEn ? "Cost center" : "Kostenstelle"}</th><td>${esc(banfDoc.costCenter || "–")}</td></tr>
        <tr><th id="invBanfThSupplier">${tLang("Lieferant","Supplier")}</th><td>${esc(banfDoc.supplier || "–")}</td></tr>
        <tr><th>${isEn ? "Need-by date" : "Bedarfstermin"}</th><td>${esc(banfDoc.needBy || "–")}</td></tr>
      </table>
    </div>
    <div>
      <h2>${isEn ? "Note" : "Bemerkung"}</h2>
      <div style="border:1px solid #e5e7eb; padding:8px; min-height:32mm;">${esc(banfDoc.note || "")}</div>
    </div>
  </div>

  <h2>${isEn ? "Line items" : "Positionen"}</h2>
  <table>
    <thead>
      <tr>
        <th style="width:16mm;text-align:right;">${isEn ? "No." : "Pos"}</th>
        <th id="invBanfThMaterial">${tLang("Artikelnummer","Item no.")}</th>
        <th id="invBanfThDesc">${tLang("Bezeichnung","Description")}</th>
        <th id="invBanfThSupplier">${tLang("Lieferant","Supplier")}</th>
        <th style="width:18mm;text-align:right;">${isEn ? "Qty" : "Menge"}</th>
        <th style="width:18mm;">${isEn ? "Unit" : "Einheit"}</th>
      </tr>
    </thead>
    <tbody>${rows}</tbody>
  </table>

  <div class="sign">
    <div class="line">${isEn ? "Date" : "Datum"}</div>
    <div class="line">${isEn ? "Signature" : "Unterschrift"}</div>
  </div>

  <script>window.onload = function(){ window.print(); window.close(); };</script>
</body>
</html>`;

    openPrint(html);
  }

  function init() {
    ensureTopButton();
    applyBanfI18n();

    // Wenn der Button schon im HTML existiert, aber nicht wired ist:
    const btn = $("invBanfBtn");
    if (btn && !btn.__banfBound) {
      btn.__banfBound = true;
      btn.addEventListener("click", (e) => {
        e.preventDefault();
        e.stopImmediatePropagation();
        openBanfDialog();
      }, { capture: true });
    }
  }

  document.addEventListener("DOMContentLoaded", init);
})();