// ops-lagerverwaltung.js
// Lagerverwaltung: Mehrlager, Buchungen, Inventur, CSV-Import/-Export,
// Spaltenauswahl und "Tabelle löschen"

(() => {
  "use strict";

  // Prevent double-loading (can happen in portals with dynamic navigation)
  if (window.__opsLagerverwaltungLoaded) {
    console.warn("[lager] ops-lagerverwaltung already loaded – skipping duplicate init.");
    return;
  }
  window.__opsLagerverwaltungLoaded = true;


  // --- LocalStorage Keys ------------------------------------------------

  const STORAGE_ARTICLES      = "opsdeck_lager_v1";
  const STORAGE_BOOKINGS      = "opsdeck_lager_bookings_v1";
  const STORAGE_BANF          = "opsdeck_lager_banf_v1";
  const STORAGE_AUDIT         = "opsdeck_lager_audit_v1";
  const STORAGE_INITIAL       = "opsdeck_lager_initial_prompt_v1";
  const STORAGE_VISIBLE_COLS  = "opsdeck_lager_visible_cols_v1";
  const STORAGE_COL_LABELS    = "opsdeck_lager_col_labels_v1"; // Spaltennamen pro Feld
  const STORAGE_SEALS        = "opsdeck_lager_seals_v1"; // Monatsabschlüsse / Versiegelungen


  // --- File Store (IndexedDB) ------------------------------------------
  // Speichert Anlagen (z.B. Lieferscheine) lokal im Browser.
  // Metadaten werden am Artikel / an der Buchung gespeichert, die Datei selbst in IndexedDB.
  const FILES_DB_NAME  = "opsdeck_lager_files_v1";
  const FILES_DB_STORE = "files";
  let __filesDbPromise = null;

  function openFilesDb() {
    if (!("indexedDB" in window)) return Promise.resolve(null);
    if (__filesDbPromise) return __filesDbPromise;

    __filesDbPromise = new Promise((resolve) => {
      try {
        const req = indexedDB.open(FILES_DB_NAME, 1);
        req.onupgradeneeded = () => {
          const db = req.result;
          if (!db.objectStoreNames.contains(FILES_DB_STORE)) {
            db.createObjectStore(FILES_DB_STORE, { keyPath: "id" });
          }
        };
        req.onsuccess = () => resolve(req.result);
        req.onerror = () => {
          console.warn("[lager] IndexedDB nicht verfügbar:", req.error);
          resolve(null);
        };
      } catch (e) {
        console.warn("[lager] IndexedDB Fehler:", e);
        resolve(null);
      }
    });

    return __filesDbPromise;
  }

  async function persistFileToServer(rec) {
    try {
      if (!rec || !rec.id || !rec.blob) return false;

      const fd = new FormData();
      fd.append("id", rec.id);
      fd.append("kind", rec.kind || "");
      fd.append("ts", rec.ts || "");
      fd.append("name", rec.name || "file");
      fd.append("type", rec.type || "");
      fd.append("size", String(rec.size || 0));
      fd.append("file", rec.blob, rec.name || "file");

      const res = await fetch("/api/files/put", {
        method: "POST",
        body: fd,
        credentials: "include"
      });

      if (!res.ok) throw new Error("upload failed: " + res.status);
      return true;
    } catch (e) {
      console.warn("[lager] persistFileToServer failed:", e);
      return false;
    }
  }

   async function storeFileInDb(file, kind) {
    const db = await openFilesDb();
    if (!db || !file) return null;

    const rec = {
      id: "f" + Math.random().toString(36).slice(2),
      ts: nowIso(),
      kind: kind || "",
      name: file.name || "file",
      type: file.type || "",
      size: file.size || 0,
      blob: file
    };

    const saved = await new Promise((resolve, reject) => {
      try {
        const tx = db.transaction(FILES_DB_STORE, "readwrite");
        tx.objectStore(FILES_DB_STORE).put(rec);
        tx.oncomplete = () => resolve(rec);
        tx.onerror = () => reject(tx.error || new Error("tx failed"));
      } catch (e) {
        reject(e);
      }
    });

    // Zusätzlich auf Festplatte speichern (best effort, ohne await -> kein UI-Block & kein Syntax-Risiko)
    persistFileToServer(saved);

    return saved;
  }


  async function getFileFromDb(fileId) {
    const db = await openFilesDb();
    if (!db || !fileId) return null;

    return await new Promise((resolve, reject) => {
      try {
        const tx = db.transaction(FILES_DB_STORE, "readonly");
        const req = tx.objectStore(FILES_DB_STORE).get(String(fileId));
        req.onsuccess = () => resolve(req.result || null);
        req.onerror = () => reject(req.error || new Error("get failed"));
      } catch (e) {
        reject(e);
      }
    });
  }

  async function clearFilesDb() {
    const db = await openFilesDb();
    if (!db) return;

    await new Promise((resolve, reject) => {
      try {
        const tx = db.transaction(FILES_DB_STORE, "readwrite");
        tx.objectStore(FILES_DB_STORE).clear();
        tx.oncomplete = () => resolve(true);
        tx.onerror = () => reject(tx.error || new Error("clear failed"));
      } catch (e) {
        reject(e);
      }
    });
  }

  async function openStoredFile(fileId) {
    try {
      const rec = await getFileFromDb(fileId);
      if (!rec || !rec.blob) {
        uiAlert(
          "Datei nicht gefunden (evtl. Browser-Daten gelöscht).",
          "File not found (browser data may have been cleared).",
          "⚠️ Hinweis",
          "⚠️ Notice",
          "OK",
          "OK"
        );
        return;
      }
      const url = URL.createObjectURL(rec.blob);
      // Öffnen in neuem Tab (User-Click kommt vom Button)
      const a = document.createElement("a");
      a.href = url;
      a.target = "_blank";
      a.rel = "noopener";
      document.body.appendChild(a);
      a.click();
      a.remove();
      setTimeout(() => URL.revokeObjectURL(url), 30000);
    } catch (e) {
      console.warn("[lager] openStoredFile failed:", e);
    }
  }


  // --- kleine Helfer ----------------------------------------------------
  function byId(id) {
    return document.getElementById(id);
  }
  // --- Sprache (DE/EN) --------------------------------------------------
  function opsLang() {
    try {
      if (typeof window.getOpsLang === "function") return window.getOpsLang();
    } catch (e) {}
    const l = document.documentElement.getAttribute("lang") || "de";
    return String(l).toLowerCase().startsWith("en") ? "en" : "de";
  }
  function tLang(deText, enText) {
    return opsLang() === "en" ? enText : deText;
  }

  // Fügt (falls nötig) ein <style> mit CSS ein (nur UI, keine Logik).
  function ensureStyleTag(styleId, cssText) {
    try {
      if (document.getElementById(styleId)) return;
      const st = document.createElement("style");
      st.id = styleId;
      st.textContent = cssText;
      document.head.appendChild(st);
    } catch (e) {
      // optional
    }
  }


  // --- Detail-Dialog: Scrollbars erzwingen (nur UI) ----------------------
  ensureStyleTag("invDetailDialogScrollStyle", `
    #invDetailDialog{
      max-height: 92vh !important;
      overflow: hidden !important;
    }
    #invDetailDialog .inv-detail-view{
      max-height: 88vh !important;
      overflow: auto !important;
      padding-right: 10px;
    }
  `);

  // --- Tabelle: echtes Vollbild/Overlay (nur UI) -------------------------
  let __invTableFs = null;

  // Wenn die Sprache per Header umgeschaltet wird, sollen dynamische UI-Labels
  // (z.B. Vollbild-Button) sofort korrekt aktualisiert werden.
  function __updateInvTableSizeToggleLabel() {
    try {
      const btn = byId("invTableSizeToggleBtn");
      if (!btn) return;
      btn.textContent = __invTableFs
        ? tLang("🗗 Normal", "🗗 Window")
        : tLang("⛶ Vollbild", "⛶ Fullscreen");
    } catch (e) {}
  }
  // Support both event names used across the app
  window.addEventListener("opsLangChanged", __updateInvTableSizeToggleLabel);
  window.addEventListener("ops:langchange", __updateInvTableSizeToggleLabel);
function ensureTableFullscreenOverlay() {
    let overlay = byId("invTableFullscreenOverlay");
    if (overlay) return overlay;

    overlay = document.createElement("div");
    overlay.id = "invTableFullscreenOverlay";
    overlay.className = "inv-table-fs-overlay";
    overlay.style.display = "none";
    overlay.innerHTML = `
      <div class="inv-table-fs-window" role="dialog" aria-modal="true" tabindex="-1">
        <div class="inv-table-fs-titlebar">
          <div class="inv-table-fs-title">Tabelle</div>
          <div class="inv-table-fs-controls">
            <button type="button" class="inv-table-fs-btn" data-act="min" title="Minimieren">—</button>
            <button type="button" class="inv-table-fs-btn" data-act="max" title="Maximieren">□</button>
            <button type="button" class="inv-table-fs-btn inv-table-fs-close" data-act="close" title="Schließen">×</button>
          </div>
        </div>
        <div class="inv-table-fs-body" id="invTableFullscreenBody"></div>
      </div>
    `;
    document.body.appendChild(overlay);

    ensureStyleTag("invTableFullscreenStyle", `
      .inv-table-fs-overlay{
        position: fixed;
        inset: 0;
        z-index: 99999;
        display: none;
        align-items: center;
        justify-content: center;
        background: rgba(0,0,0,0.45);
        backdrop-filter: blur(6px);
      }
      .inv-table-fs-window{
        position: fixed;
        inset: 14px;
        border-radius: 16px;
        overflow: hidden;
        background: rgba(2,6,23,0.92);
        border: 1px solid rgba(255,255,255,0.10);
        box-shadow: 0 16px 60px rgba(0,0,0,0.55);
        display: flex;
        flex-direction: column;
      }
      .inv-table-fs-window.is-max{
        inset: 0;
        border-radius: 0;
      }
      .inv-table-fs-titlebar{
        height: 44px;
        display: flex;
        align-items: center;
        justify-content: space-between;
        padding: 0 10px 0 14px;
        background: rgba(255,255,255,0.04);
        border-bottom: 1px solid rgba(255,255,255,0.08);
        user-select: none;
      }
      .inv-table-fs-title{
        font-weight: 600;
        opacity: .9;
        letter-spacing: .2px;
      }
      .inv-table-fs-controls{
        display: flex;
        gap: 6px;
        align-items: center;
      }
      .inv-table-fs-btn{
        width: 38px;
        height: 30px;
        border-radius: 10px;
        border: 1px solid rgba(255,255,255,0.10);
        background: rgba(255,255,255,0.04);
        color: inherit;
        cursor: pointer;
      }
      .inv-table-fs-btn:hover{
        background: rgba(255,255,255,0.08);
        border-color: rgba(255,255,255,0.16);
      }
      .inv-table-fs-close:hover{
        background: rgba(248,113,113,0.16);
        border-color: rgba(248,113,113,0.25);
      }
      .inv-table-fs-body{
        flex: 1;
        min-height: 0;
        padding: 10px;
      }
  /* die vorhandene table-wrap ODER scrollFixWrapper im Vollbild */
.table-wrap.inv-table-in-fs,
#scrollFixWrapper.inv-table-in-fs{
  height: 100% !important;
  max-height: none !important;
  width: 100% !important;
}

    `);

    // Klick auf dunklen Hintergrund schließt
    overlay.addEventListener("click", (e) => {
      if (e.target === overlay) closeTableFullscreen();
    });

    // Buttons
    overlay.querySelectorAll("[data-act]").forEach((b) => {
      b.addEventListener("click", () => {
        const act = b.getAttribute("data-act");
        if (act === "max") {
          const win = overlay.querySelector(".inv-table-fs-window");
          if (win) win.classList.toggle("is-max");
          return;
        }
        closeTableFullscreen();
      });
    });

    return overlay;
  }

  function openTableFullscreen(tableWrap, toggleBtn) {
    if (!tableWrap) return false;
    if (__invTableFs) return true;

    const overlay = ensureTableFullscreenOverlay();
    if (!overlay) return false;

    const body = byId("invTableFullscreenBody") || overlay.querySelector(".inv-table-fs-body");
    if (!body) return false;

    __invTableFs = {
      tableWrap,
      parent: tableWrap.parentNode,
      next: tableWrap.nextSibling,
      toggleBtn,
      prevBodyOverflow: document.body.style.overflow,
      onKey: null,
    };

    // Titel evtl. anpassen (optional)
    try {
      const titleEl = overlay.querySelector(".inv-table-fs-title");
      if (titleEl) titleEl.textContent = tLang("Tabelle", "Table");
    } catch (e) {}

    // Control-Button Titles in aktueller Sprache
    try {
      const bMin = overlay.querySelector('[data-act="min"]');
      const bMax = overlay.querySelector('[data-act="max"]');
      const bClose = overlay.querySelector('[data-act="close"]');
      if (bMin) bMin.setAttribute("title", tLang("Minimieren", "Minimize"));
      if (bMax) bMax.setAttribute("title", tLang("Maximieren", "Maximize"));
      if (bClose) bClose.setAttribute("title", tLang("Schließen", "Close"));
    } catch (e) {}

    tableWrap.classList.add("inv-table-in-fs");
    body.appendChild(tableWrap);

    overlay.style.display = "flex";
    try { document.body.style.overflow = "hidden"; } catch (e) {}

    if (toggleBtn) toggleBtn.textContent = tLang("🗗 Normal", "🗗 Window");

    __invTableFs.onKey = (ev) => {
      if (ev.key === "Escape") closeTableFullscreen();
    };
    document.addEventListener("keydown", __invTableFs.onKey);

    try {
      const win = overlay.querySelector(".inv-table-fs-window");
      if (win) win.focus();
    } catch (e) {}

    return true;
  }

  function closeTableFullscreen() {
    if (!__invTableFs) return;
    const { tableWrap, parent, next, toggleBtn, prevBodyOverflow, onKey } = __invTableFs;

    try { document.removeEventListener("keydown", onKey); } catch (e) {}

    const overlay = byId("invTableFullscreenOverlay");
    if (overlay) overlay.style.display = "none";

    try { document.body.style.overflow = prevBodyOverflow || ""; } catch (e) {}

    try { tableWrap.classList.remove("inv-table-in-fs"); } catch (e) {}

    try {
      if (parent) {
        if (next && next.parentNode === parent) parent.insertBefore(tableWrap, next);
        else parent.appendChild(tableWrap);
      }
    } catch (e) {
      // ignore
    }

    if (toggleBtn) toggleBtn.textContent = tLang("⛶ Vollbild", "⛶ Fullscreen");
    __invTableFs = null;
  }


  // "Tabelle groß" Toggle (nur Höhe ändern – kein echtes Overlay, damit nichts kaputt geht)
  function ensureTableSizeToggleButton(rootEl, printBtn) {
    let btn = byId("invTableSizeToggleBtn");
    if (btn) return btn;
    if (!rootEl) return null;

    // Anker: möglichst neben "Tabelle drucken"
    const anchor = printBtn || byId("invPrintTableBtn") || byId("invNewArticleBtn") || null;
    const wrap = anchor && anchor.parentElement ? anchor.parentElement : rootEl;

    btn = document.createElement("button");
    btn.id = "invTableSizeToggleBtn";
    btn.className = "btn btn-ghost";
    btn.type = "button";
    btn.textContent = tLang("⛶ Vollbild", "⛶ Fullscreen");

    // Einfügen direkt nach Print-Button (falls vorhanden), sonst ans Ende
    try {
      if (anchor && anchor.parentElement) {
        if (anchor.nextSibling) anchor.parentElement.insertBefore(btn, anchor.nextSibling);
        else anchor.parentElement.appendChild(btn);
      } else {
        wrap.appendChild(btn);
      }
    } catch (e) {
      // wenn Einfügen nicht klappt, einfach ignorieren
    }

    // CSS für den "groß"-Modus
ensureStyleTag("invTableLargeStyle", `
  .table-wrap.inv-table-large,
  #scrollFixWrapper.inv-table-large{
    height: 72vh !important;
    max-height: 72vh !important;
  }
  @media (max-height: 820px){
    .table-wrap.inv-table-large,
    #scrollFixWrapper.inv-table-large{
      height: 65vh !important;
      max-height: 65vh !important;
    }
  }
`);


    // Toggle-Logik
    btn.addEventListener("click", () => {
  const tableWrap =
  rootEl.querySelector(".table-wrap") ||
  rootEl.querySelector("#scrollFixWrapper") ||
  document.getElementById("scrollFixWrapper");

      if (!tableWrap) return;

      // bevorzugt: echtes Overlay/Vollbild (wie ein Fenster)
      if (__invTableFs && __invTableFs.tableWrap === tableWrap) {
        closeTableFullscreen();
        return;
      }
      const ok = openTableFullscreen(tableWrap, btn);

      // Fallback: falls Overlay nicht möglich ist, nur Höhe umschalten
      if (!ok) {
        const on = tableWrap.classList.toggle("inv-table-large");
        btn.textContent = on ? tLang("🗗 Normal", "🗗 Window") : tLang("⛶ Vollbild", "⛶ Fullscreen");
        try { requestAnimationFrame(() => tableWrap.scrollTop = tableWrap.scrollTop); } catch(e){}
      }
    });

    return btn;
  }


  // Insertiert einen Filter für gesperrte Artikel (Zugang/BANF), falls im HTML nicht vorhanden.
  function ensureBlockedFilterEl() {
    let el = byId("invFilterBlocked");
    if (el) return el;
    const toolbar = document.querySelector(".inv-toolbar");
    if (!toolbar) return null;
    const anchor = byId("invFilterBestand") || byId("invFilterStatus");
    const anchorWrap = anchor && anchor.closest ? anchor.closest(".inv-filter") : null;
    const wrap = document.createElement("div");
    wrap.className = "inv-filter";
    wrap.innerHTML =
      '<label for="invFilterBlocked">🔒 Sperre</label>' +
      '<div class="select-wrapper">' +
      '<select id="invFilterBlocked">' +
        '<option value="">Alle</option>' +
        '<option value="blocked">Nur gesperrt</option>' +
        '<option value="unblocked">Nur aktiv</option>' +
      '</select>' +
      '</div>';
    if (anchorWrap && anchorWrap.parentNode) {
      anchorWrap.parentNode.insertBefore(wrap, anchorWrap.nextSibling);
    } else {
      toolbar.appendChild(wrap);
    }
    return byId("invFilterBlocked");
  }


  // Moderner Pflicht-Eingabedialog (statt window.prompt) – wird nur für Sperren/Entsperren genutzt.
  function ensureReasonDialog() {
    let dlg = byId("invReasonDialog");
    if (dlg) return dlg;
    try {
      dlg = document.createElement("dialog");
      dlg.id = "invReasonDialog";
      dlg.className = "dialog";
      dlg.innerHTML =
        '<div class="card" style="max-width:520px;width:92vw;">' +
          '<h3 id="invReasonTitle">Grund</h3>' +
          '<div id="invReasonIntro" style="margin-top:6px;opacity:.85;white-space:pre-line;"></div>' +
          '<div style="margin-top:10px;">' +
            '<label for="invReasonText">Grund (Pflicht)</label>' +
            '<textarea id="invReasonText" rows="3" style="width:100%;"></textarea>' +
            '<div id="invReasonError" style="display:none;margin-top:6px;font-size:.95em;color:#f87171;"></div>' +
          '</div>' +
          '<div style="display:flex;gap:8px;justify-content:flex-end;margin-top:14px;">' +
            '<button type="button" id="invReasonCancel" class="btn btn-ghost">Abbrechen</button>' +
            '<button type="button" id="invReasonOk" class="btn btn-primary">OK</button>' +
          '</div>' +
        '</div>';
      document.body.appendChild(dlg);
      return dlg;
    } catch (e) {
      console.warn("[lager] Reason-Dialog konnte nicht erstellt werden:", e);
      return null;
    }
  }

  function openReasonDialogRequired(opts) {
  const dlg = ensureReasonDialog();
  if (!dlg) return false;

  const titleEl = byId("invReasonTitle");
  const introEl = byId("invReasonIntro");
  const textEl  = byId("invReasonText");
  const errEl   = byId("invReasonError");
  const okBtn   = byId("invReasonOk");
  const cancelBtn = byId("invReasonCancel");
  const labelEl = dlg.querySelector('label[for="invReasonText"]');

  if (titleEl) titleEl.textContent = opts.title || "Grund";
  if (introEl) introEl.textContent = opts.intro || "";
  if (labelEl) labelEl.textContent = opts.labelText || "Grund (Pflicht)";
  if (textEl) {
    textEl.value = opts.defaultValue || "";
    textEl.placeholder = opts.placeholder || "";
  }
  if (errEl) { errEl.style.display = "none"; errEl.textContent = ""; }
  if (okBtn) okBtn.textContent = opts.okText || "OK";
  if (cancelBtn) cancelBtn.textContent = opts.cancelText || "Abbrechen";

  const showErr = (msg) => {
    if (!errEl) return;
    errEl.textContent = String(msg || "");
    errEl.style.display = "block";
  };

  const cleanup = () => {
    if (okBtn) okBtn.onclick = null;
    if (cancelBtn) cancelBtn.onclick = null;
    dlg.oncancel = null;
  };

  const onCancel = () => {
    cleanup();
    closeDialog(dlg);
    if (typeof opts.onCancel === "function") opts.onCancel();
  };

  if (cancelBtn) cancelBtn.onclick = onCancel;
  dlg.oncancel = onCancel;

  if (okBtn) okBtn.onclick = () => {
    const val = (textEl ? String(textEl.value || "") : "").trim();

    if (errEl) { errEl.style.display = "none"; errEl.textContent = ""; }

    if (!val) {
      showErr(opts.emptyErrorText || "Bitte einen Grund eingeben.");
      if (textEl) textEl.focus();
      return;
    }

    if (typeof opts.validate === "function") {
      const res = opts.validate(val);
      if (res === false) {
        showErr(opts.validateErrorText || "Ungültige Eingabe.");
        if (textEl) textEl.focus();
        return;
      }
      if (typeof res === "string") {
        showErr(res);
        if (textEl) textEl.focus();
        return;
      }
      if (res && typeof res === "object" && res.ok === false) {
        showErr(res.error || opts.validateErrorText || "Ungültige Eingabe.");
        if (textEl) textEl.focus();
        return;
      }
    }

    cleanup();
    closeDialog(dlg);
    if (typeof opts.onOk === "function") opts.onOk(val);
  };

  openDialog(dlg);
  if (textEl) setTimeout(() => { try { textEl.focus(); } catch (e) {} }, 0);
  return true;
}

  // --- Delete Meta Dialog (Name + Grund) -------------------------------
  function ensureDeleteMetaDialog() {
    let dlg = byId("invDeleteMetaDialog");
    if (dlg) return dlg;
    try {
      dlg = document.createElement("dialog");
      dlg.id = "invDeleteMetaDialog";
      dlg.className = "dialog";
      dlg.innerHTML =
        '<div class="card" style="max-width:560px;width:92vw;">' +
          '<h3 id="invDeleteMetaTitle">⚠️ Endgültig löschen</h3>' +
          '<div id="invDeleteMetaIntro" style="margin-top:6px;opacity:.85;white-space:pre-line;"></div>' +
          '<div style="margin-top:12px;display:grid;gap:10px;">' +
            '<div>' +
              '<label for="invDeleteMetaName">' + tLangSafe("Name / Kürzel (Pflicht)", "Name / initials (required)") + '</label>' +
              '<input id="invDeleteMetaName" type="text" style="width:100%;" autocomplete="name" />' +
            '</div>' +
            '<div>' +
              '<label for="invDeleteMetaReason">' + tLangSafe("Grund (Pflicht)", "Reason (required)") + '</label>' +
              '<textarea id="invDeleteMetaReason" rows="3" style="width:100%;"></textarea>' +
            '</div>' +
            '<div id="invDeleteMetaError" style="display:none;margin-top:0;font-size:.95em;color:#f87171;"></div>' +
          '</div>' +
          '<div class="row" style="justify-content:flex-end;gap:10px;margin-top:14px;">' +
            '<button id="invDeleteMetaCancel" class="btn btn-ghost">' + tLangSafe("Abbrechen", "Cancel") + '</button>' +
            '<button id="invDeleteMetaOk" class="btn btn-primary">' + tLangSafe("OK", "OK") + '</button>' +
          '</div>' +
        '</div>';
      document.body.appendChild(dlg);
      return dlg;
    } catch (e) {
      console.warn("[lager] Delete-Meta-Dialog konnte nicht erstellt werden:", e);
      return null;
    }
  }

  function openDeleteMetaDialogRequired(opts) {
    const dlg = ensureDeleteMetaDialog();
    if (!dlg) return Promise.resolve(null);

    const titleEl  = byId("invDeleteMetaTitle");
    const introEl  = byId("invDeleteMetaIntro");
    const nameEl   = byId("invDeleteMetaName");
    const reasonEl = byId("invDeleteMetaReason");
    const errEl    = byId("invDeleteMetaError");
    const okBtn    = byId("invDeleteMetaOk");
    const cancelBtn= byId("invDeleteMetaCancel");
// Labels & Buttons bei jedem Öffnen aktualisieren (wegen Language-Switch)
const nameLabel   = dlg.querySelector('label[for="invDeleteMetaName"]');
const reasonLabel = dlg.querySelector('label[for="invDeleteMetaReason"]');

if (nameLabel)   nameLabel.textContent   = tLangSafe("Name / Kürzel (Pflicht)", "Name / initials (required)");
if (reasonLabel) reasonLabel.textContent = tLangSafe("Grund (Pflicht)", "Reason (required)");

if (cancelBtn) cancelBtn.textContent = tLangSafe("Abbrechen", "Cancel");
if (okBtn)     okBtn.textContent     = tLangSafe("OK", "OK");

    if (titleEl) titleEl.textContent = opts.title || tLangSafe("⚠️ Endgültig löschen", "⚠️ Delete permanently");
    if (introEl) introEl.textContent = opts.intro || "";

    if (nameEl) {
      nameEl.value = opts.defaultName || "";
      nameEl.placeholder = opts.namePlaceholder || tLangSafe("z.B. MM / Max Mustermann", "e.g. JD / Jane Doe");
    }
    if (reasonEl) {
      reasonEl.value = opts.defaultReason || "";
      reasonEl.placeholder = opts.reasonPlaceholder || tLangSafe("Warum wird die Tabelle gelöscht?", "Why is the table being deleted?");
    }

    const showErr = (msg) => {
      if (!errEl) return;
      errEl.textContent = String(msg || "");
      errEl.style.display = "block";
    };
    const clearErr = () => {
      if (!errEl) return;
      errEl.textContent = "";
      errEl.style.display = "none";
    };

    return new Promise((resolve) => {
      const cleanup = () => {
        if (okBtn) okBtn.onclick = null;
        if (cancelBtn) cancelBtn.onclick = null;
        dlg.oncancel = null;
      };

      const finish = (val) => {
        cleanup();
        closeDialog(dlg);
        resolve(val);
      };

      dlg.oncancel = (ev) => {
        try { ev.preventDefault(); } catch (_) {}
        finish(null);
      };

      if (cancelBtn) cancelBtn.onclick = () => finish(null);

      if (okBtn) okBtn.onclick = () => {
        clearErr();
        const name = String(nameEl ? nameEl.value : "").trim();
        const reason = String(reasonEl ? reasonEl.value : "").trim();

        if (!name) {
          showErr(opts.emptyNameErrorText || tLangSafe("Bitte Name/Kürzel eingeben.", "Please enter a name/initials."));
          try { nameEl && nameEl.focus(); } catch (_) {}
          return;
        }
        if (!reason) {
          showErr(opts.emptyReasonErrorText || tLangSafe("Bitte einen Grund eingeben.", "Please enter a reason."));
          try { reasonEl && reasonEl.focus(); } catch (_) {}
          return;
        }

        finish({ name, reason });
      };

      openDialog(dlg);
      setTimeout(() => { try { (nameEl || reasonEl)?.focus(); } catch (e) {} }, 0);
    });
  }

  function nowIso() {
    return new Date().toISOString();
  }

function toNumber(value, fallback = 0) {
  if (value === null || value === undefined) return fallback;

  // Schon eine Zahl?
  if (typeof value === "number") return Number.isFinite(value) ? value : fallback;

  let s = String(value).trim();
  if (s === "") return fallback;

  // Entferne Leerzeichen (z.B. "1 234,56")
  s = s.replace(/\s/g, "");

  // Entferne Währungssymbole/Text (z.B. "61,25€", "EUR", "$14.10")
  // Erlaubt bleiben Ziffern, Minus, Punkt, Komma.
  s = s.replace(/[^0-9,\.\-]/g, "");

  if (s === "" || s === "-" || s === "," || s === ".") return fallback;

  // Deutsches Format -> "1.234,56" => "1234.56"
  // Wenn sowohl Punkt als auch Komma vorkommen, ist Punkt i.d.R. Tausendertrenner.
  if (s.includes(",") && s.includes(".")) {
    s = s.replace(/\./g, "").replace(",", ".");
  } else if (s.includes(",")) {
    s = s.replace(",", ".");
  }

  const n = Number(s);
  return Number.isFinite(n) ? n : fallback;
}



// Entfernt ggf. ein BOM am Anfang eines Strings
// - echtes Unicode-BOM: U+FEFF
// - häufige Fehlvariante: "ï»¿" (UTF-8-BOM als Latin-1 interpretiert)
function stripBom(str) {
  if (typeof str !== "string") return "";
  return str.replace(/^(?:\uFEFF|\u00EF\u00BB\u00BF)/, "");
}


  function escapeHtml(value) {
    if (value === null || value === undefined) return "";
    return String(value)
      .replace(/&/g, "&amp;")
      .replace(/</g, "&lt;")
      .replace(/>/g, "&gt;")
      .replace(/"/g, "&quot;")
      .replace(/'/g, "&#039;");
  }

  function pad2(n) {
    return String(n).padStart(2, "0");
  }

  function formatDateTime(iso) {
    if (!iso) return "";
    try {
      const d = new Date(iso);
      return d.toLocaleString("de-DE");
    } catch {
      return iso;
    }
  }

  function openDialog(dlg) {
  if (!dlg) return;

  // Prevent overlays (e.g., fullscreen table overlay) from eating clicks while a dialog is open
  try {
    const fs = document.getElementById("invTableFullscreenOverlay");
    if (fs) {
      if (fs.dataset.prevPointerEvents === undefined) {
        fs.dataset.prevPointerEvents = fs.style.pointerEvents || "";
      }
      fs.style.pointerEvents = "none";
    }
  } catch (_) {}

  // Close any other open dialogs to avoid stacked backdrops capturing clicks
  try {
    document.querySelectorAll("dialog[open]").forEach((d) => {
      if (d !== dlg) {
        try { d.close(); } catch (_) {}
        try { d.removeAttribute("open"); } catch (_) {}
      }
    });
  } catch (_) {}

  // Bring to front & ensure it can receive clicks
  try { dlg.style.pointerEvents = "auto"; } catch (_) {}
  try { dlg.style.zIndex = "2147483647"; } catch (_) {}

  // IMPORTANT: If this dialog is already open (sometimes left open non-modally),
  // close & reopen modally to ensure it's in the top-layer and fully clickable.
  try {
    if (typeof dlg.showModal === "function") {
      // If already open (e.g. previously opened non-modally), we may need to close & reopen.
      // Mark this as an internal reopen so confirm() handlers don't treat the intermediate "close"
      // as a user cancel.
      if (dlg.open) {
        // Mark this as an internal reopen so confirm() handlers don't treat the intermediate "close"
        // as a user cancel. Keep the marker alive a bit longer to avoid timing races where the
        // 'close' event fires after a 0ms timeout.
        try { dlg.__ops_internalReopen = true; } catch (_) {}
        try { dlg.__ops_internalReopenUntil = Date.now() + 600; } catch (_) {}
        try { dlg.close(); } catch (_) {}
        try { dlg.removeAttribute("open"); } catch (_) {}
      }
      dlg.showModal();

      // Clear marker after a short grace period (covers delayed 'close' events)
      if (dlg.__ops_internalReopen) {
        setTimeout(() => {
          try { delete dlg.__ops_internalReopen; } catch (_) {}
          try { delete dlg.__ops_internalReopenUntil; } catch (_) {}
        }, 650);
      }
    } else {
      dlg.setAttribute("open", "open");
    }
  } catch (e) {
    try { delete dlg.__ops_internalReopen; } catch (_) {}

    // Fallback: show as non-modal
    try { dlg.setAttribute("open", "open"); } catch (_) {}
    console.warn("[lager] Dialog open failed:", e);
  }
}

function closeDialog(dlg) {
  if (!dlg) return;

  try { dlg.close(); } catch (_) {}
  try { dlg.removeAttribute("open"); } catch (_) {}

  // Restore overlay clickability once no dialogs are open
  try {
    if (!document.querySelector("dialog[open]")) {
      const fs = document.getElementById("invTableFullscreenOverlay");
      if (fs) {
        const prev = fs.dataset.prevPointerEvents;
        if (prev !== undefined) {
          fs.style.pointerEvents = prev;
          delete fs.dataset.prevPointerEvents;
        } else {
          fs.style.pointerEvents = "";
        }
      }
    }
  } catch (_) {}
}

  // --- Modal Info/Alert (ersetzt window.alert, i18n-fähig) ------------
  function tLangSafe(de, en) {
    try {
      if (typeof window.tLang === "function") return window.tLang(de, en);
    } catch (_) {}
    return de;
  }

  function ensureInfoDialog() {
    let dlg = byId("invInfoDialog");
    if (dlg) return dlg;

    dlg = document.createElement("dialog");
    dlg.id = "invInfoDialog";
    dlg.className = "dialog";
    dlg.innerHTML = `
      <div class="card" style="max-width:560px">
        <h3 id="invInfoTitle" style="margin:0 0 10px 0;font-size:18px;"></h3>
        <div id="invInfoMsg" style="white-space:pre-wrap;line-height:1.45;"></div>
        <div style="display:flex;justify-content:flex-end;gap:8px;margin-top:14px;">
          <button type="button" class="btn btn-primary" id="invInfoOk">OK</button>
        </div>
      </div>
    `;
    document.body.appendChild(dlg);

    dlg.addEventListener("click", (e) => {
      if (e.target === dlg) closeDialog(dlg);
    });

    return dlg;
  }

  function showInfo(message, title, okText) {
    const dlg = ensureInfoDialog();
    const titleEl = dlg.querySelector("#invInfoTitle");
    const msgEl = dlg.querySelector("#invInfoMsg");
    const okBtn = dlg.querySelector("#invInfoOk");

    if (titleEl) titleEl.textContent = String(title || "");
    if (msgEl) msgEl.textContent = String(message || "");
    if (okBtn) okBtn.textContent = String(okText || "OK");

    const onOk = (ev) => {
      ev.preventDefault();
      try { okBtn && okBtn.removeEventListener("click", onOk); } catch (_) {}
      closeDialog(dlg);
    };
    if (okBtn) okBtn.addEventListener("click", onOk);

    openDialog(dlg);
  }

  function uiAlert(deMsg, enMsg, titleDe, titleEn, okDe, okEn) {
    showInfo(
      tLangSafe(deMsg, enMsg),
      tLangSafe(titleDe || "ℹ️ Hinweis", titleEn || "ℹ️ Info"),
      tLangSafe(okDe || "OK", okEn || "OK")
    );
  }

  // --- Modern Confirm Dialog (ersetzt window.confirm) --------------------
  function ensureConfirmDialog() {
    let dlg = byId("invConfirmDialog");

    if (!dlg) {
      dlg = document.createElement("dialog");
      dlg.id = "invConfirmDialog";
      dlg.className = "dialog";
      dlg.innerHTML = `
        <div class="card" style="max-width:560px;width:92vw;">
          <h3 id="invConfirmTitle">⚠️ Bestätigen</h3>
          <p id="invConfirmMessage" style="white-space:pre-line;margin-top:8px;"></p>
          <div style="margin-top:14px;display:flex;justify-content:flex-end;gap:8px;align-items:center;">
            <button type="button" class="btn" id="invConfirmCancel">Abbrechen</button>
            <button type="button" class="btn btn-primary" id="invConfirmOk">OK</button>
          </div>
        </div>
      `;
      document.body.appendChild(dlg);
    }

    // One-time wiring: prevents "dead buttons" and avoids races between sequential confirms
    if (!dlg.__invConfirmWired) {
      dlg.__invConfirmWired = true;

      const okBtn = dlg.querySelector("#invConfirmOk");
      const cancelBtn = dlg.querySelector("#invConfirmCancel");

      // last user choice; resolved on `close` so sequential confirms can't overlap close-events
      dlg.__invConfirmResult = null;

      const onOk = (e) => {
        e && e.preventDefault && e.preventDefault();
        dlg.__invConfirmResult = true;
        closeDialog(dlg);
      };

      const onCancel = (e) => {
        e && e.preventDefault && e.preventDefault();
        dlg.__invConfirmResult = false;
        closeDialog(dlg);
      };

      if (okBtn) okBtn.addEventListener("click", onOk);
      if (cancelBtn) cancelBtn.addEventListener("click", onCancel);

      // ESC key
      dlg.addEventListener("cancel", (e) => {
        e && e.preventDefault && e.preventDefault();
        onCancel(e);
      });

      // Resolve on close (ignore internal reopen cycles)
      dlg.addEventListener("close", () => {
        const until = dlg.__ops_internalReopenUntil || 0;
        if (dlg.__ops_internalReopen || (until && Date.now() < until)) return;

        const active = dlg.__invConfirmActive;
        dlg.__invConfirmActive = null;

        const res = !!dlg.__invConfirmResult;
        dlg.__invConfirmResult = null;

        if (active && typeof active.resolve === "function") {
          try { active.resolve(res); } catch (_) {}
        }
      });

      // Click on the backdrop cancels
      dlg.addEventListener("click", (e) => {
        if (e && e.target === dlg) onCancel(e);
      });
    }

    return dlg;
  }

  function showConfirm(msg, title, okText, cancelText) {
    const dlg = ensureConfirmDialog();
    const titleEl = dlg.querySelector("#invConfirmTitle");
    const msgEl = dlg.querySelector("#invConfirmMessage");
    const okBtn = dlg.querySelector("#invConfirmOk");
    const cancelBtn = dlg.querySelector("#invConfirmCancel");

    if (titleEl) titleEl.textContent = title || tLangSafe("⚠️ Bestätigen", "⚠️ Confirm");
    if (msgEl) msgEl.textContent = msg || "";
    if (okBtn) okBtn.textContent = okText || "OK";
    if (cancelBtn) cancelBtn.textContent = cancelText || tLangSafe("Abbrechen", "Cancel");

    // If a previous confirm is still pending for any reason, cancel it to avoid orphaned states.
    if (dlg.__invConfirmActive && typeof dlg.__invConfirmActive.resolve === "function") {
      try { dlg.__invConfirmActive.resolve(false); } catch (_) {}
      dlg.__invConfirmActive = null;
    }

    return new Promise((resolve) => {
      dlg.__invConfirmResult = null;
      dlg.__invConfirmActive = { resolve };
      openDialog(dlg);
    });
  }

  function uiConfirm(deMsg, enMsg, titleDe, titleEn, okDe, okEn, cancelDe, cancelEn) {
    return showConfirm(
      tLangSafe(deMsg, enMsg),
      tLangSafe(titleDe || "⚠️ Bestätigen", titleEn || "⚠️ Confirm"),
      tLangSafe(okDe || "OK", okEn || "OK"),
      tLangSafe(cancelDe || "Abbrechen", cancelEn || "Cancel")
    );
  }

// Bestätigung per Eingabe (z.B. "LÖSCHEN" / "DELETE")
function uiTypeToConfirm(deIntro, enIntro, titleDe, titleEn, requiredDe, requiredEn, okDe, okEn, cancelDe, cancelEn) {
  const required = tLangSafe(requiredDe || "LÖSCHEN", requiredEn || "DELETE");
  return new Promise((resolve) => {
    const opened = openReasonDialogRequired({
      title: tLangSafe(titleDe || "⚠️ Endgültig löschen", titleEn || "⚠️ Delete permanently"),
      intro:
        tLangSafe(deIntro || "", enIntro || "") +
        "\n\n" +
        tLangSafe(
          'Zur Bestätigung bitte „' + required + '“ eingeben.',
          'To confirm, type “' + required + '”.'
        ),
      labelText: tLangSafe("Bestätigung (Pflicht)", "Confirmation (required)"),
      placeholder: required,
      okText: tLangSafe(okDe || "Endgültig löschen", okEn || "Delete permanently"),
      cancelText: tLangSafe(cancelDe || "Abbrechen", cancelEn || "Cancel"),
      emptyErrorText: tLangSafe("Bitte die Bestätigung eingeben.", "Please enter the confirmation."),
      validate: (val) => {
        const got = String(val || "").trim().toUpperCase();
        const must = String(required || "").trim().toUpperCase();
        return got === must
          || tLangSafe('Bitte genau „' + required + '“ eingeben.', 'Please type exactly “' + required + '”.');
      },
      onOk: () => resolve(true),
      onCancel: () => resolve(false)
    });
    if (!opened) resolve(false);
  });
}






  // --- State & Spaltenkonfiguration -------------------------------------

const state = {
  articles: [],
  bookings: [],
  banf: [],
  audit: [],
  seals: [], // Monatsabschlüsse / Versiegelungen
  filters: {
    search: "",
    lager: "",
    kategorie: "",
    status: "", // "" | unterMin | gleichMin | ueberMin | null | positiv | low | zero | ok
    blocked: "" // "" | blocked | unblocked
  },
  selectedIds: new Set(),
  visibleCols: null,   // sichtbare Spalten
  columnLabels: null   // Mapping: feld -> Spaltenüberschrift
};

const COLUMN_CONFIG = [
  { key: "artikelnummer",          label: "Artikelnummer",          mandatory: false },
  { key: "interneArtikelnummer",   label: "Interne Artikelnr.",     mandatory: false },
  { key: "bezeichnung",            label: "Bezeichnung",            mandatory: false },
  { key: "hersteller",             label: "Hersteller",             mandatory: false },
  { key: "lieferant",              label: "Lieferant",              mandatory: false },
  { key: "kategorie",              label: "Kategorie",              mandatory: false },
  { key: "lager",                  label: "Lager",                  mandatory: false },
  { key: "mindestbestand",         label: "Mindestbestand",         mandatory: false },
  { key: "bestand",                label: "Bestand",                mandatory: false },
  { key: "einheit",                label: "Einheit",                mandatory: false },
  { key: "einzelpreis",            label: "Einzelpreis",            mandatory: false },
  { key: "gesamtpreis",            label: "Gesamtpreis",            mandatory: false },
  { key: "notizen",                label: "Notizen",                mandatory: false },
  { key: "technischerPlatz",       label: "TechnischerPlatz",       mandatory: false },
  { key: "equipment",              label: "Equipment",              mandatory: false },
  { key: "anlage",                 label: "Anlage",                 mandatory: false },
  { key: "regal",                  label: "Regal",                  mandatory: false },
  { key: "fach",                   label: "Fach",                   mandatory: false },
  { key: "position",               label: "Position / Fachnummer",  mandatory: false },
  { key: "niveau",                 label: "Niveau",                 mandatory: false },
  { key: "lagerort",               label: "Lagerort",               mandatory: false },
  // "aktionen" absichtlich nicht – ist immer sichtbar
];


  // --- LocalStorage Laden / Speichern -----------------------------------

function loadArticles() {
  try {
    const raw = localStorage.getItem(STORAGE_ARTICLES);
    if (!raw) return [];

    const arr = JSON.parse(raw);
    if (!Array.isArray(arr)) return [];

    return arr.map((a) => {
      const created = a.createdAt || nowIso();

      return {
        id: String(a.id || ("a" + Math.random().toString(36).slice(2))),

        // 🔑 Identifikation
        artikelnummer: a.artikelnummer || "",
        interneArtikelnummer: a.interneArtikelnummer || "", // ✅ FIX

        // 📦 Stammdaten
        bezeichnung: a.bezeichnung || "",
        hersteller: a.hersteller || "",
        lieferant: a.lieferant || "",                       // ✅ FIX
        kategorie: a.kategorie || "",
        lager: a.lager || "",
        lagerart: a.lagerart || "",

        // 📍 Lagerplatz
        regal: a.regal || "",
        fach: a.fach || "",
        position: a.position || "",
        niveau: a.niveau || "",
        lagerplatztyp: a.lagerplatztyp || "",
        sicherheitsstufe: a.sicherheitsstufe || "",
        lagerort: a.lagerort || "",

        // 📊 Bestände
        mindestbestand: toNumber(a.mindestbestand, 0),
        bestand: toNumber(a.bestand, 0),
        einheit: a.einheit || "",

        // 💰 Preise
        wertEinzeln: toNumber(a.wertEinzeln, 0),
        waehrung: a.waehrung || "",

        // 📝 Sonstiges
        notizen: a.notizen || "",
        technischerPlatz: a.technischerPlatz || "",
        equipment: a.equipment || "",
        anlage: a.anlage || "",
        wareneingang: a.wareneingang || "",

        // 📎 Anlagen (z.B. Lieferscheine)
        attachments: Array.isArray(a.attachments) ? a.attachments : [],

// 🔒 Sperrstatus (Zugang/BANF gesperrt; Entnahme erlaubt)
receiptBlocked: !!a.receiptBlocked,
receiptBlockedReason: a.receiptBlockedReason || "",
receiptBlockedAt: a.receiptBlockedAt || "",
receiptBlockedBy: a.receiptBlockedBy || "",

// 🗃️ Archiv (optional – bleibt erhalten, wenn vorhanden)
archived: !!a.archived,
archivedReason: a.archivedReason || "",
archivedAt: a.archivedAt || "",
archivedBy: a.archivedBy || "",

        // 🕒 Meta
        createdAt: created,
        updatedAt: a.updatedAt || created,
        lastBooking: a.lastBooking || null
      };
    });
  } catch (e) {
    console.warn("[lager] Fehler beim Laden der Artikel:", e);
    return [];
  }
}


  function saveArticles(list) {
    try {
      localStorage.setItem(STORAGE_ARTICLES, JSON.stringify(list || []));
    } catch (e) {
      console.warn("[lager] Fehler beim Speichern der Artikel:", e);
    }
  }

  function loadBookings() {
    try {
      const raw = localStorage.getItem(STORAGE_BOOKINGS);
      if (!raw) return [];
      const arr = JSON.parse(raw);
      if (!Array.isArray(arr)) return [];
      return arr.map((b) => {
        const id = (b && b.id != null) ? String(b.id) : ("b" + Math.random().toString(36).slice(2));
        const ts = (b && b.ts) ? String(b.ts) : nowIso();
        return Object.assign({}, b, {
          id: id,
          ts: ts,
          articleId: (b && b.articleId != null) ? String(b.articleId) : "",
          artikelnummer: (b && b.artikelnummer != null) ? String(b.artikelnummer) : "",
          bezeichnung: (b && b.bezeichnung != null) ? String(b.bezeichnung) : "",
          lager: (b && b.lager != null) ? String(b.lager) : "",
          lagerort: (b && b.lagerort != null) ? String(b.lagerort) : "",
          amount: toNumber(b && b.amount, 0),
          bestandVorher: toNumber(b && b.bestandVorher, 0),
          bestandNachher: toNumber(b && b.bestandNachher, 0)
        });
      });
    } catch (e) {
      console.warn("[lager] Fehler beim Laden der Buchungen:", e);
      return [];
    }
  }


  function saveBookings(list) {
    try {
      localStorage.setItem(STORAGE_BOOKINGS, JSON.stringify(list || []));
    } catch (e) {
      console.warn("[lager] Fehler beim Speichern der Buchungen:", e);
    }
  }
  // --- BANF & Audit-Log Laden / Speichern -------------------------------

  function loadBanf() {
    try {
      const raw = localStorage.getItem(STORAGE_BANF);
      if (!raw) return [];
      const arr = JSON.parse(raw);
      return Array.isArray(arr) ? arr : [];
    } catch (e) {
      console.warn("[lager] Fehler beim Laden der BANF:", e);
      return [];
    }
  }

  function saveBanf(list) {
    try {
      localStorage.setItem(STORAGE_BANF, JSON.stringify(list || []));
    } catch (e) {
      console.warn("[lager] Fehler beim Speichern der BANF:", e);
    }
  }

  function loadAudit() {
    try {
      const raw = localStorage.getItem(STORAGE_AUDIT);
      if (!raw) return [];
      const arr = JSON.parse(raw);
      return Array.isArray(arr) ? arr : [];
    } catch (e) {
      console.warn("[lager] Fehler beim Laden des Audit-Logs:", e);
      return [];
    }
  }

  function saveAudit(list) {
    try {
      localStorage.setItem(STORAGE_AUDIT, JSON.stringify(list || []));
    } catch (e) {
      console.warn("[lager] Fehler beim Speichern des Audit-Logs:", e);
    }
  }

  // --- Monatsabschluss / Versiegelung (Seals) ----------------------------

function loadSeals() {
  try {
    const raw = localStorage.getItem(STORAGE_SEALS);
    if (!raw) return [];
    const arr = JSON.parse(raw);
    return Array.isArray(arr) ? arr : [];
  } catch (e) {
    console.warn("[lager] Fehler beim Laden der Seals:", e);
    return [];
  }
}

function saveSeals(list) {
  try {
    localStorage.setItem(STORAGE_SEALS, JSON.stringify(list || []));
  } catch (e) {
    console.warn("[lager] Fehler beim Speichern der Seals:", e);
  }
}

// Stabiles JSON (Keys sortiert) -> reproduzierbare Hashes
function stableStringify(value) {
  const seen = new WeakSet();
  const norm = (v) => {
    if (v === null || v === undefined) return v;
    if (typeof v !== "object") return v;
    if (seen.has(v)) return "[Circular]";
    seen.add(v);
    if (Array.isArray(v)) return v.map(norm);
    const out = {};
    Object.keys(v).sort().forEach((k) => { out[k] = norm(v[k]); });
    return out;
  };
  return JSON.stringify(norm(value));
}

async function sha256Hex(str) {
  try {
    const enc = new TextEncoder();
    const buf = enc.encode(String(str || ""));
    const digest = await crypto.subtle.digest("SHA-256", buf);
    const bytes = Array.from(new Uint8Array(digest));
    return bytes.map(b => b.toString(16).padStart(2, "0")).join("");
  } catch (e) {
    // Fallback: kein "echtes" SHA-256 (nur Notlösung)
    let h = 2166136261;
    const s = String(str || "");
    for (let i = 0; i < s.length; i++) {
      h ^= s.charCodeAt(i);
      h = Math.imul(h, 16777619);
    }
    return ("fnv1a-" + (h >>> 0).toString(16));
  }
}

function ymFromIso(iso) {
  try {
    const d = new Date(iso);
    const y = d.getFullYear();
    const m = String(d.getMonth() + 1).padStart(2, "0");
    return y + "-" + m;
  } catch {
    return "";
  }
}

function monthRange(ym) {
  // ym: "YYYY-MM"
  const m = String(ym || "").trim();
  if (!/^\d{4}-\d{2}$/.test(m)) return null;
  const y = Number(m.slice(0, 4));
  const mo = Number(m.slice(5, 7)) - 1;
  const start = new Date(Date.UTC(y, mo, 1, 0, 0, 0));
  const end   = new Date(Date.UTC(y, mo + 1, 1, 0, 0, 0));
  return { startIso: start.toISOString(), endIso: end.toISOString() };
}

function inRange(iso, startIso, endIso) {
  const t = new Date(iso || 0).getTime();
  const a = new Date(startIso).getTime();
  const b = new Date(endIso).getTime();
  return Number.isFinite(t) && t >= a && t < b;
}

function getLatestSealHash() {
  const seals = Array.isArray(state.seals) ? state.seals : [];
  if (!seals.length) return "";
  const last = seals[seals.length - 1];
  return last && last.hash ? String(last.hash) : "";
}

async function buildSealPayloadForMonth(ym) {
  const r = monthRange(ym);
  if (!r) throw new Error("Ungültiger Monat: " + ym);

  const bookings = (state.bookings || []).filter(b => inRange(b.ts, r.startIso, r.endIso));
  const audit    = (state.audit || []).filter(e => inRange(e.ts, r.startIso, r.endIso));

  // Snapshot der Artikel zum Zeitpunkt des Seal-Klicks (technisch: aktueller Stand)
  // Für "stärker": später serverseitig / export auf WORM-Speicher.
  const articlesSnapshot = (state.articles || []).map(a => ({
    id: a.id,
    artikelnummer: a.artikelnummer || "",
    interneArtikelnummer: a.interneArtikelnummer || "",
    bezeichnung: a.bezeichnung || "",
    lager: a.lager || "",
    lagerort: a.lagerort || "",
    bestand: a.bestand ?? 0,
    einheit: a.einheit || "",
    wertEinzeln: a.wertEinzeln ?? 0,
    waehrung: a.waehrung || "",
    updatedAt: a.updatedAt || a.createdAt || ""
  }));

  const prevHash = getLatestSealHash();

  const payload = {
    schema: "opsdeck.seal.v1",
    period: ym,
    range: { startIso: r.startIso, endIso: r.endIso },
    prevHash: prevHash || "",
    counts: { bookings: bookings.length, audit: audit.length, articles: articlesSnapshot.length },
    bookings: bookings,
    audit: audit,
    articlesSnapshot: articlesSnapshot
  };

  const canonical = stableStringify(payload);
  const hash = await sha256Hex(canonical);

  return { payload, canonical, hash };
}

function downloadText(filename, text, mime) {
  try {
    const blob = new Blob([text], { type: mime || "text/plain;charset=utf-8" });
    const a = document.createElement("a");
    a.href = URL.createObjectURL(blob);
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    a.remove();
    setTimeout(() => URL.revokeObjectURL(a.href), 250);
  } catch (e) {
    console.warn("[lager] Download fehlgeschlagen:", e);
  }
}


// --- Monatsabschluss Excel-Template (XLSX) -------------------------------
// Kein window.XLSX nötig: wir liefern ein fertiges Template als Base64 und laden es als Datei runter.
// Hinweis: Das Template enthält Formeln und Grafiken, aber keine "Excel-Tabellen" (damit Excel nicht reparieren muss).
const MONTHCLOSE_TEMPLATE_XLSX_B64 = `
UEsDBBQAAAAIAAR/m1tGx01IlQAAAM0AAAAQAAAAZG9jUHJvcHMvYXBwLnhtbE3PTQvCMAwG4L9SdreZih6kDkQ9ip68zy51hbYp
bYT67+0EP255ecgboi6JIia2mEXxLuRtMzLHDUDWI/o+y8qhiqHke64x3YGMsRoPpB8eA8OibdeAhTEMOMzit7Dp1C5GZ3XPlkJ3
sjpRJsPiWDQ6sScfq9wcChDneiU+ixNLOZcrBf+LU8sVU57mym/8ZAW/B7oXUEsDBBQAAAAIAAR/m1sOAzjQ7wAAACsCAAARAAAA
ZG9jUHJvcHMvY29yZS54bWzNksFOwzAMhl8F5d66zdQhoq4XECeQkJgE4hY53hataaPEqN3b04atE4IH4Bj7z+fPkmv0CvtAL6H3
FNhSvBld20WFfiMOzF4BRDyQ0zGfEt3U3PXBaZ6eYQ9e41HvCWRRrMERa6NZwwzM/EIUTW1QYSDNfTjjDS54/xnaBDMI1JKjjiOU
eQmimSf609jWcAXMMKbg4neBzEJM1T+xqQPinByjXVLDMOTDKuWmHUp4f356TetmtousO6TpV7SKT5424jL5bXX/sH0UjSxklZUy
k7fbslLVWhV3H7PrD7+rsOuN3dl/bHwRbGr4dRfNF1BLAwQUAAAACAAEf5tbmVycIxAGAACcJwAAEwAAAHhsL3RoZW1lL3RoZW1l
MS54bWztWltz2jgUfu+v0Hhn9m0LxjaBtrQTc2l227SZhO1OH4URWI1seWSRhH+/RzYQy5YN7ZJNups8BCzp+85FR+foOHnz7i5i
6IaIlPJ4YNkv29a7ty/e4FcyJBFBMBmnr/DACqVMXrVaaQDDOH3JExLD3IKLCEt4FMvWXOBbGi8j1uq0291WhGlsoRhHZGB9Xixo
QNBUUVpvXyC05R8z+BXLVI1lowETV0EmuYi08vlsxfza3j5lz+k6HTKBbjAbWCB/zm+n5E5aiOFUwsTAamc/VmvH0dJIgILJfZQF
ukn2o9MVCDINOzqdWM52fPbE7Z+Mytp0NG0a4OPxeDi2y9KLcBwE4FG7nsKd9Gy/pEEJtKNp0GTY9tqukaaqjVNP0/d93+ubaJwK
jVtP02t33dOOicat0HgNvvFPh8Ouicar0HTraSYn/a5rpOkWaEJG4+t6EhW15UDTIABYcHbWzNIDll4p+nWUGtkdu91BXPBY7jmJ
Ef7GxQTWadIZljRGcp2QBQ4AN8TRTFB8r0G2iuDCktJckNbPKbVQGgiayIH1R4Ihxdyv/fWXu8mkM3qdfTrOa5R/aasBp+27m8+T
/HPo5J+nk9dNQs5wvCwJ8fsjW2GHJ247E3I6HGdCfM/29pGlJTLP7/kK6048Zx9WlrBdz8/knoxyI7vd9lh99k9HbiPXqcCzIteU
RiRFn8gtuuQROLVJDTITPwidhphqUBwCpAkxlqGG+LTGrBHgE323vgjI342I96tvmj1XoVhJ2oT4EEYa4pxz5nPRbPsHpUbR9lW8
3KOXWBUBlxjfNKo1LMXWeJXA8a2cPB0TEs2UCwZBhpckJhKpOX5NSBP+K6Xa/pzTQPCULyT6SpGPabMjp3QmzegzGsFGrxt1h2jS
PHr+BfmcNQockRsdAmcbs0YhhGm78B6vJI6arcIRK0I+Yhk2GnK1FoG2camEYFoSxtF4TtK0EfxZrDWTPmDI7M2Rdc7WkQ4Rkl43
Qj5izouQEb8ehjhKmu2icVgE/Z5ew0nB6ILLZv24fobVM2wsjvdH1BdK5A8mpz/pMjQHo5pZCb2EVmqfqoc0PqgeMgoF8bkePuV6
eAo3lsa8UK6CewH/0do3wqv4gsA5fy59z6XvufQ9odK3NyN9Z8HTi1veRm5bxPuuMdrXNC4oY1dyzcjHVK+TKdg5n8Ds/Wg+nvHt
+tkkhK+aWS0jFpBLgbNBJLj8i8rwKsQJ6GRbJQnLVNNlN4oSnkIbbulT9UqV1+WvuSi4PFvk6a+hdD4sz/k8X+e0zQszQ7dyS+q2
lL61JjhK9LHMcE4eyww7ZzySHbZ3oB01+/ZdduQjpTBTl0O4GkK+A226ndw6OJ6YkbkK01KQb8P56cV4GuI52QS5fZhXbefY0dH7
58FRsKPvPJYdx4jyoiHuoYaYz8NDh3l7X5hnlcZQNBRtbKwkLEa3YLjX8SwU4GRgLaAHg69RAvJSVWAxW8YDK5CifEyMRehw55dc
X+PRkuPbpmW1bq8pdxltIlI5wmmYE2eryt5lscFVHc9VW/Kwvmo9tBVOz/5ZrcifDBFOFgsSSGOUF6ZKovMZU77nK0nEVTi/RTO2
EpcYvOPmx3FOU7gSdrYPAjK5uzmpemUxZ6by3y0MCSxbiFkS4k1d7dXnm5yueiJ2+pd3wWDy/XDJRw/lO+df9F1Drn723eP6bpM7
SEycecURAXRFAiOVHAYWFzLkUO6SkAYTAc2UyUTwAoJkphyAmPoLvfIMuSkVzq0+OX9FLIOGTl7SJRIUirAMBSEXcuPv75Nqd4zX
+iyBbYRUMmTVF8pDicE9M3JD2FQl867aJguF2+JUzbsaviZgS8N6bp0tJ//bXtQ9tBc9RvOjmeAes4dzm3q4wkWs/1jWHvky3zlw
2zreA17mEyxDpH7BfYqKgBGrYr66r0/5JZw7tHvxgSCb/NbbpPbd4Ax81KtapWQrET9LB3wfkgZjjFv0NF+PFGKtprGtxtoxDHmA
WPMMoWY434dFmhoz1YusOY0Kb0HVQOU/29QNaPYNNByRBV4xmbY2o+ROCjzc/u8NsMLEjuHti78BUEsDBBQAAAAIAAR/m1stkZg8
zwMAAHcNAAAYAAAAeGwvd29ya3NoZWV0cy9zaGVldDEueG1sjVdrb9s2FP0rrIoNHdBFtuJXPNtA3Dht0RoIamTF9qWgrSuLCEW6
JBWl+fW7lGTZmURF+eCIEs859/BxeTnLpHrQMYAhTwkXeu7Fxhymvq93MSRUX8gDCPwSSZVQg0219/VBAQ1zUML9oNcb+QllwlvM
8nd3ajGTqeFMwJ0iOk0Sqn4tgcts7vW944tvbB8b+8JfzA50Dxsw94c7hS2/YglZAkIzKYiCaO5d96er/sAC8h5/M8j02TOxVrZS
PtjG53Du9TxLLYD82hw4y8WIkYevEJkPwDkSBh6hO8Me4Q67zb2tNEYm9juGaajBV5GSzyByTeCAfTGYQ61zQVKSWo8/y4C9yo8N
6vz5GPltPrA4UFuq4YPk31lo4rk38UgIEU25+SazT1AO1tDy7STX+S/Jir4Bdt6lGqMpwRhBwkTxnz6Vg3wGGPQcgKAEBP8DBC7A
oAQMugKGJWDYFTAqAaOugHEJGHcFTErApCvgqgRc5bNbTEc+lzfU0MVMyYwo2xvZ7EO+IIrFN/eYsPtiYxR+ZYgzi1vg4cw3yGTb
/q5ELdtR30GZBtRNgQocqC8gxDONeQNy1Y6s6/noszIbVGaDnObSQbOWghry7h/8+3O9/qPJt4vg5NEV4xp3rGKUZxgruRYRFfsm
ozbWaLG5X79bAu5yEf4o+r65nd5iRNFi9ojdHxtcXlYuL1tdXm8xe/JU65CaNCE+uY/VM7CmCVu6mE52nWN5bnclQmgye1kza3u+
bnVQWR20Wt1gmkx1kzEX7mRs4OD8N93//jYIJn+JPRDHQl8NKmMrYQSN8aT4cQ5883H6sd3isLI4bJ/NfMuQa2XYA3CyB02Txql0
0ZwcDx0SlQWn3eFrdlfTVbvdUWV31MXuMt3FKRKLFsMuopr2uNIed9GuLDaJuhhqopNKdNJF9Hw0m3RdJDXdq0r3qj0X1tMVyfNj
k7qLqqbe753OnV53fZsV3OpOprr82bHXb5W/YVEECsQzoVstedos7OKoC5+OoH77GXQS/q1Rsu0Aeil5Og/6rxwIxRL7IrXBgtZg
nYj7ij5g0dgYQduZ8DKCU5rut+fpT0xkwFD6qXmcC7QrI98w0KCK9aHp8XQjaRJRrQ2h6Id8xSJebSGDfZ423h8TpjbltsIFlorw
5SBsqywTgi75L5rqDP+swLLXhTVVeyY04Vh/Y6V/McbUqooquWhgoZ9XT0WZnj/GeGsBZTvg90hKc2xgGRcqmjHcgPklaKq6XINk
FLEd3MhdipnKFPcgBZzaO4KO2UFjRFOG9xD1OSwuAtVda/EfUEsDBBQAAAAIAAR/m1vi1W74XAEAAJMHAAAYAAAAeGwvZHJhd2lu
Z3MvZHJhd2luZzEueG1s7ZXfboMgFMZfhfAAwz9rlxgkWWq27GbrKxDESiJgDrR1bz9Qk9XtZmsar+oN3+F4PuDwi9KzqwANujOu
xK33fUGIE63U3D3YXpqQaSxo7kMIB1IDPytz0B3JkmRLXA+S166V0ldTBjNqjdzJrns2orXAaANWMypsx3JK4hD1R9OwZAyjomDP
bEtJHKKes7OiZLKQg0diKPHmaZuEByPxWeJsk40BYfQAvG+VeAGuJaPm9HoR78NGxPtpD0jVJU4xMmGyxLuWg0dprA7ZZUGYI79N
hgZ0yPBiXm1qXcGvaJ7myuALq4p7jo6grrAS8RzBSxSjmjclrnaaDeAvBrZplJCVFUctjZ9cQHbcK2tcq3qHERSx6/BWj50mixNf
xkEvL1F0KnjG12LdD7Cu5CxL1wMtW4KW3UFbCbRsFdDS7EZftOTxG7R8hO6foOVL0PI7aCuBlt8SNBJ/xuwLUEsDBBQAAAAIAAR/
m1sxI5jLmgAAAP8BAAAjAAAAeGwvZHJhd2luZ3MvX3JlbHMvZHJhd2luZzEueG1sLnJlbHPFkUsKg0AQRK8y9AHSfiCLoK6ycRu8
QDO2jsT5MNMBc/sYkoVCFtm5rCp4PKjqxjPJ5F0yU0hqsbNLNRiRcEFM2rCldPKB3boMPlqSNcYRA+k7jYxFlp0xbhnQVFum6p6B
/yH6YZg0X71+WHbyA4zaUBRQHcWRpQZc5k/1XfLTSgTV9jXEts8BjxIpdiLFcSLlTqR8i+Du7+YFUEsDBBQAAAAIAAR/m1t39yKF
jQAAAPMAAAAjAAAAeGwvd29ya3NoZWV0cy9fcmVscy9zaGVldDEueG1sLnJlbHONzzsKwzAQBNCriD2A106RIsiu0rgNvsAirz7E
+iDJxLl93CQ4kCLlzMCDkTdeqLoYinWpiM0vofRga00XxKIseypNTBz2Rcfsqe4xG0yk7mQYT217xnw0YJBHU0zPxP+IUWun+BrV
6jnUHzDOmR4uGBATZcO1B9yWd/lZu2Z3QYxzD3mcO8BB4tfH4QVQSwMEFAAAAAgABH+bWwnqkVP2AQAA3QQAABgAAAB4bC93b3Jr
c2hlZXRzL3NoZWV0Mi54bWyNlO9u2jAQwF/F8gM0NBQ2VUmkQss2qZVQ0bbPJrkQC8fO7GNpefqdHYNQS1m/gO/P7+5850vWG7t1
DQCyl1Zpl/MGsbtNElc20Ap3ZTrQZKmNbQWSaDeJ6yyIKkCtStLRaJq0QmpeZEG3tEVmdqikhqVlbte2wr7OQJk+59f8oHiWmwa9
IimyTmxgBfizW1qSkmOUSragnTSaWahzfnd9uwj+weGXhN6dnJm/ydqYrRd+VDkfcR9ZA3tddUqGXAxN9wg1zkEpipdyJkqUf2FJ
bjlfG0TTejtViQJJVVuzBx1yggLypVq6d85DkBjUX/FPrJcfr+OLOj0fKl+EvlKf1sLB3KjfssIm5185q6AWO4XPpv8OsVcTH680
yoVf1g++pGblzlE1EaYKWqmHf/ESe3wKjD8A0gikb4H0A2AcgfEbIP0IuInAzWczTCIw+SwwjcA09H5oVuj0vUBRZNb0zHpviuYP
YVzh5tRgqf2jXaElqyQOizuLcgtK05sFmyVIIb0hKSM+u4zPYA+ybPROb87A88vwI23FuZz3/8tJb1dX7MloWliha3E2+cPlKA9S
70H1YPEMu7jMfgMnWnzPJtT8ww4M0/CL/yTsRmrHFK0SLe3VFxq5HR78INDOhvEOGxeODX1/wHoHstfG4EHwMz9+0Yp/UEsDBBQA
AAAIAAR/m1v7Q5yB9QEAANkEAAAYAAAAeGwvd29ya3NoZWV0cy9zaGVldDMueG1sjZTbbuIwEIZfxfIDNBQKW1VJpEJLd6VWQkW7
vTbJhFj4kLWHTcvT79gJCLWAegOew/d7PPYkba3b+BoA2btWxme8RmzuksQXNWjhr2wDhiKVdVogmW6d+MaBKCOkVTIcDCaJFtLw
PI2+hctTu0UlDSwc81uthfuYgrJtxq/53vEq1zUGR5KnjVjDEvB3s3BkJQeVUmowXlrDHFQZv7++m8f8mPBHQuuP1iycZGXtJhi/
yowPeFA2wD6WjZJxL4a2eYYKZ6AU6Q05EwXKf7CgtIyvLKLVIU5VokByVc7uwMQ9QQHlUi3Nl+ROpBcNR/zb18sPxwlFHa/3lc9j
X6lPK+FhZtWbLLHO+C1nJVRiq/DVtj+h79U46BVW+fjL2i6X3KzYeqqmh6kCLU33L977Hh8DozPAsAeGn4HhGWDUA6PPwO0Z4KYH
br67w7gHxt8FJj0wib3vmhU7/SBQ5KmzLXMhm9TCIl5XPDk1WJrwaJfoKCqJw/zeodyAMvRmwaUJkmQIJEWPTy/jU9iBLGqzNesT
8Owy/ExTcWrPh8vYEoUp2Ys1NK5gSjih8HhZ4VGaHagWHJ5g55fZJ/BC41c2ocbv3393E2HoX4RbS+OZojGigb36QdftusfeGTSv
8Wq7aYvLmr494EICxStrcW+E+z58zfL/UEsDBBQAAAAIAAR/m1v4QEKXgwIAAIIHAAAYAAAAeGwvd29ya3NoZWV0cy9zaGVldDQu
eG1sjZXrT9swEMD/lSiTpk2TSJvSB6ytxBtGmSoQQ9o3t70mFo6d2RcC/et3dtyKlTTrl8SP+93Ddz4PS6WfTQqAwWsmpBmFKWJ+
HEVmnkLGzIHKQdLOUumMIU11EplcA1s4KBNR3Gr1ooxxGY6Hbm2qx0NVoOASpjowRZYx/XYKQpWjsB2uF+55kqJdiMbDnCXwAPiY
TzXNoo2WBc9AGq5koGE5Ck/axxMn7wR+cSjNu3FgI5kp9WwnN4tR2AqtZgnB20MuuLMVoMonsMQzEIL0xWHA5shfYEpio3CmEFVm
98lLZEhLS61WIJ1NEECy5Ev+QbhS4pXaEP94f8NNONap9+O155fuXOmcZszAmRJPfIHpKByEwQKWrBB4r8pr8GfVtfrmShj3DcpK
tk1hzAtD3niYPMi4rP7s1Z/xPkDsgXgb6O4AOh7obAOdHcChBw73danrge4W0GntAHoe6O1roe+B/r7AwAODLSDeBRx54GjbwuGu
xLXWmWvte7DtTbI/ZHtX8trrdLddvqOqsFxVnjNk46FWZaCtPOmzA1farkqoGLm0F/wBNe1y4nBMUJENIyRVdiGae+y0GXtM9Qo4
1oBnzeCJRv4MQlI/AV2Dnzfjp0Bm56ksZFIDXzTDE+pYdTYvm7F7MohkMPjyu0gY/aPgQqJkaQZfa7RdNWu7A5lADXbdjF1wuQJR
gq4785tm9omoGZSQuCC+kf+fPw36cfy9zv0fzbpulUFq70htsy6K2/8Ez5HpGRVObR4mzfCVLuSCnP+pkK/+xSOq+nWjrq6BfZ3u
mE64NIGgfk8vy0GfLpWuunI1oYfF3azqWXDDlB5J0FaA9pdK4XpiL9vm2R3/BVBLAwQUAAAACAAEf5tbfWstawsCAABiBQAAGAAA
AHhsL3dvcmtzaGVldHMvc2hlZXQ1LnhtbI2UbW/aMBDHv4qVD9BAKLBVSaRCWzapk1BRN2nvTLgkFo6d2Zel7aff2QkVYgTxJvHD
/f4+3/kubrXZ2xIA2VsllU2CErG+C0OblVBxe6NrULSTa1NxpKkpQlsb4DsPVTKMRqNZWHGhgjT2a2uTxrpBKRSsDbNNVXHzvgCp
2yQYB4eFF1GU6BbCNK55ARvA13ptaBZ+quxEBcoKrZiBPAnux3crb+8Nfgpo7dGYuZtstd67yfddEowCp6yAvW9qKfxZDHX9DDku
QUrSiwLGMxR/YU1mSbDViLpy++QlcqSl3OgPUP5MkEC25Ev9n3En0ou6K/7p/Q0+r+OcOh4fPH/ycaU4bbmFpZa/xA7LJPgSsB3k
vJH4ottv0Mdq6vQyLa3/srazpWWWNZa86WHyoBKq+/O3PsbHwGQAiHogOgWiAWDSA5MTIBoCbnvg9hQYDwDTHpieAqMBYNYDs9M7
fB0A5j0w98nqoutT88CRp7HRLTPOmtTcwOfXh4oyIpR75Rs0tCuIw/TeoNiDVPTIwcQhkqTbCLMeX1zGF/ABIitVo4oz8PIy/Exl
dO7Mh8vYCiyvkCqtAPaoUPGygjMqj9eotGDwksjT9a78bgp+Ngqrqx05JxFSQg+F2GXYdZ8f3BRCWSapnqlz3Mzp3Zmu6roJNQ7/
ZLqy98OSmiAYZ0D7udZ4mLh39NlW039QSwMEFAAAAAgABH+bW96t8aj1AQAAfgQAABgAAAB4bC93b3Jrc2hlZXRzL3NoZWV0Ni54
bWyNlG1v2jAQx7+KZWlS96aGQLuqSiK1lD1oq4SKtr02cCFW/ZDZx9L20+/sBMQYtHsDPvt+f9+d75K3zj+GGgDZk9E2FLxGbK6F
CMsajAznrgFLJ5XzRiKZfi1C40GuEmS0yAaDS2GksrzM097Ml7nboFYWZp6FjTHSP9+Cdm3Bh3y78aDWNcYNUeaNXMMc8Hsz82SJ
ncpKGbBBOcs8VAW/GV5Pk39y+KGgDXtrFjNZOPcYjS+rgg94VLbAnueNVukuhq75BhVOQGvSyziTS1S/YUZuBV84RGfiOUWJEmmr
8u4FbLoTNJAvxdL849yJ9KIxxV99vHyXTgxqf72N/GOqK9VpIQNMnP6pVlgX/IqzFVRyo/HBtZ+hr9VF1Fs6HdIvazvf4Ziz5SZQ
ND1MERhlu3/51Nd4H7g6AWQ9kB0CoxPAqAdGh0B2Ahj3wPgAyC5PABc9kFIXXe6pcHcSZZl71zIfvUktLlL1UyJUL2VjD87R06ki
DsuvLiB1FNJLQS6QFOO+WPb07ev0jX2RtWZTi1bW1JlHFCavK3yCIA0Suj52/d3/wC14PMJO3wodQWkmDbt3luY4qrCzd+//VhJU
zm2TdvWNk3kv/VrZwDT1Ok3V+Qd6E991ZGfQUKUH60YiLWv6QICPDnReOYdbI77i7pNT/gFQSwMEFAAAAAgABH+bW+eNOaTmAQAA
aQQAABgAAAB4bC93b3Jrc2hlZXRzL3NoZWV0Ny54bWyFlG1v2jAQx7+KZWlva6DQTVUSidLuQWolVLTttSEXYuGHzD6Wtp9+Zycg
RoG+SXzn+13u/jk7a53fhBoA2YvRNuS8RmxuhQirGowMV64BSzuV80YimX4tQuNBlgkyWowGgxthpLK8yJJv7ovMbVErC3PPwtYY
6V/vQLs250O+czyrdY3RIYqskWtYAP5s5p4ssc9SKgM2KGeZhyrn0+HtQ4pPAb8UtOFgzWInS+c20fhR5nzAY2YL7HXRaJW+xdA1
j1DhDLSmfCPO5ArVX5hTWM6XDtGZuE9VokRyVd69gU3fBA0US7U074K7JH3S2OKfvl6+bycWdbjeVf416Uo6LWWAmdO/VYl1zr9w
VkIltxqfXfsdeq0mMd/K6ZCerO1ih9TGahuomh6mCoyy3Vu+9BofAjdngFEPjI6ByRngugeuj4HxGWDcA+Nj4FwPkx5IrYuu9yTc
vURZZN61zMdoyhYXSf3UCOmlbJzBBXraVcRh8UiD5jOBlCo6xKrH7i5jU/sma82mHtUG9Al+dpn/BkEaXAINlS1P4PeX8ScaRa+k
bsHjCfrho+IRlGaf/icFKbebx07KeAifpF8rG5imsaYDdPWZ5Pfd8HUGnZ/0b7rpT8ua7gLwMYD2K+dwZ8Qftr9din9QSwMEFAAA
AAgABH+bW8DEyTP+AQAA2wQAABgAAAB4bC93b3Jrc2hlZXRzL3NoZWV0OC54bWyNlG1P2zAQx7+K5Ul7N9IGWhBLIkGhYtKQKqqx
1256Saw6dmZfFuDT7+ykVcfajjetz77f3/fgS9IZu3EVALKXWmmX8gqxuY4il1dQC3dmGtB0UhhbCyTTlpFrLIh1gGoVxaPRNKqF
1DxLwt7CZolpUUkNC8tcW9fCvt6CMl3Kx3y78STLCv1GlCWNKGEJ+KNZWLKincpa1qCdNJpZKFJ+M76eB//g8Cyhc3tr5jNZGbPx
xrd1ykfcK2tgr8tGyXAXQ9N8hwJnoBTpxZyJHOVvWJBbylcG0dT+nKJEgbRVWPMGOtwJCsiXYmn+ce5FBlGf4q8hXr5Lxwe1v95G
Pg91pTqthIOZUT/lGquUX3G2hkK0Cp9M9wBDrSZeLzfKhV/W9b5jSiNvHUUzwBRBLXX/L16GGu8B8egIEA9A/A44esP5AJy/By6P
ABcDcPHRGyYDMPkoMB2Aaah9X6xQ6TuBIkus6Zj13qTmF6FdIXMqsNT+0S7R0qkkDrNHowUmEZKU34jyAbs9jd2saHxU69wbSGxa
vTmkMTutsQShvjwIVx1A706jz8ZWYGUJlh0RuP/P3fT6W3eAm5/m7u1Gff4Ux1dfbavLv/mISr+dgL4XfuwfhS2ldkzRINHInl1S
w23/3HuDJjY0t5+3sKzo6wPWO9B5YQxuDd/x3fcs+wNQSwMEFAAAAAgABH+bW29r0z/1AQAA3AYAABQAAAB4bC9jaGFydHMvY2hh
cnQxLnhtbN1VS27bMBC9imoEyK50UrQLQxGg1kbRhREjLZr1WBxJrClSmKET5S69TS9WfmQ7qRsgSIMusiHF+bx5M0+U8qoFcl97
qDAbOm34YtI618+E4KrFDvit7dF4T22pA+eP1AhJcKtM02lxPp1+EBFiUuRxL3KnnEa/DUVOqmqLHGZrK+9WlArM4BklOlBmIgJU
/68oEWRFYZNYX63Iw4q9KS6uWIJDUqBvkVxWmhpMk91wtjASQ7CLKZQS/Zr6FKFlMbbfa+tKQijyNdCnNBn/NFeU3YC+mFRWh4Ya
stteBfRo1Fv2hVEGF6MvoOSQXNNgsiSR7p1DQXZ0hXWR18Xpr59rJPZc3OmbxVkuvFXs3JEcj11q8yJTJHZz4DYRYqtV5C0Cfqgc
i1XgGzfb7m8cT+Yn5zO/vEtUd1Ei5njQxxMXIXFxlBhzRBxcA/21km4kd/Y+zguGL3I0HJ2jQRzE8iTK4TiFK9B+GEELhcaBU9Yk
f6fMEoaIsg+CYWU5uaPcr+FuPPECdPDD0jdVbZZAmzQCYw0Gpn5Sj7kqsszl8EAUvdaXdc3oHko1CuRtf+r0AkJF9p9JSe9GfiXK
Xfuv2f9UL4541EccPogaGzRyt+/nTjF85wzh3xVfGn03wgW3VNx/1GA2XI5Z/p6ntyHdWnH4nxW/AVBLAwQUAAAACAAEf5tb9yvA
FfMBAADyBQAAFAAAAHhsL2NoYXJ0cy9jaGFydDIueG1s3VTNbtswDH6VLAjQ7TKlxbqD4RrIlmDYoWjQFSuwG2PRthZZMkilc99l
x71JX2z6cZL1Dyi2nHaRLH7kx4+k5LxsgNyXDkoc9a02fDZunOsyIbhssAV+azs0HqksteD8kWohCX4oU7danEyn70WkGBd53Ivc
KafRb32RkyqbIodsZeXtklKCDP4iRQvKjEWg6v6VJZIsKWwSq8sleVqxM8XFFd829d0vU+PohkcL4ww0LZrR62sk9yZ4uxhDKdKv
qVARahZD/Z22bkYIRb4C+pha47/mikY3oM/GpdWhoprspvMaB6PesENCGSBGn0DJPkHTYLIkkf44h4Ts6BKrIq+Ko7ufKyT2WtzR
q8VxLrxVbOEojocytTlIG4ndHLhJgthqFXWLwB8yx2Ql+MLNpn1K42Q+eZf55TRJ3XqJGONJnw9cTE4yvzwMjDEiNq6G7lpJN4g7
Po39gv6zHAyPztEg9sPyImb94xAuQftmhFkoNA6csibhrTLn0EeWnRP0S8sJjuNu4bulK1Wuz4HWyW6swQgp8xxUkmWe9feU6pW+
qCpGd1//oNrbHoo/lPpPpKSHkcX/8dbDo37hmz7I9GKLh/mI/V9CY41Gbvdd3ym6b8Hg/lXxhdG3A12ApeLugwaz5tkQ5S9/ug3p
Kov9X774DVBLAwQUAAAACAAEf5tbdgoLB38BAACMAwAAFAAAAHhsL2NoYXJ0cy9jaGFydDMueG1srVPBTsMwDP2VUU0CLnjjsMPU
VSrbEUS1Sdy9xm0j0qRyAtv+jRs/RpJ2bBJCSMAlTmy/Zz9bScsG2W06LGm0b5W2i6RxrpsD2LKhFu2N6Uj7SGW4ReefXINg3Eld
twpuJ5MZRIokS6PNUiedIm/2WcqybLIU51sjDgX3Beb4ixItSp1AoOr+yhJJCg5GULUu2NPCpyseLrvHmhi1I6lodPWAjlii2hG7
65DsIoR7oD97nRAkwyC/U8blTOhvkpb9ZF6RD0ujDNvRK6pFMg2SLHkeKfa9axJchgXx2TvwWsdrqrK0yi5jc+9vW2Lr67rLi9U0
BR+CY05sxA6KlP6XibF1K7RN35U1SoqkH5zSoXIsVqIXqV/abxsd5+PbuT+ms77hYy5EpKf+Ab4K8NVXeERCnKS43yrr1TdmVxCX
pN3ZrGGIVtLL2ShZUq7rs0HDaVdwWqCimrQ42sIM2+OIOAZD+pO0j1odzgoKabs7hfrZ5gOqxi7ihs8Cp/+XfQBQSwMEFAAAAAgA
BH+bWxvHw5TAAgAAkAwAAA0AAAB4bC9zdHlsZXMueG1s3VfbjpswEP0VxAeUELQoVCFSGylSpbZaafehr04wiSVfqDFb0q+vxwaS
bDxRttq+lCiLPWfOzPF4DJtla46cPh0oNVEvuGzL+GBM8zFJ2t2BCtJ+UA2VFqmVFsTYqd4nbaMpqVogCZ7MZ7M8EYTJeLWUndgI
00Y71UlTxrM4WS1rJU+WeewN1pUIGr0QXsZrwtlWM+dLBONHb56DYae40pGxUmgZp2Bpf3s49TNQOcQRTCoNxsRn8H+3g/sZ4G6t
dWCcXyqzhtWyIcZQLTd24jjOeAVFw/j52Fhpe02O6fwhPhHczSbZKl1RfZHGm1ZLTmtjCZrtD3A3qkkANEYJO6gY2StJnIaRcc6M
3NaVsTm40vtK6f3Wln22dpdTA65DjjsZztfJuZNgPUfddzK889nChoGt145y/gRBftRT0VIbqq8j311fKmisCPZwHNpKD0Mfxk8g
0Xk0H/ss7MNfhY0a9qLM586uQLr5z04Z+qhpzXo37+spPxY9RaKn/zT6+2jHKvM+2m9EJ03Dj58420tBfVfcnXC1JCMv+qVJ80x7
Mz4T+nrok6lFXMNcNN9kjeCxVcbf4WnIT/Gjbce4YXKYHVhVUXnVgza8IVv7uL2Ib/0rWpOOm+cJLOPT+ButWCeKyesR1jx4ncZf
4dCm+fTMtLmYrGhPq/UwtadwfX4c/QWE18jGXWEE43gsjACG5cEUYBzPwvL8T+tZoOvxGKZtEUQWKGeBcjwrhKzdB8sT5hT2Cq+0
KLIsz7GK+lfGlYI1Vrc8h284GqYNGFgeyPS2WuO7jXfI7T7A9vRWh2ArxTsRWylea0DCdQNGUYR3G8sDDGwXsN6B/OE80FNhTpaN
/4iEtGEnGEeKAkOgF8M9mudIdXL4hPcHOyVZVhRhBLCwgizDEDiNOIIpAA0YkmXuPfjqfZSM76nk9Btk9QdQSwMEFAAAAAgABH+b
W5eKuxzAAAAAEwIAAAsAAABfcmVscy8ucmVsc52SuW7DMAxAf8XQnjAH0CGIM2XxFgT5AVaiD9gSBYpFnb+v2qVxkAsZeT08Etwe
aUDtOKS2i6kY/RBSaVrVuAFItiWPac6RQq7ULB41h9JARNtjQ7BaLD5ALhlmt71kFqdzpFeIXNedpT3bL09Bb4CvOkxxQmlISzMO
8M3SfzL38ww1ReVKI5VbGnjT5f524EnRoSJYFppFydOiHaV/Hcf2kNPpr2MitHpb6PlxaFQKjtxjJYxxYrT+NYLJD+x+AFBLAwQU
AAAACAAEf5tbgI5/FMIBAAA0BgAADwAAAHhsL3dvcmtib29rLnhtbL2UwW7UQAyGXyWaByDb7e62rJpKhRZYUUFFUQ9cVpOMk1id
eFYzThf6LBz7GNz2xXASRU0FiriEQzTxb8vz2dL8Z3vn71Pn7qPvlaWQqJJ5t47jkJVQ6fDK7YAkkztfaZbQF3HYedAmlABc2Xg+
m63iSiOp87O+142Ph4FjyBgdidgIdwj78JxvwugBA6ZokX8kqv23oKIKCSt8BJOomYpC6fYfnMdHR6ztbeadtYk66hJ34BmzP+Tb
BvKrTkOrsE6/aAFJ1GomDXP0gduKtr8WxgeQ4i6q2b1Dy+AvNcN77+odUtG0kSniwRjtHvqzW+La/8saXZ5jBpcuqysg7vbowTaA
FErcBRWRriBRh58p+IBZyc1QcsvGdAOykA3W5dcoCb8xLeN0PG9AbiazvaBcUzFgmo8wzf8P0xUZGBAdjxAdT020h6KmAmjAsxjh
WUzLc0VMupSi7be6ODwJ2IBrOcK1nJbrowsMJJ+1L1a1GkFaTYt0rQvwh19/eXUnI1An00Jd1AZ5uyGGwiMfnoZYpyNYp61h9S5l
IEcC80laBtHFMbMbHzVHayrzxfLotThjbe1b0T7TtdOmN73esM9/A1BLAwQUAAAACAAEf5tbnyaZaNcAAADwBQAAGgAAAHhsL19y
ZWxzL3dvcmtib29rLnhtbC5yZWxzxdTNDoIwDAfwVyF7AIugiAY8efFqfIEFy0cEtqw16tuLesAaD17ITkvb7N/fqdkBW82N6alu
LAW3ru0pVzWz3QBQUWOnaWYs9sOkNK7TPJSuAquLs64QojBMwH1mqG32mRkc7xb/STRl2RS4M8Wlw55/BMPVuDPViKyCo3YVcq7g
1o5tgtcznw3JKtifcuX2p7kC36BIgCL/oFiAYv+ghQAt/IOWArT0D0oEKPEPWgnQyj8oFaB0QhDxvUUaNe9arF9PuJ6Hvzhuf5Xv
5tflC58IEAd++wBQSwMEFAAAAAgABH+bW0uVraZhAQAAcwkAABMAAABbQ29udGVudF9UeXBlc10ueG1szZbLTsMwEEV/Jcq2SlwK
FITaboAtdMEPGGfSWI0f8rivv2ectJWoSiCkUrOJY3vmnjseyfLkY2cBo60qNU7jwnv7xBiKAhTH1FjQtJMbp7inqVswy8WSL4CN
hsMxE0Z70D7xQSOeTV4g56vSR69bWkZp9DR2UGIcPdeBgTWNubWlFNzTPlvr7ISS7AkpZVYxWEiLAwqI2VlC2PkZsM97X4NzMoNo
zp1/44qi2LZk6HclYNosccajyXMpIDNipSglReuAZ1gAeFWmteigmezphKH+3nTmVzJNQIqcO2OROuagPe7QkpCdWBIC52VziUci
SXeuD0K3M8j+yKbj3Ri3rPqBrBq6n/H3Hh/1f/GROb6ReoGHn+4+9kIt6x9dqf5TH7c98XHXEx/3PfEx7omPh574eLySD1HQFOvh
YncF8SvBFuzu98X/2d3viHbs0JtPY5aXfgaEMVVc6gOfVW+t2RdQSwECFAMUAAAACAAEf5tbRsdNSJUAAADNAAAAEAAAAAAAAAAA
AAAAgAEAAAAAZG9jUHJvcHMvYXBwLnhtbFBLAQIUAxQAAAAIAAR/m1sOAzjQ7wAAACsCAAARAAAAAAAAAAAAAACAAcMAAABkb2NQ
cm9wcy9jb3JlLnhtbFBLAQIUAxQAAAAIAAR/m1uZXJwjEAYAAJwnAAATAAAAAAAAAAAAAACAAeEBAAB4bC90aGVtZS90aGVtZTEu
eG1sUEsBAhQDFAAAAAgABH+bWy2RmDzPAwAAdw0AABgAAAAAAAAAAAAAAICBIggAAHhsL3dvcmtzaGVldHMvc2hlZXQxLnhtbFBL
AQIUAxQAAAAIAAR/m1vi1W74XAEAAJMHAAAYAAAAAAAAAAAAAACAAScMAAB4bC9kcmF3aW5ncy9kcmF3aW5nMS54bWxQSwECFAMU
AAAACAAEf5tbMSOYy5oAAAD/AQAAIwAAAAAAAAAAAAAAgAG5DQAAeGwvZHJhd2luZ3MvX3JlbHMvZHJhd2luZzEueG1sLnJlbHNQ
SwECFAMUAAAACAAEf5tbd/cihY0AAADzAAAAIwAAAAAAAAAAAAAAgAGUDgAAeGwvd29ya3NoZWV0cy9fcmVscy9zaGVldDEueG1s
LnJlbHNQSwECFAMUAAAACAAEf5tbCeqRU/YBAADdBAAAGAAAAAAAAAAAAAAAgIFiDwAAeGwvd29ya3NoZWV0cy9zaGVldDIueG1s
UEsBAhQDFAAAAAgABH+bW/tDnIH1AQAA2QQAABgAAAAAAAAAAAAAAICBjhEAAHhsL3dvcmtzaGVldHMvc2hlZXQzLnhtbFBLAQIU
AxQAAAAIAAR/m1v4QEKXgwIAAIIHAAAYAAAAAAAAAAAAAACAgbkTAAB4bC93b3Jrc2hlZXRzL3NoZWV0NC54bWxQSwECFAMUAAAA
CAAEf5tbfWstawsCAABiBQAAGAAAAAAAAAAAAAAAgIFyFgAAeGwvd29ya3NoZWV0cy9zaGVldDUueG1sUEsBAhQDFAAAAAgABH+b
W96t8aj1AQAAfgQAABgAAAAAAAAAAAAAAICBsxgAAHhsL3dvcmtzaGVldHMvc2hlZXQ2LnhtbFBLAQIUAxQAAAAIAAR/m1vnjTmk
5gEAAGkEAAAYAAAAAAAAAAAAAACAgd4aAAB4bC93b3Jrc2hlZXRzL3NoZWV0Ny54bWxQSwECFAMUAAAACAAEf5tbwMTJM/4BAADb
BAAAGAAAAAAAAAAAAAAAgIH6HAAAeGwvd29ya3NoZWV0cy9zaGVldDgueG1sUEsBAhQDFAAAAAgABH+bW29r0z/1AQAA3AYAABQA
AAAAAAAAAAAAAIABLh8AAHhsL2NoYXJ0cy9jaGFydDEueG1sUEsBAhQDFAAAAAgABH+bW/crwBXzAQAA8gUAABQAAAAAAAAAAAAA
AIABVSEAAHhsL2NoYXJ0cy9jaGFydDIueG1sUEsBAhQDFAAAAAgABH+bW3YKCwd/AQAAjAMAABQAAAAAAAAAAAAAAIABeiMAAHhs
L2NoYXJ0cy9jaGFydDMueG1sUEsBAhQDFAAAAAgABH+bWxvHw5TAAgAAkAwAAA0AAAAAAAAAAAAAAIABKyUAAHhsL3N0eWxlcy54
bWxQSwECFAMUAAAACAAEf5tbl4q7HMAAAAATAgAACwAAAAAAAAAAAAAAgAEWKAAAX3JlbHMvLnJlbHNQSwECFAMUAAAACAAEf5tb
gI5/FMIBAAA0BgAADwAAAAAAAAAAAAAAgAH/KAAAeGwvd29ya2Jvb2sueG1sUEsBAhQDFAAAAAgABH+bW58mmWjXAAAA8AUAABoA
AAAAAAAAAAAAAIAB7ioAAHhsL19yZWxzL3dvcmtib29rLnhtbC5yZWxzUEsBAhQDFAAAAAgABH+bW0uVraZhAQAAcwkAABMAAAAA
AAAAAAAAAIAB/SsAAFtDb250ZW50X1R5cGVzXS54bWxQSwUGAAAAABYAFgDWBQAAjy0AAAAA
`.trim();

function downloadBase64File(filename, b64text, mime) {
  try {
    const base64Clean = String(b64text || "").replace(/\s+/g, "");
    const binary = atob(base64Clean);
    const len = binary.length;
    const bytes = new Uint8Array(len);
    for (let i = 0; i < len; i++) bytes[i] = binary.charCodeAt(i);
    const blob = new Blob([bytes], { type: mime || "application/octet-stream" });
    const a = document.createElement("a");
    a.href = URL.createObjectURL(blob);
    a.download = filename;
    document.body.appendChild(a);
    a.click();
    a.remove();
    setTimeout(() => URL.revokeObjectURL(a.href), 500);
  } catch (e) {
    console.warn("[lager] Base64-Download fehlgeschlagen:", e);
        uiAlert(
      "Download fehlgeschlagen: " + (e && e.message ? e.message : e),
      "Download failed: " + (e && e.message ? e.message : e),
      "\u274c Fehler",
      "\u274c Error",
      "OK",
      "OK"
    );
  }
}


function base64ToBytes(b64text) {
  const base64Clean = String(b64text || "").replace(/\s+/g, "");
  const binary = atob(base64Clean);
  const len = binary.length;
  const bytes = new Uint8Array(len);
  for (let i = 0; i < len; i++) bytes[i] = binary.charCodeAt(i);
  return bytes;
}

function utf8ToBytes(str) {
  return new TextEncoder().encode(String(str == null ? "" : str));
}

// Minimaler ZIP-Builder (STORE, ohne Kompression) – damit Diagramm-Export als *eine* Datei klappt.
const __crcTable = (() => {
  const table = new Uint32Array(256);
  for (let i = 0; i < 256; i++) {
    let c = i;
    for (let k = 0; k < 8; k++) c = (c & 1) ? (0xEDB88320 ^ (c >>> 1)) : (c >>> 1);
    table[i] = c >>> 0;
  }
  return table;
})();

function crc32(buf) {
  let crc = 0xFFFFFFFF;
  for (let i = 0; i < buf.length; i++) {
    crc = __crcTable[(crc ^ buf[i]) & 0xFF] ^ (crc >>> 8);
  }
  return (crc ^ 0xFFFFFFFF) >>> 0;
}

function zipStore(files) {
  // files: [{name: string, data: Uint8Array}]
  const locals = [];
  const centrals = [];
  let offset = 0;
  let centralSize = 0;

  for (const f of (files || [])) {
    const nameBytes = utf8ToBytes(f.name);
    const data = f.data || new Uint8Array(0);
    const crc = crc32(data);
    const size = data.length;

    // Local file header (30 bytes)
    const lh = new Uint8Array(30 + nameBytes.length);
    const dv = new DataView(lh.buffer);
    dv.setUint32(0, 0x04034b50, true);     // signature
    dv.setUint16(4, 20, true);             // version needed
    dv.setUint16(6, 0, true);              // flags
    dv.setUint16(8, 0, true);              // compression 0 = store
    dv.setUint16(10, 0, true);             // mod time
    dv.setUint16(12, 0, true);             // mod date
    dv.setUint32(14, crc, true);           // crc32
    dv.setUint32(18, size, true);          // compressed size
    dv.setUint32(22, size, true);          // uncompressed size
    dv.setUint16(26, nameBytes.length, true);
    dv.setUint16(28, 0, true);             // extra len
    lh.set(nameBytes, 30);

    locals.push(lh, data);

    // Central directory header (46 bytes)
    const ch = new Uint8Array(46 + nameBytes.length);
    const dc = new DataView(ch.buffer);
    dc.setUint32(0, 0x02014b50, true);     // signature
    dc.setUint16(4, 20, true);             // version made by
    dc.setUint16(6, 20, true);             // version needed
    dc.setUint16(8, 0, true);              // flags
    dc.setUint16(10, 0, true);             // compression
    dc.setUint16(12, 0, true);             // mod time
    dc.setUint16(14, 0, true);             // mod date
    dc.setUint32(16, crc, true);
    dc.setUint32(20, size, true);
    dc.setUint32(24, size, true);
    dc.setUint16(28, nameBytes.length, true);
    dc.setUint16(30, 0, true);             // extra
    dc.setUint16(32, 0, true);             // comment
    dc.setUint16(34, 0, true);             // disk start
    dc.setUint16(36, 0, true);             // int attrs
    dc.setUint32(38, 0, true);             // ext attrs
    dc.setUint32(42, offset, true);        // local header offset
    ch.set(nameBytes, 46);

    centrals.push(ch);
    centralSize += ch.length;

    offset += lh.length + data.length;
  }

  const eocd = new Uint8Array(22);
  const de = new DataView(eocd.buffer);
  de.setUint32(0, 0x06054b50, true);       // signature
  de.setUint16(4, 0, true);                // disk
  de.setUint16(6, 0, true);                // disk start
  de.setUint16(8, (files || []).length, true);
  de.setUint16(10, (files || []).length, true);
  de.setUint32(12, centralSize, true);
  de.setUint32(16, offset, true);
  de.setUint16(20, 0, true);               // comment len

  const blob = new Blob([...locals, ...centrals, eocd], { type: "application/zip" });
  return blob;
}

function downloadBlob(filename, blob) {
  const a = document.createElement("a");
  a.href = URL.createObjectURL(blob);
  a.download = filename;
  document.body.appendChild(a);
  a.click();
  a.remove();
  setTimeout(() => URL.revokeObjectURL(a.href), 500);
}
function downloadMonthCloseExcelTemplate() {
  const now = new Date();
  const y = now.getFullYear();
  const m = String(now.getMonth() + 1).padStart(2, "0");
  const fn = `monatsabschluss-template-${y}-${m}.xlsx`;
  downloadBase64File(fn, MONTHCLOSE_TEMPLATE_XLSX_B64, "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
}


// --- Monatsabschluss (Reporting Pack ohne XLSX-Lib) ---------------------
// Exportiert:
//  - Excel-Template (mit Tabellen + Grafiken)
//  - CSV-Dateien für die 8-Sheet-Struktur (Übersicht wird im Template über Formeln berechnet)

function fmtNumDe(n, decimals) {
  const x = Number(n);
  if (!Number.isFinite(x)) return "";
  const d = (decimals == null ? 0 : Number(decimals));
  const s = d > 0 ? x.toFixed(d) : String(Math.round(x));
  return s.replace(".", ",");
}

function pad2Local(n) {
  return String(n).padStart(2, "0");
}

function splitLocalDateTime(iso) {
  const dt = new Date(iso || 0);
  if (!Number.isFinite(dt.getTime())) return { date: "", time: "" };
  const y = dt.getFullYear();
  const m = pad2Local(dt.getMonth() + 1);
  const d = pad2Local(dt.getDate());
  const hh = pad2Local(dt.getHours());
  const mm = pad2Local(dt.getMinutes());
  const ss = pad2Local(dt.getSeconds());
  return { date: `${y}-${m}-${d}`, time: `${hh}:${mm}:${ss}` };
}

function csvVal(v) {
  if (v === null || v === undefined) return "";
  let s = String(v);
  if (s.indexOf('"') !== -1) s = s.replace(/"/g, '""');
  if (s.indexOf(";") !== -1 || s.indexOf("\n") !== -1 || s.indexOf("\r") !== -1) s = '"' + s + '"';
  return s;
}

function toCsv(headers, rows) {
  const lines = [headers.map(csvVal).join(";")];
  (rows || []).forEach((r) => {
    lines.push((r || []).map(csvVal).join(";"));
  });
  return "\uFEFF" + lines.join("\n"); // BOM für Excel/Umlaute
}

async function exportMonthCloseReportingPackFlow(mode) {
  const _m = String(mode || "").toLowerCase();
  const forceTemplate = (_m === "template" || _m === "charts" || _m === "diagramme" || _m === "chart");

  const ym = prompt("📦 Monatsabschluss – Monat (YYYY-MM):", monthDefaultPrevMonth());
  if (!ym) return;
  const range = monthRange(ym);
  if (!range) {
        uiAlert(
      "Ungültiger Monat: " + ym,
      "Invalid month: " + ym,
      "\ud83d\udce6 Monatsabschluss",
      "\ud83d\udce6 Month close",
      "OK",
      "OK"
    );
    return;
  }

  const bookingsM = (state.bookings || []).filter(b => inRange(b.ts, range.startIso, range.endIso));
  const auditM    = (state.audit || []).filter(e => inRange(e.ts, range.startIso, range.endIso));

  // Audit-Map: bookingId -> payload
  const auditByBookingId = {};
  auditM.forEach((e) => {
    if (!e || e.type !== "booking.created" || !e.payload) return;
    const bid = e.payload.bookingId;
    if (bid) auditByBookingId[String(bid)] = e.payload;
  });

  // Seal / Integrität
  const sealRec = (state.seals || []).find(s => String(s.period || "") === String(ym)) || null;
  let sealHash = "";
  let prevHash = "";
  let integrityStatus = "OK";
  let integrityExplanation = "Der Hash dient der Integritätsprüfung. Jede nachträgliche Änderung führt zu einer Abweichung.";
  let sealCreatedAt = nowIso();

  let built = null;
  if (sealRec) {
    try {
      built = await buildSealPayloadForMonth(ym);
      sealHash = built.hash || "";
      prevHash = built.payload && built.payload.prevHash ? String(built.payload.prevHash) : (sealRec.prevHash || "");
      if (sealRec.hash && sealHash && String(sealRec.hash) !== String(sealHash)) {
        integrityStatus = "Abweichung";
      }
      sealCreatedAt = sealRec.createdAt || sealCreatedAt;
    } catch (e) {
      // Fallback: Hash über Monatsdaten
      const minimal = {
        schema: "opsdeck.monthclose.v1",
        period: ym,
        range,
        counts: { bookings: bookingsM.length, audit: auditM.length, articles: (state.articles || []).length },
        bookings: bookingsM,
        audit: auditM
      };
      sealHash = await sha256Hex(stableStringify(minimal));
      prevHash = "";
      integrityStatus = "OK";
      integrityExplanation = "Kein gültiger Seal rekonstruierbar. Hash basiert auf den exportierten Monatsdaten.";
    }
  } else {
    const minimal = {
      schema: "opsdeck.monthclose.v1",
      period: ym,
      range,
      counts: { bookings: bookingsM.length, audit: auditM.length, articles: (state.articles || []).length },
      bookings: bookingsM,
      audit: auditM
    };
    sealHash = await sha256Hex(stableStringify(minimal));
    prevHash = "";
    integrityStatus = "OK";
    integrityExplanation = "Kein Seal gespeichert. Hash basiert auf den exportierten Monatsdaten.";
  }

  // Bestand Ende: bevorzugt Seal-Snapshot (wenn vorhanden)
  const endSnapshot = (built && built.payload && Array.isArray(built.payload.articlesSnapshot))
    ? built.payload.articlesSnapshot
    : (state.articles || []).map(a => ({
        id: a.id,
        artikelnummer: a.artikelnummer || "",
        bezeichnung: a.bezeichnung || "",
        lager: a.lager || "",
        bestand: a.bestand ?? 0,
        wertEinzeln: a.wertEinzeln ?? 0
      }));

  // Meta: Bewegungs-/Artikel-Mapping
  const metaByKey = {};
  bookingsM.forEach((b) => {
    const k = String(b.artikelnummer || "") + "||" + String(b.lager || "");
    if (!metaByKey[k]) metaByKey[k] = { artikelnummer: b.artikelnummer || "", bezeichnung: b.bezeichnung || "", lager: b.lager || "" };
  });
  endSnapshot.forEach((a) => {
    const k = String(a.artikelnummer || "") + "||" + String(a.lager || "");
    if (!metaByKey[k]) metaByKey[k] = { artikelnummer: a.artikelnummer || "", bezeichnung: a.bezeichnung || "", lager: a.lager || "" };
  });

  // Bewegungen (Detail)
  const bewegungenRows = [];
  const aggByKey = {}; // artikelnummer||lager
  const ccAgg = {};    // kostenstelle

  bookingsM.forEach((b) => {
    const dt = splitLocalDateTime(b.ts);
    const dir = (b.direction === "in") ? "Zugang" : "Entnahme";
    const aPayload = auditByBookingId[String(b.id)] || {};
    const unitPriceRaw = (dir === "Zugang") ? (aPayload.unitPrice != null ? aPayload.unitPrice : "") : (aPayload.avgPriceAfter != null ? aPayload.avgPriceAfter : "");
    let unit = toNumber(unitPriceRaw, NaN);
    if (!Number.isFinite(unit)) {
      // Fallback: Preis aus End-Snapshot
      const k = String(b.artikelnummer || "") + "||" + String(b.lager || "");
      const snap = endSnapshot.find(x => String(x.artikelnummer || "") === String(b.artikelnummer || "") && String(x.lager || "") === String(b.lager || ""));
      unit = snap ? toNumber(snap.wertEinzeln, 0) : 0;
    }
    const qty = toNumber(b.amount, 0);
    const valueMove = qty * unit * (dir === "Zugang" ? 1 : -1);

    const grund = String(b.reason || "");
    const note = String(b.note || "");
    const grundNotiz = note ? (grund ? (grund + " / " + note) : note) : grund;

    bewegungenRows.push([
      dt.date,
      dt.time,
      b.artikelnummer || "",
      b.bezeichnung || "",
      b.lager || "",
      dir,
      fmtNumDe(qty, 0),
      fmtNumDe(unit, 2),
      fmtNumDe(valueMove, 2),
      b.kostenstelle || "",
      b.mitarbeiter || "",
      grundNotiz || ""
    ]);

    const key = String(b.artikelnummer || "") + "||" + String(b.lager || "");
    if (!aggByKey[key]) aggByKey[key] = { inQty: 0, outQty: 0, inVal: 0, outValAbs: 0 };
    if (dir === "Zugang") {
      aggByKey[key].inQty += qty;
      aggByKey[key].inVal += Math.max(0, valueMove);
    } else {
      aggByKey[key].outQty += qty;
      aggByKey[key].outValAbs += Math.abs(valueMove);

      const cc = String(b.kostenstelle || "").trim();
      if (cc) {
        if (!ccAgg[cc]) ccAgg[cc] = { cnt: 0, qty: 0, val: 0 };
        ccAgg[cc].cnt += 1;
        ccAgg[cc].qty += qty;
        ccAgg[cc].val += Math.abs(valueMove);
      }
    }
  });

  // Bestand Ende (CSV)
  const endRows = [];
  const endByKey = {};
  endSnapshot.forEach((a) => {
    const key = String(a.artikelnummer || "") + "||" + String(a.lager || "");
    const qtyEnd = toNumber(a.bestand, 0);
    const unit = toNumber(a.wertEinzeln, 0);
    const total = qtyEnd * unit;
    endByKey[key] = { qtyEnd, unit, total };
    endRows.push([
      a.artikelnummer || "",
      a.bezeichnung || "",
      a.lager || "",
      fmtNumDe(qtyEnd, 0),
      fmtNumDe(unit, 2),
      fmtNumDe(total, 2)
    ]);
  });

  // Bestand Anfang (aus Ende - Bewegungen)
  const allKeys = Object.keys(Object.assign({}, metaByKey, endByKey, aggByKey));
  const startRows = [];
  let totalMaterialStart = 0;
  allKeys.forEach((key) => {
    const meta = metaByKey[key] || { artikelnummer: "", bezeichnung: "", lager: "" };
    const end = endByKey[key] || { qtyEnd: 0, unit: 0, total: 0 };
    const agg = aggByKey[key] || { inQty: 0, outQty: 0, inVal: 0, outValAbs: 0 };
    const qtyStart = (end.qtyEnd - agg.inQty + agg.outQty);
    const unit = end.unit;
    const total = qtyStart * unit;
    totalMaterialStart += total;
    startRows.push([
      meta.artikelnummer || "",
      meta.bezeichnung || "",
      meta.lager || "",
      fmtNumDe(qtyStart, 0),
      fmtNumDe(unit, 2),
      fmtNumDe(total, 2)
    ]);
  });

  // Entnahmen_Zugänge (Verdichtung)
  const entzugRows = [];
  allKeys.forEach((key) => {
    const meta = metaByKey[key] || { artikelnummer: "", bezeichnung: "", lager: "" };
    const agg = aggByKey[key] || { inQty: 0, outQty: 0, inVal: 0, outValAbs: 0 };
    if (!meta.artikelnummer) return;
    entzugRows.push([
      meta.artikelnummer || "",
      meta.bezeichnung || "",
      meta.lager || "",
      fmtNumDe(agg.outQty, 0),
      fmtNumDe(agg.outValAbs, 2),
      fmtNumDe(agg.inQty, 0),
      fmtNumDe(agg.inVal, 2)
    ]);
  });

  // Kostenstellen
  const totalCcVal = Object.values(ccAgg).reduce((s, x) => s + (x.val || 0), 0);
  const kostenstellenRows = Object.keys(ccAgg).sort().map((cc) => {
    const x = ccAgg[cc];
    const share = totalCcVal > 0 ? (x.val / totalCcVal) : 0;
    return [
      cc,
      fmtNumDe(x.cnt, 0),
      fmtNumDe(x.qty, 0),
      fmtNumDe(x.val, 2),
      fmtNumDe(share, 6)
    ];
  });

  // Lagerübersicht (auf Basis Bestand Ende)
  const lagerAgg = {};
  let totalMaterialEnd = 0;
  endSnapshot.forEach((a) => {
    const lager = String(a.lager || "");
    const qty = toNumber(a.bestand, 0);
    const unit = toNumber(a.wertEinzeln, 0);
    const val = qty * unit;
    totalMaterialEnd += val;
    if (!lagerAgg[lager]) lagerAgg[lager] = { articles: 0, qty: 0, val: 0 };
    lagerAgg[lager].articles += 1;
    lagerAgg[lager].qty += qty;
    lagerAgg[lager].val += val;
  });
  const lagerRows = Object.keys(lagerAgg).sort().map((lager) => {
    const x = lagerAgg[lager];
    const share = totalMaterialEnd > 0 ? (x.val / totalMaterialEnd) : 0;
    return [
      lager,
      fmtNumDe(x.articles, 0),
      fmtNumDe(x.qty, 0),
      fmtNumDe(x.val, 2),
      fmtNumDe(share, 6)
    ];
  });

  // Audit_Integrität (1 Zeile)
  const auditRows = [[
    String(ym),
    splitLocalDateTime(sealCreatedAt).date + " " + splitLocalDateTime(sealCreatedAt).time,
    sealHash,
    prevHash,
    integrityStatus,
    integrityExplanation
  ]];

  // Übersicht Meta (als TXT für Copy/Paste)
  const statusOverview = (sealRec && integrityStatus === "OK") ? "Abgeschlossen – OK" : (sealRec ? "Abgeschlossen – Abweichung" : "Abgeschlossen – OK");
  const metaTxt = [
    `Monat (YYYY-MM): ${ym}`,
    `Abschlussdatum / Uhrzeit: ${new Date().toLocaleString("de-DE")}`,
    `Status: ${statusOverview}`,
    "",
    "Hinweis: Öffne das Template, füge die CSV-Inhalte je Sheet ab Zelle A1 ein (oder importiere sie).",
    "Die Übersicht und Grafiken aktualisieren sich automatisch über Tabellen/Formeln."
  ].join("\n");

  // CSVs erstellen
  const csvBestandAnfang = toCsv([
    "Artikelnummer","Bezeichnung","Lager","Bestand Monatsanfang","Einzelwert","Gesamtwert"
  ], startRows);
  const csvBestandEnde = toCsv([
    "Artikelnummer","Bezeichnung","Lager","Stand Monatsende","Einzelwert","Gesamtwert"
  ], endRows);
  const csvBewegungen = toCsv([
    "Datum","Uhrzeit","Artikelnummer","Bezeichnung","Lager","Richtung (Zugang / Entnahme)","Menge","Einzelwert","Wertbewegung (+ / −)","Kostenstelle","Mitarbeiter","Grund / Notiz"
  ], bewegungenRows);
  const csvEntzug = toCsv([
    "Artikelnummer","Bezeichnung","Lager","Gesamtmenge Entnahme","Gesamtwert Entnahme","Gesamtmenge Zugang","Gesamtwert Zugang"
  ], entzugRows);
  const csvKostenstellen = toCsv([
    "Kostenstelle","Anzahl Entnahmen","Gesamtmenge","Gesamtwert","Anteil am Monatswert (%)"
  ], kostenstellenRows);
  const csvLager = toCsv([
    "Lager","Anzahl Artikel","Gesamtbestand","Materialwert","Anteil %"
  ], lagerRows);
  const csvAudit = toCsv([
    "Monat","Abschlusszeitpunkt","Seal-Hash","Vorheriger Hash","Status","Erklärung"
  ], auditRows);

  
// Wenn SheetJS (window.XLSX) verfügbar ist: echte, gefüllte XLSX erzeugen (1 Klick).
const hasXlsx = (typeof window !== "undefined" && window.XLSX && window.XLSX.utils && typeof window.XLSX.writeFile === "function");
if (hasXlsx && !forceTemplate) {
  try {
    const base = "monatsabschluss-" + ym;
    const wb = XLSX.utils.book_new();

    // Übersicht (Key/Value – Auditor-Einstieg)
    const totalArticles = endSnapshot.length;
    const totalBookings = bookingsM.length;
    const totalIn = bookingsM.filter(b => b && b.direction === "in").length;
    const totalOut = bookingsM.filter(b => b && b.direction !== "in").length;
    const diffAbs = (totalMaterialEnd - totalMaterialStart);
    const diffPct = (totalMaterialStart > 0) ? (diffAbs / totalMaterialStart) : 0;
    const activeCostCenters = Object.keys(ccAgg).length;
    const statusOverview = (sealRec && integrityStatus === "OK") ? "Abgeschlossen – OK"
      : (sealRec ? "Abgeschlossen – Abweichung" : "Abgeschlossen – OK");

    const overviewRows = [
      ["Monat (YYYY-MM)", ym],
      ["Abschlussdatum / Uhrzeit", new Date().toLocaleString("de-DE")],
      ["Status", statusOverview],
      ["Anzahl Artikel gesamt", totalArticles],
      ["Anzahl Buchungen gesamt", totalBookings],
      ["Anzahl Entnahmen", totalOut],
      ["Anzahl Zugänge", totalIn],
      ["Materialwert Anfang Monat", totalMaterialStart],
      ["Materialwert Ende Monat", totalMaterialEnd],
      ["Differenz absolut", diffAbs],
      ["Differenz %", diffPct * 100],
      ["Anzahl Kostenstellen aktiv", activeCostCenters],
      ["Hinweistext", "Dieser Monatsabschluss umfasst alle Lagerbewegungen, Artikelstände und Kostenstellenbuchungen des Monats."]
    ];
    const wsOverview = XLSX.utils.aoa_to_sheet([["Feld", "Wert"], ...overviewRows]);
    XLSX.utils.book_append_sheet(wb, wsOverview, "Übersicht");

    // Helper: rows in Zahlen umwandeln (wenn sie als 'de-DE' Strings vorliegen)
    function n(v) { return (typeof v === "number") ? v : toNumber(v, v); }

    // Bestand_Anfang
    const wsStart = XLSX.utils.aoa_to_sheet([
      ["Artikelnummer","Bezeichnung","Lager","Bestand Monatsanfang","Einzelwert","Gesamtwert"],
      ...startRows.map(r => [r[0], r[1], r[2], n(r[3]), n(r[4]), n(r[5])])
    ]);
    XLSX.utils.book_append_sheet(wb, wsStart, "Bestand_Anfang");

    // Bestand_Ende
    const wsEnd = XLSX.utils.aoa_to_sheet([
      ["Artikelnummer","Bezeichnung","Lager","Stand Monatsende","Einzelwert","Gesamtwert"],
      ...endRows.map(r => [r[0], r[1], r[2], n(r[3]), n(r[4]), n(r[5])])
    ]);
    XLSX.utils.book_append_sheet(wb, wsEnd, "Bestand_Ende");

    // Bewegungen
    const wsMoves = XLSX.utils.aoa_to_sheet([
      ["Datum","Uhrzeit","Artikelnummer","Bezeichnung","Lager","Richtung (Zugang / Entnahme)","Menge","Einzelwert","Wertbewegung (+ / −)","Kostenstelle","Mitarbeiter","Grund / Notiz"],
      ...bewegungenRows.map(r => [r[0], r[1], r[2], r[3], r[4], r[5], n(r[6]), n(r[7]), n(r[8]), r[9], r[10], r[11]])
    ]);
    XLSX.utils.book_append_sheet(wb, wsMoves, "Bewegungen");

    // Entnahmen_Zugänge
    const wsAgg = XLSX.utils.aoa_to_sheet([
      ["Artikelnummer","Bezeichnung","Lager","Gesamtmenge Entnahme","Gesamtwert Entnahme","Gesamtmenge Zugang","Gesamtwert Zugang"],
      ...entzugRows.map(r => [r[0], r[1], r[2], n(r[3]), n(r[4]), n(r[5]), n(r[6])])
    ]);
    XLSX.utils.book_append_sheet(wb, wsAgg, "Entnahmen_Zugänge");

    // Kostenstellen (Anteil in %)
    const wsCC = XLSX.utils.aoa_to_sheet([
      ["Kostenstelle","Anzahl Entnahmen","Gesamtmenge","Gesamtwert","Anteil am Monatswert (%)"],
      ...kostenstellenRows.map(r => [r[0], n(r[1]), n(r[2]), n(r[3]), n(r[4]) * 100])
    ]);
    XLSX.utils.book_append_sheet(wb, wsCC, "Kostenstellen");

    // Lagerübersicht (Anteil in %)
    const wsLager = XLSX.utils.aoa_to_sheet([
      ["Lager","Anzahl Artikel","Gesamtbestand","Materialwert","Anteil %"],
      ...lagerRows.map(r => [r[0], n(r[1]), n(r[2]), n(r[3]), n(r[4]) * 100])
    ]);
    XLSX.utils.book_append_sheet(wb, wsLager, "Lagerübersicht");

    // Audit_Integrität
    const wsAudit = XLSX.utils.aoa_to_sheet([
      ["Monat","Abschlusszeitpunkt","Seal-Hash","Vorheriger Hash","Status","Erklärung"],
      ...auditRows
    ]);
    XLSX.utils.book_append_sheet(wb, wsAudit, "Audit_Integrität");

    // Schreiben
    XLSX.writeFile(wb, base + ".xlsx", { bookType: "xlsx", compression: true });

    // Audit-Event
    try {
      addAuditEvent("month.close.xlsx.exported", {
        period: ym,
        counts: { bookings: bookingsM.length, audit: auditM.length, articles: endSnapshot.length },
        integrity: integrityStatus,
        hash: sealHash || ""
      });
    } catch (e) {}

    return; // ✅ XLSX-Export fertig – keine CSV/Template-Fallbacks
  } catch (e) {
    console.warn("[lager] XLSX-Export fehlgeschlagen, fallback auf Template+CSV:", e);
    // Fallback läuft weiter unten
  }
}

// Template + CSVs als *eine* ZIP-Datei (Browser blockiert sonst oft viele Einzel-Downloads)
  try {
    const base = "monatsabschluss-" + ym;

    const files = [
      { name: base + "-template.xlsx", data: base64ToBytes(MONTHCLOSE_TEMPLATE_XLSX_B64) },
      { name: base + "-meta.txt", data: utf8ToBytes(metaTxt) },
      { name: base + "-Bestand_Anfang.csv", data: utf8ToBytes(csvBestandAnfang) },
      { name: base + "-Bestand_Ende.csv", data: utf8ToBytes(csvBestandEnde) },
      { name: base + "-Bewegungen.csv", data: utf8ToBytes(csvBewegungen) },
      { name: base + "-Entnahmen_Zugaenge.csv", data: utf8ToBytes(csvEntzug) },
      { name: base + "-Kostenstellen.csv", data: utf8ToBytes(csvKostenstellen) },
      { name: base + "-Lageruebersicht.csv", data: utf8ToBytes(csvLager) },
      { name: base + "-Audit_Integritaet.csv", data: utf8ToBytes(csvAudit) }
    ];

    const zipName = base + (forceTemplate ? "-diagramme" : "-template-csv") + ".zip";
    downloadBlob(zipName, zipStore(files));
  } catch (e) {
    console.warn("[lager] ZIP-Export fehlgeschlagen, fallback auf Einzel-Downloads:", e);
    const base = "monatsabschluss-" + ym;
    downloadBase64File(base + "-template.xlsx", MONTHCLOSE_TEMPLATE_XLSX_B64, "application/vnd.openxmlformats-officedocument.spreadsheetml.sheet");
    downloadText(base + "-meta.txt", metaTxt, "text/plain;charset=utf-8");
    downloadText(base + "-Bestand_Anfang.csv", csvBestandAnfang, "text/csv;charset=utf-8");
    downloadText(base + "-Bestand_Ende.csv", csvBestandEnde, "text/csv;charset=utf-8");
    downloadText(base + "-Bewegungen.csv", csvBewegungen, "text/csv;charset=utf-8");
    downloadText(base + "-Entnahmen_Zugaenge.csv", csvEntzug, "text/csv;charset=utf-8");
    downloadText(base + "-Kostenstellen.csv", csvKostenstellen, "text/csv;charset=utf-8");
    downloadText(base + "-Lageruebersicht.csv", csvLager, "text/csv;charset=utf-8");
    downloadText(base + "-Audit_Integritaet.csv", csvAudit, "text/csv;charset=utf-8");
  }

  // Audit-Event

  try {
    addAuditEvent("month.close.reporting.exported", {
      period: ym,
      counts: { bookings: bookingsM.length, audit: auditM.length, articles: endSnapshot.length },
      integrity: integrityStatus,
      hash: sealHash || ""
    });
  } catch (e) {}
}


// --- Reporting UI (Button + Dialog) -----------------------------------
function ensureReportingDialog() {
  let dlg = byId("invReportingDialog");
  if (dlg) return dlg;

  dlg = document.createElement("dialog");
  dlg.id = "invReportingDialog";
  dlg.className = "dialog";
  dlg.innerHTML = `
    <div class="card" style="max-width:820px;">
      <div style="display:flex; align-items:center; justify-content:space-between; gap:10px;">
        <h3 style="margin:0;">📊 Reporting</h3>
        <button type="button" class="btn btn-ghost" id="invReportingCloseBtn">Schließen</button>
      </div>
      <p style="margin:8px 0 14px 0; opacity:.85;">
        Monatsabschluss = <b>Excel-Datei + CSV-Daten</b>. Das Template enthält Formeln und Grafiken.
      </p>

      <div style="display:flex; gap:10px; flex-wrap:wrap; margin-bottom:12px;">
        <button type="button" class="btn btn-primary" id="invReportingMonthCloseBtn">📦 Monatsabschluss (Excel)</button>
        <button type="button" class="btn btn-ghost" id="invReportingMonthCloseChartsBtn">📈 Monatsabschluss (Diagramme: Template + CSV)</button>
        <button type="button" class="btn btn-ghost" id="invReportingTemplateBtn">📄 Excel-Template herunterladen</button>
        <button type="button" class="btn btn-ghost" id="invReportingProtocolBtn">🧾 Monatsabschluss-Protokoll</button>
        <button type="button" class="btn btn-ghost" id="invReportingBundleBtn">📦 Monatsmappe exportieren</button>
      </div>

      <details style="margin-top:10px;">
        <summary style="cursor:pointer;">🔐 Erweitert / Prüfung</summary>
        <div style="margin-top:10px; display:flex; gap:10px; flex-wrap:wrap;">
          <button type="button" class="btn btn-ghost" id="invReportingSealBtn">🔐 Monatsabschluss / Seal</button>
        </div>
        <p style="margin:10px 0 0 0; opacity:.8; font-size:13px;">
          Seal ist optional: Er macht nachträgliche Änderungen sichtbar (Integrität). Für den Alltag reicht der Monatsabschluss-Export.
        </p>
      </details>
    </div>
  `;
  document.body.appendChild(dlg);

  // Close handlers
  const closeBtn = byId("invReportingCloseBtn");
  if (closeBtn) closeBtn.addEventListener("click", () => closeDialog(dlg));
  dlg.addEventListener("click", (e) => { if (e.target === dlg) closeDialog(dlg); });

  // Actions
  const monthCloseBtn = byId("invReportingMonthCloseBtn");
  if (monthCloseBtn) monthCloseBtn.addEventListener("click", async () => {
    closeDialog(dlg);
    await exportMonthCloseReportingPackFlow("xlsx");
  });

  const monthCloseChartsBtn = byId("invReportingMonthCloseChartsBtn");
  if (monthCloseChartsBtn) monthCloseChartsBtn.addEventListener("click", async () => {
    closeDialog(dlg);
    await exportMonthCloseReportingPackFlow("charts");
  });

  const tmplBtn = byId("invReportingTemplateBtn");
  if (tmplBtn) tmplBtn.addEventListener("click", () => downloadMonthCloseExcelTemplate());

  const protoBtn = byId("invReportingProtocolBtn");
  if (protoBtn) protoBtn.addEventListener("click", async () => {
    closeDialog(dlg);
    await printMonthProtocolFlow();
  });

  const bundleBtn = byId("invReportingBundleBtn");
  if (bundleBtn) bundleBtn.addEventListener("click", async () => {
    closeDialog(dlg);
    await exportMonthlyBundleFlow();
  });

  const sealBtn = byId("invReportingSealBtn");
  if (sealBtn) sealBtn.addEventListener("click", async () => {
    closeDialog(dlg);
    await sealMonthFlow();
  });

  return dlg;
}



async function sealMonthFlow() {
  const now = new Date();
  // Default: Vormonat (typischer Monatsabschluss)
  const def = (() => {
    const d = new Date(Date.UTC(now.getUTCFullYear(), now.getUTCMonth(), 1));
    d.setUTCMonth(d.getUTCMonth() - 1);
    const y = d.getUTCFullYear();
    const m = String(d.getUTCMonth() + 1).padStart(2, "0");
    return y + "-" + m;
  })();

  const ym = prompt("Monat für Monatsabschluss (YYYY-MM):", def);
  if (!ym) return;

  // Duplikate verhindern
  const exists = (state.seals || []).some(s => String(s.period || "") === String(ym));
  if (exists) {
        uiAlert(
      "Für " + ym + " existiert bereits eine Versiegelung.",
      "A seal already exists for " + ym + ".",
      "\u26a0\ufe0f Hinweis",
      "\u26a0\ufe0f Note",
      "OK",
      "OK"
    );
    return;
  }

  const okConfirm = await uiConfirm(
    "Monatsabschluss/Versiegelung für " + ym + " erstellen?\n\nHinweis: technisch stark, aber ohne Server/WORM nicht 'gerichtsfest'.",
    "Create monthly close/seal for " + ym + "?\n\nNote: technically strong, but without a server/WORM it is not legally tamper-proof.",
    "⚠️ Monatsabschluss",
    "⚠️ Monthly close",
    "OK",
    "OK",
    "Abbrechen",
    "Cancel"
  );
  if (!okConfirm) {
    return;
  }

  try {
    const built = await buildSealPayloadForMonth(ym);
    const rec = {
      id: "seal_" + Date.now().toString(36) + "_" + Math.random().toString(36).slice(2, 7),
      ts: nowIso(),
      period: ym,
      range: built.payload.range,
      prevHash: built.payload.prevHash || "",
      hash: built.hash,
      counts: built.payload.counts
    };

    state.seals = state.seals || [];
    state.seals.push(rec);
    saveSeals(state.seals);

    // Audit-Event dazu
    try {
      addAuditEvent("month.sealed", {
        period: ym,
        sealId: rec.id,
        hash: rec.hash,
        prevHash: rec.prevHash || "",
        counts: rec.counts
      });
    } catch (e) {}

    // Export: Seal-Paket als JSON (Payload + Hash + Record)
    const sealPackage = { record: rec, payload: built.payload, hash: built.hash };
    downloadText("seal-" + ym + ".json", JSON.stringify(sealPackage, null, 2), "application/json");

        uiAlert(
      "Versiegelung erstellt.\n\nMonat: " + ym + "\nHash: " + rec.hash.slice(0, 16) + "…\nExport wurde heruntergeladen.",
      "Seal created.\n\nMonth: " + ym + "\nHash: " + rec.hash.slice(0, 16) + "…\nExport downloaded.",
      "\u2705 Fertig",
      "\u2705 Done",
      "OK",
      "OK"
    );
    refresh && refresh(); // falls verfügbar
    renderSealStatus && renderSealStatus();
  } catch (e) {
        uiAlert(
      "Versiegelung fehlgeschlagen: " + (e && e.message ? e.message : e),
      "Seal failed: " + (e && e.message ? e.message : e),
      "\u274c Fehler",
      "\u274c Error",
      "OK",
      "OK"
    );
  }
}

async function verifySeals() {
  const seals = Array.isArray(state.seals) ? state.seals : [];
  const issues = [];
  for (const s of seals) {
    try {
      const built = await buildSealPayloadForMonth(String(s.period || ""));
      if (String(built.hash) !== String(s.hash || "")) {
        issues.push({ period: s.period, expected: s.hash, actual: built.hash });
      }
    } catch (e) {
      issues.push({ period: s.period, error: (e && e.message ? e.message : String(e)) });
    }
  }
  return issues;
}

function ensureSealBanner(root) {
  if (!root) return null;
  let el = byId("invSealBanner");
  if (el) return el;
  el = document.createElement("div");
  el.id = "invSealBanner";
  el.style.cssText = "margin:10px 0 0 0; padding:10px 12px; border-radius:12px; border:1px solid rgba(0,0,0,.15); display:none; font-size:13px;";
  // möglichst oben im Card Header Bereich
  try {
    const card = root.querySelector(".card");
    if (card) card.insertBefore(el, card.firstChild);
    else root.prepend(el);
  } catch (e) {
    root.prepend(el);
  }
  return el;
}

async function renderSealStatus() {
  try {
    const root = byId("lagerverwaltung");
    if (!root) return;

    // UI: Badge-Styles für gesperrte Artikel
    ensureReceiptBlockStyles();
    const banner = ensureSealBanner(root);
    if (!banner) return;

    const issues = await verifySeals();
    if (issues.length) {
      banner.style.display = "block";
      banner.innerHTML =
        "<b>ℹ️ Hinweis:</b><br>" +
"Es gibt abgeschlossene Monate mit Abweichungen. Details siehe „Reporting“."+
        "<span style='font-family:monospace; font-size:12px; opacity:.85;'>" +
        "</span>";
    } else if ((state.seals || []).length) {
      banner.style.display = "block";
      banner.style.background = "rgba(16,185,129,.10)";
      const last = state.seals[state.seals.length - 1];
      banner.innerHTML =
        "<b>🔐 Versiegelungen ok</b><br>" +
        "Anzahl: " + (state.seals.length) + " · Letzte: " + escapeHtml(String(last.period || "")) +
        " · Hash: <span style='font-family:monospace;'>" + escapeHtml(String(last.hash || "").slice(0, 16)) + "…</span>";
    } else {
      banner.style.display = "none";
    }
  } catch (e) {
    // egal
  }
}

// --- Monatsmappe Export + Protokoll (Print-View) -------------------------

function csvFromArticles(list) {
  const rows = Array.isArray(list) ? list : [];
  // CSV entspricht exportCsv(), aber als String zurückgegeben
  const header = [
    getColumnLabel("artikelnummer"),
    getColumnLabel("bezeichnung"),
    getColumnLabel("hersteller"),
    getColumnLabel("kategorie"),
    getColumnLabel("lager"),
    "Lagerart",
    "Lagerplatztyp",
    "Sicherheitsstufe",
    getColumnLabel("lagerort"),
    getColumnLabel("mindestbestand"),
    getColumnLabel("bestand"),
    getColumnLabel("einheit"),
    getColumnLabel("notizen"),
    "Technischer Platz",
    "Equipment",
    "Anlage",
    getColumnLabel("wertEinzeln"),
    getColumnLabel("waehrung"),
    "Wareneingang",
    getColumnLabel("createdAt"),
    getColumnLabel("updatedAt")
  ];

  function csvVal(v) {
    if (v === null || v === undefined) return "";
    let s = String(v);
    if (s.indexOf('"') !== -1) s = s.replace(/"/g, '""');
    if (s.indexOf(";") !== -1 || s.indexOf("\n") !== -1 || s.indexOf("\r") !== -1) s = '"' + s + '"';
    return s;
  }

  const lines = [header.join(";")];
  rows.forEach((a) => {
    const row = [
      a.artikelnummer || "",
      a.bezeichnung || "",
      a.hersteller || "",
      a.kategorie || "",
      a.lager || "",
      a.lagerart || "",
      a.lagerplatztyp || "",
      a.sicherheitsstufe || "",
      a.lagerort || "",
      String(a.mindestbestand ?? ""),
      String(a.bestand ?? ""),
      a.einheit || "",
      a.notizen || "",
      a.technischerPlatz || "",
      a.equipment || "",
      a.anlage || "",
      (a.wertEinzeln != null ? String(a.wertEinzeln) : ""),
      a.waehrung || "",
      a.wareneingang || "",
      a.createdAt || "",
      a.updatedAt || ""
    ];
    lines.push(row.map(csvVal).join(";"));
  });

  // BOM für Excel
  return "\uFEFF" + lines.join("\n");
}

function monthDefaultPrevMonth() {
  const now = new Date();
  const d = new Date(Date.UTC(now.getUTCFullYear(), now.getUTCMonth(), 1));
  d.setUTCMonth(d.getUTCMonth() - 1);
  const y = d.getUTCFullYear();
  const m = String(d.getUTCMonth() + 1).padStart(2, "0");
  return y + "-" + m;
}

function fmtYmLabel(ym) {
  return String(ym || "").trim();
}

function buildMonthProtocolHtml(opts) {
  const o = opts || {};
  const ym = fmtYmLabel(o.ym);
  const createdAt = o.createdAt || nowIso();
  const hash = String(o.hash || "");
  const prevHash = String(o.prevHash || "");
  const counts = o.counts || {};
  const sealId = String(o.sealId || "");
  const range = o.range || {};
  const company = (typeof APP !== "undefined" && APP && APP.company) ? APP.company : "OpsDeck360";

  const hShort = hash ? (hash.slice(0, 16) + "…") : "–";
  const pShort = prevHash ? (prevHash.slice(0, 16) + "…") : "–";

  const css = `
    <style>
      body { font-family: system-ui, -apple-system, Segoe UI, Roboto, Arial; margin: 18mm; color:#111; }
      .header { display:flex; justify-content:space-between; align-items:flex-end; gap:12px; border-bottom:1px solid #ddd; padding-bottom:10px; margin-bottom:14px; }
      .company { font-size:10pt; text-transform:uppercase; letter-spacing:.06em; opacity:.8; }
      .title { font-size:18pt; font-weight:800; }
      .meta { font-size:9.5pt; text-align:right; opacity:.85; }
      h2 { font-size:12pt; margin:16px 0 6px; }
      table { width:100%; border-collapse:collapse; font-size:10pt; }
      th, td { border:1px solid #ddd; padding:6px 8px; vertical-align:top; }
      th { background:#f6f6f6; text-align:left; }
      .mono { font-family: ui-monospace, SFMono-Regular, Menlo, Consolas, monospace; font-size:9.5pt; }
      .grid { display:grid; grid-template-columns: 1fr 1fr; gap:12px; }
      .box { border:1px solid #ddd; border-radius:10px; padding:10px 12px; }
      .sign { margin-top:18mm; display:flex; justify-content:space-between; gap:12mm; }
      .line { border-top:1px solid #333; padding-top:3mm; width:80mm; font-size:10pt; }
      .small { font-size:9pt; opacity:.85; }
      @media print {
        button { display:none !important; }
      }
    </style>
  `;

  const esc = (v) => escapeHtml(v == null ? "" : String(v));

  return `<!doctype html>
<html lang="de">
<head>
  <meta charset="utf-8" />
  <title>Monatsabschluss-Protokoll ${esc(ym)}</title>
  ${css}
</head>
<body>
  <div class="header">
    <div>
      <div class="company">${esc(company)}</div>
      <div class="title">Monatsabschluss-Protokoll</div>
      <div class="small">Periode: <b>${esc(ym)}</b></div>
    </div>
    <div class="meta">
      <div>Erstellt: ${esc(new Date(createdAt).toLocaleString("de-DE"))}</div>
      <div>Seal-ID: ${esc(sealId || "-")}</div>
    </div>
  </div>

  <div class="grid">
    <div class="box">
      <h2>Umfang</h2>
      <table>
        <tr><th>Zeitraum (UTC)</th><td class="mono">${esc(range.startIso || "-")} → ${esc(range.endIso || "-")}</td></tr>
        <tr><th>Artikel (akt. Snapshot)</th><td>${esc(String(counts.articles ?? "-"))}</td></tr>
        <tr><th>Buchungen im Monat</th><td>${esc(String(counts.bookings ?? "-"))}</td></tr>
        <tr><th>Audit-Events im Monat</th><td>${esc(String(counts.audit ?? "-"))}</td></tr>
      </table>
    </div>

    <div class="box">
      <h2>Integrität</h2>
      <table>
        <tr><th>Hash (SHA-256)</th><td class="mono">${esc(hash || "-")}</td></tr>
        <tr><th>Kurz</th><td class="mono">${esc(hShort)}</td></tr>
        <tr><th>Vorheriger Hash</th><td class="mono">${esc(prevHash || "-")}</td></tr>
        <tr><th>Prev kurz</th><td class="mono">${esc(pShort)}</td></tr>
      </table>
      <div class="small" style="margin-top:6px;">
        Hinweis: Der Hash ist reproduzierbar, solange Buchungen/Audit/Artikel-Snapshot nicht verändert werden.
        Ohne externen WORM-Speicher/Server bleibt es „technisch stark“, aber nicht automatisch rechtlich „gerichtsfest“.
      </div>
    </div>
  </div>

  <h2>Unterschriften</h2>
  <div class="sign">
    <div class="line">Datum</div>
    <div class="line">Name / Unterschrift</div>
  </div>

  <script>
    window.onload = function() { window.print(); };
  </script>
</body>
</html>`;
}

async function exportMonthlyBundleFlow() {
  const ym = prompt("Monatsmappe exportieren (YYYY-MM):", monthDefaultPrevMonth());
  if (!ym) return;

  const range = monthRange(ym);
  if (!range) {
        uiAlert(
      "Ungültiger Monat: " + ym,
      "Invalid month: " + ym,
      "\ud83d\udce6 Monatsabschluss",
      "\ud83d\udce6 Month close",
      "OK",
      "OK"
    );
    return;
  }

  // Daten filtern
  const bookingsM = (state.bookings || []).filter(b => inRange(b.ts, range.startIso, range.endIso));
  const auditM    = (state.audit || []).filter(e => inRange(e.ts, range.startIso, range.endIso));

  // Seal: wenn vorhanden, nutzen – sonst anbieten, einen zu erstellen
  let sealRec = (state.seals || []).find(s => String(s.period || "") === String(ym)) || null;
  let sealPackage = null;

  if (!sealRec) {
    const doSeal = await uiConfirm(
      "Für " + ym + " gibt es noch keinen Seal.\n\nJetzt einen Monatsabschluss/Seal erstellen (empfohlen)?",
      "No seal exists for " + ym + ".\n\nCreate a monthly close/seal now (recommended)?",
      "🧾 Seal",
      "🧾 Seal",
      "OK",
      "OK",
      "Abbrechen",
      "Cancel"
    );
    if (doSeal) {
      await sealMonthFlow(); // Nutzer wählt Monat erneut -> ok
      sealRec = (state.seals || []).find(s => String(s.period || "") === String(ym)) || null;
    }
  }

  if (sealRec) {
    try {
      // Rekonstruiere Payload und Hash reproduzierbar (stark für Export/Prüfung)
      const built = await buildSealPayloadForMonth(ym);
      sealPackage = { record: sealRec, payload: built.payload, hash: built.hash };
    } catch (e) {
      sealPackage = { record: sealRec, error: (e && e.message ? e.message : String(e)) };
    }
  } else {
    // Minimal: Hash über Monatsdaten (ohne Seal-Kette)
    const minimal = {
      schema: "opsdeck.monthbundle.v1",
      period: ym,
      range,
      counts: { bookings: bookingsM.length, audit: auditM.length, articles: (state.articles || []).length },
      bookings: bookingsM,
      audit: auditM
    };
    const canon = stableStringify(minimal);
    const h = await sha256Hex(canon);
    sealPackage = { record: null, payload: minimal, hash: h };
  }

  // Artikel CSV (voller Bestand als „Snapshot“)
  const csv = csvFromArticles(state.articles || []);
  const bookingsJson = JSON.stringify(bookingsM, null, 2);
  const auditJson = JSON.stringify(auditM, null, 2);
  const sealJson = JSON.stringify(sealPackage, null, 2);

  // Protokoll (druckbar) als HTML exportieren
  const protoHtml = buildMonthProtocolHtml({
    ym,
    createdAt: nowIso(),
    hash: sealPackage && sealPackage.hash ? sealPackage.hash : "",
    prevHash: sealPackage && sealPackage.record ? (sealPackage.record.prevHash || "") : "",
    sealId: sealPackage && sealPackage.record ? (sealPackage.record.id || "") : "",
    range: range,
    counts: {
      articles: (state.articles || []).length,
      bookings: bookingsM.length,
      audit: auditM.length
    }
  });

  // 1-Klick: mehrere Downloads nacheinander (ohne ZIP-Lib)
  const base = "monatsmappe-" + ym;
  const queue = [
    { name: base + "-artikel.csv",  text: csv,         mime: "text/csv;charset=utf-8" },
    { name: base + "-buchungen.json", text: bookingsJson, mime: "application/json" },
    { name: base + "-audit.json",     text: auditJson,    mime: "application/json" },
    { name: base + "-seal.json",      text: sealJson,     mime: "application/json" },
    { name: base + "-protokoll.html", text: protoHtml,    mime: "text/html;charset=utf-8" }
  ];

  // Browser mögen nicht zu viele Downloads "auf einmal" -> kleine Delays
  let i = 0;
  const tick = () => {
    const item = queue[i++];
    if (!item) return;
    downloadText(item.name, item.text, item.mime);
    setTimeout(tick, 250);
  };
  tick();

  // Audit-Event
  try {
    addAuditEvent("month.bundle.exported", {
      period: ym,
      counts: { articles: (state.articles || []).length, bookings: bookingsM.length, audit: auditM.length },
      hash: sealPackage && sealPackage.hash ? sealPackage.hash : ""
    });
  } catch (e) {}
}

async function printMonthProtocolFlow() {
  const ym = prompt("Monatsabschluss-Protokoll drucken (YYYY-MM):", monthDefaultPrevMonth());
  if (!ym) return;

  const range = monthRange(ym);
  if (!range) {
        uiAlert(
      "Ungültiger Monat: " + ym,
      "Invalid month: " + ym,
      "\ud83d\udce6 Monatsabschluss",
      "\ud83d\udce6 Month close",
      "OK",
      "OK"
    );
    return;
  }

  const bookingsM = (state.bookings || []).filter(b => inRange(b.ts, range.startIso, range.endIso));
  const auditM    = (state.audit || []).filter(e => inRange(e.ts, range.startIso, range.endIso));

  // Seal oder Minimal-Hash
  let sealRec = (state.seals || []).find(s => String(s.period || "") === String(ym)) || null;
  let hash = "";
  let prevHash = "";
  let sealId = "";

  if (sealRec) {
    try {
      const built = await buildSealPayloadForMonth(ym);
      hash = built.hash;
      prevHash = built.payload.prevHash || "";
      sealId = sealRec.id || "";
    } catch (e) {
      // fallback minimal
    }
  }
  if (!hash) {
    const minimal = {
      schema: "opsdeck.monthprotocol.v1",
      period: ym,
      range,
      counts: { bookings: bookingsM.length, audit: auditM.length, articles: (state.articles || []).length },
      bookings: bookingsM,
      audit: auditM
    };
    hash = await sha256Hex(stableStringify(minimal));
  }

  const html = buildMonthProtocolHtml({
    ym,
    createdAt: nowIso(),
    hash,
    prevHash,
    sealId,
    range,
    counts: { articles: (state.articles || []).length, bookings: bookingsM.length, audit: auditM.length }
  });

  const win = window.open("", "_blank", "width=980,height=760");
  if (!win) {
        uiAlert(
      "Popup-Blocker aktiv – bitte Popups erlauben, um das Protokoll zu drucken.",
      "Popup blocker is active — please allow popups to print the protocol.",
      "\u26a0\ufe0f Hinweis",
      "\u26a0\ufe0f Note",
      "OK",
      "OK"
    );
    return;
  }
  win.document.open();
  win.document.write(html);
  win.document.close();
}

  function addAuditEvent(type, payload) {
    // Leichtgewichtiges Audit-Event (erweiterbar: Hash-Kette, User-Identity, etc.)
    try {
      const evt = {
        id: "evt" + Math.random().toString(36).slice(2),
        ts: nowIso(),
        type: String(type || "event"),
        payload: payload || {}
      };
      state.audit.push(evt);
      saveAudit(state.audit);
    } catch (e) {
      console.warn("[lager] Audit-Event konnte nicht geschrieben werden:", e);
    }
  }

  // --- Audit-Historie UI ---------------------------------------------
  function ensureAuditDialog() {
    let dlg = byId("invAuditDialog");
    if (dlg) return dlg;

    dlg = document.createElement("dialog");
    dlg.id = "invAuditDialog";
    dlg.className = "dialog";
    dlg.innerHTML = `
      <div class="card" style="max-width:920px;">
        <div style="display:flex; align-items:center; gap:10px; justify-content:space-between;">
          <h3 style="margin:0;">Historie / Audit-Log</h3>
          <button type="button" class="btn btn-ghost" id="invAuditCloseBtn">Schließen</button>
        </div>
        <p style="margin:8px 0 12px 0; opacity:.8;">
          Hier siehst du alle Änderungen (append-only). Ohne Login wird der "Bearbeiter" aus Buchungsfeldern übernommen.
        </p>
        <div style="display:flex; gap:10px; flex-wrap:wrap; margin-bottom:10px;">
          <button type="button" class="btn btn-ghost" id="invAuditExportBtn">Audit exportieren (JSON)</button>
          <label class="btn btn-ghost" style="cursor:pointer;">
            Audit importieren (JSON)
            <input id="invAuditImportFile" type="file" accept="application/json" style="display:none;">
          </label>
        </div>
        <div style="max-height:60vh; overflow:auto; border:1px solid rgba(0,0,0,.08); border-radius:12px;">
          <table class="table" style="width:100%; border-collapse:collapse;">
            <thead>
              <tr>
                <th style="text-align:left; padding:10px;">Zeit</th>
                <th style="text-align:left; padding:10px;">Typ</th>
                <th style="text-align:left; padding:10px;">Artikel</th>
                <th style="text-align:left; padding:10px;">Menge</th>
                <th style="text-align:left; padding:10px;">Preis</th>
                <th style="text-align:left; padding:10px;">Bearbeiter</th>
                <th style="text-align:left; padding:10px;">Details</th>
              </tr>
            </thead>
            <tbody id="invAuditTableBody"></tbody>
          </table>
        </div>
      </div>
    `;
    document.body.appendChild(dlg);

    // Close
    const closeBtn = byId("invAuditCloseBtn");
    if (closeBtn) closeBtn.addEventListener("click", () => closeDialog(dlg));
    dlg.addEventListener("click", (e) => { if (e.target === dlg) closeDialog(dlg); });

    // Export
    const exportBtn = byId("invAuditExportBtn");
    if (exportBtn) exportBtn.addEventListener("click", () => {
      try {
        const blob = new Blob([JSON.stringify(state.audit || [], null, 2)], { type: "application/json" });
        const a = document.createElement("a");
        a.href = URL.createObjectURL(blob);
        a.download = "lager-audit-log.json";
        document.body.appendChild(a);
        a.click();
        a.remove();
        setTimeout(() => URL.revokeObjectURL(a.href), 250);
      } catch (e) {
                uiAlert(
          "Audit-Export fehlgeschlagen: " + (e && e.message ? e.message : e),
          "Audit export failed: " + (e && e.message ? e.message : e),
          "\u274c Fehler",
          "\u274c Error",
          "OK",
          "OK"
        );
      }
    });

    // Import (optional: append)
    const importFile = byId("invAuditImportFile");
    if (importFile) importFile.addEventListener("change", async () => {
      const file = importFile.files && importFile.files[0];
      if (!file) return;
      try {
        const text = await file.text();
        const list = JSON.parse(text);
        if (!Array.isArray(list)) throw new Error("JSON ist kein Array");
        // Append-only: wir hängen an und speichern neu
        state.audit = (state.audit || []).concat(list);
        saveAudit(state.audit);
        renderAuditTable();
                uiAlert(
          "Audit importiert (angehängt).",
          "Audit imported (appended).",
          "\u2705 Fertig",
          "\u2705 Done",
          "OK",
          "OK"
        );
      } catch (e) {
                uiAlert(
          "Audit-Import fehlgeschlagen: " + (e && e.message ? e.message : e),
          "Audit import failed: " + (e && e.message ? e.message : e),
          "\u274c Fehler",
          "\u274c Error",
          "OK",
          "OK"
        );
      } finally {
        importFile.value = "";
      }
    });

    return dlg;
  }

  function renderAuditTable() {
    const body = byId("invAuditTableBody") || byId("auditHistoryBody");
    if (!body) return;

    const legacy = body.id === "auditHistoryBody";
    const list = Array.isArray(state.audit) ? state.audit.slice().reverse() : [];

    body.innerHTML = list.map((evt) => {
      const p = evt && evt.payload ? evt.payload : {};
      const art = p.artikelnummer || p.bezeichnung || p.articleId || "";
      const qty = (p.amount !== undefined ? p.amount : (p.menge !== undefined ? p.menge : ""));
      const preis = (p.unitPrice !== undefined ? p.unitPrice : (p.preis !== undefined ? p.preis : ""));
      const waehr = (p.currency || p.waehrung || "");
      const employee = (p.employee || p.mitarbeiter || p.user || "");
      const tsLocal = formatDateTime(evt.ts) || (evt.ts || "");

      if (legacy) {
        const avgAfter = (p.avgPriceAfter != null && p.avgPriceAfter !== "") ? p.avgPriceAfter : "";
        let priceText = "";
        if (preis !== "" || avgAfter !== "") {
          if (avgAfter !== "") {
            priceText =
              String(preis || "") + (waehr ? (" " + waehr) : "") +
              " → " +
              String(avgAfter || "") + (waehr ? (" " + waehr) : "");
          } else {
            priceText = String(preis || "") + (waehr ? (" " + waehr) : "");
          }
        }

        return `
          <tr>
            <td>${escapeHtml(tsLocal)}</td>
            <td>${escapeHtml(evt.type || "")}</td>
            <td>${escapeHtml(String(art || ""))}</td>
            <td>${escapeHtml(String(qty || ""))}</td>
            <td>${escapeHtml(String(priceText || ""))}</td>
            <td>${escapeHtml(String(employee || ""))}</td>
          </tr>
        `;
      }

      const details = (() => {
        try { return JSON.stringify(p); } catch (e) { return ""; }
      })();

      return `
        <tr>
          <td style="padding:10px; white-space:nowrap;">${escapeHtml(tsLocal)}</td>
          <td style="padding:10px; white-space:nowrap;">${escapeHtml(evt.type || "")}</td>
          <td style="padding:10px;">${escapeHtml(String(art || ""))}</td>
          <td style="padding:10px;">${escapeHtml(String(qty || ""))}</td>
          <td style="padding:10px;">${escapeHtml(String(preis || ""))} ${escapeHtml(String(waehr || ""))}</td>
          <td style="padding:10px;">${escapeHtml(String(employee || ""))}</td>
          <td style="padding:10px;"><span style="opacity:.75; font-family:monospace; font-size:12px;">${escapeHtml(details)}</span></td>
        </tr>
      `;
    }).join("");
  }

  function wireLegacyAuditToolbar() {
    // Legacy HTML: <dialog id="auditHistoryDialog"> mit <tbody id="auditHistoryBody">
    // Wir ergänzen hier (a) horizontales Scrollen + nowrap und (b) Import/Export-Buttons,
    // ohne die bestehende Struktur zu zerstören.
    const legacyDlg = byId("auditHistoryDialog");
    if (!legacyDlg) return;

    // 1) Horizontales Scrollen erzwingen (rechts sichtbare Spalten)
    try {
      const scrollWrap = legacyDlg.querySelector("div[style*='max-height']") || legacyDlg.querySelector("div");
      const table = legacyDlg.querySelector("table");
      if (scrollWrap) {
        scrollWrap.style.overflow = "auto";
      }
      if (table) {
        // min-width sorgt dafür, dass unten ein horizontaler Scrollbalken erscheint
        table.style.width = "max-content";
        table.style.minWidth = "1100px";
        // nowrap für Header + Cells
        legacyDlg.querySelectorAll("th, td").forEach((el) => {
          el.style.whiteSpace = "nowrap";
        });
      }
    } catch (_) {}

    // 2) Toolbar (Import/Export) hinzufügen, falls sie im HTML fehlt
    const hasExportBtn = !!(byId("invAuditExportBtn") || legacyDlg.querySelector("#invAuditExportBtn"));
    if (!hasExportBtn) {
      const h3 = legacyDlg.querySelector("h3");
      const wrap = document.createElement("div");
      wrap.style.display = "flex";
      wrap.style.gap = "10px";
      wrap.style.flexWrap = "wrap";
      wrap.style.margin = "10px 0";

      const exportBtn = document.createElement("button");
      exportBtn.type = "button";
      exportBtn.id = "invAuditExportBtn";
      exportBtn.className = "btn btn-ghost";
      exportBtn.textContent = "Audit exportieren (JSON)";

      const label = document.createElement("label");
      label.className = "btn btn-ghost";
      label.style.cursor = "pointer";
      label.textContent = "Audit importieren (JSON)";

      const input = document.createElement("input");
      input.type = "file";
      input.id = "invAuditImportFile";
      input.accept = "application/json";
      input.style.display = "none";
      label.appendChild(input);

      wrap.appendChild(exportBtn);
      wrap.appendChild(label);

      // Toolbar direkt nach der Überschrift einsetzen
      if (h3 && h3.parentElement) {
        h3.insertAdjacentElement("afterend", wrap);
      } else {
        legacyDlg.prepend(wrap);
      }
    }

    // 3) Listener einmalig binden
    const exportBtn = byId("invAuditExportBtn");
    if (exportBtn && !exportBtn.dataset.wired) {
      exportBtn.dataset.wired = "1";
      exportBtn.addEventListener("click", () => {
        try {
          const blob = new Blob([JSON.stringify(state.audit || [], null, 2)], { type: "application/json" });
          const a = document.createElement("a");
          a.href = URL.createObjectURL(blob);
          a.download = "lager-audit-log.json";
          document.body.appendChild(a);
          a.click();
          a.remove();
          setTimeout(() => URL.revokeObjectURL(a.href), 250);
        } catch (e) {
                    uiAlert(
            "Audit-Export fehlgeschlagen: " + (e && e.message ? e.message : e),
            "Audit export failed: " + (e && e.message ? e.message : e),
            "\u274c Fehler",
            "\u274c Error",
            "OK",
            "OK"
          );
        }
      });
    }

    const importFile = byId("invAuditImportFile");
    if (importFile && !importFile.dataset.wired) {
      importFile.dataset.wired = "1";
      importFile.addEventListener("change", async () => {
        const file = importFile.files && importFile.files[0];
        if (!file) return;
        try {
          const text = await file.text();
          const list = JSON.parse(text);
          if (!Array.isArray(list)) throw new Error("JSON ist kein Array");
          state.audit = (state.audit || []).concat(list);
          saveAudit(state.audit);
          renderAuditTable();
                    uiAlert(
            "Audit importiert (angehängt).",
            "Audit imported (appended).",
            "\u2705 Fertig",
            "\u2705 Done",
            "OK",
            "OK"
          );
        } catch (e) {
                    uiAlert(
            "Audit-Import fehlgeschlagen: " + (e && e.message ? e.message : e),
            "Audit import failed: " + (e && e.message ? e.message : e),
            "\u274c Fehler",
            "\u274c Error",
            "OK",
            "OK"
          );
        } finally {
          importFile.value = "";
        }
      });
    }
  }

  function openAuditHistory() {
    // Wenn im HTML bereits eine Historie-Dialog-Struktur vorhanden ist, diese nutzen
    const legacyDlg = byId("auditHistoryDialog");
    if (legacyDlg) {
      wireLegacyAuditToolbar();
      renderAuditTable();
      openDialog(legacyDlg);
      return;
    }

    const dlg = ensureAuditDialog();
    renderAuditTable();
    openDialog(dlg);
  }





  function loadVisibleCols() {
    try {
      const raw = localStorage.getItem(STORAGE_VISIBLE_COLS);
      if (!raw) return null;
      const arr = JSON.parse(raw);
      return Array.isArray(arr) ? arr : null;
    } catch (e) {
      console.warn("[lager] sichtbare Spalten konnten nicht geladen werden:", e);
      return null;
    }
  }

  function saveVisibleCols(cols) {
    try {
      localStorage.setItem(STORAGE_VISIBLE_COLS, JSON.stringify(cols || []));
    } catch (e) {
      console.warn("[lager] sichtbare Spalten konnten nicht gespeichert werden:", e);
    }
  }
  
function loadColumnLabels() {
  try {
    const raw = localStorage.getItem(STORAGE_COL_LABELS);
    if (!raw) return null;
    const obj = JSON.parse(raw);
    if (!obj || typeof obj !== "object") return null;

    // BOM / führende Leerzeichen aus allen gespeicherten Labels entfernen
    const cleaned = {};
    for (const key in obj) {
      if (!Object.prototype.hasOwnProperty.call(obj, key)) continue;
      const val = obj[key];
      if (typeof val === "string") {
         cleaned[key] = stripBom(val).trim();
      }
    }
    return cleaned;
  } catch (e) {
    console.warn("[lager] Spaltennamen konnten nicht geladen werden:", e);
    return null;
  }
}

       


function saveColumnLabels(map) {
  try {
    localStorage.setItem(STORAGE_COL_LABELS, JSON.stringify(map || {}));
  } catch (e) {
    console.warn("[lager] Spaltennamen konnten nicht gespeichert werden:", e);
  }
}

// Liefert die sichtbare Bezeichnung für eine Spalte
function getColumnLabel(key) {
  const labels = state.columnLabels || {};
  const raw = labels && typeof labels[key] === "string" ? labels[key] : "";
  const val = stripBom(raw).trim();
  if (val) return val;
  const cfg = COLUMN_CONFIG.find(c => c.key === key);
  return cfg ? cfg.label : key;
}

// Setzt die Überschrift <th> anhand der columnLabels
function applyColumnLabels() {
  const table = byId("invTable");
  if (!table) return;
  COLUMN_CONFIG.forEach(col => {
    const th = table.querySelector('thead th[data-col="' + col.key + '"]');
    if (th) {
      th.textContent = getColumnLabel(col.key);
    }
  });
}


  // --- Filter & Zusammenfassung -----------------------------------------

  function getDistinctCategories() {
    const s = new Set();
    state.articles.forEach(a => {
      const k = (a.kategorie || "").trim();
      if (k) s.add(k);
    });
    return Array.from(s).sort((a, b) =>
      a.localeCompare(b, "de", { sensitivity: "base" })
    );
  }

  function renderCategoryFilterOptions(selectEl) {
    if (!selectEl) return;
    const selected = selectEl.value || "";
    const cats = getDistinctCategories();
    selectEl.innerHTML = "";

    const optAll = document.createElement("option");
    optAll.value = "";
    optAll.textContent = "Alle";
    selectEl.appendChild(optAll);

    cats.forEach((c) => {
      const o = document.createElement("option");
      o.value = c;
      o.textContent = c;
      if (c === selected) o.selected = true;
      selectEl.appendChild(o);
    });
  }
  // Alle verschiedenen Lager aus den Artikeln sammeln
  function getDistinctLager() {
    const s = new Set();
    state.articles.forEach(a => {
      const l = String(a.lager || "")
        .replace(/[\u200B-\u200D\uFEFF]/g, "")
        .replace(/\s+/g, " ")
        .trim();
      if (l) s.add(l);
    });
    return Array.from(s).sort((a, b) =>
      a.localeCompare(b, "de", { sensitivity: "base" })
    );
  }

  // Lager-Filter-Dropdown mit Werten füllen
  function renderLagerFilterOptions(selectEl) {
    if (!selectEl) return;
    const selected = selectEl.value || "";
    const lagerList = getDistinctLager();

    selectEl.innerHTML = "";

    const optAll = document.createElement("option");
    optAll.value = "";
    optAll.textContent = "Alle";
    selectEl.appendChild(optAll);

    lagerList.forEach(lager => {
      const opt = document.createElement("option");
      opt.value = lager;
      opt.textContent = lager;
      if (lager === selected) opt.selected = true;
      selectEl.appendChild(opt);
    });
  }

  function getFilteredArticles() {
    let list = state.articles.slice();

    const q = state.filters.search.trim().toLowerCase();
    if (q) {
      list = list.filter((a) => {
        const text = [
          a.artikelnummer,
          a.bezeichnung,
          a.hersteller,
          a.kategorie,
          a.lager,
          a.lagerart,
          a.lagerplatztyp,
          a.sicherheitsstufe,
          a.lagerort,
          a.einheit,
          a.notizen,
          a.technischerPlatz,
          a.equipment,
          a.anlage,
          a.waehrung,
          a.wareneingang,
          a.wertEinzeln != null ? String(a.wertEinzeln) : ""
        ].filter(Boolean).join(" ").toLowerCase();
        return text.indexOf(q) !== -1;
      });
    }

    if (state.filters.lager) {
      list = list.filter((a) => (a.lager || "") === state.filters.lager);
    }

    if (state.filters.kategorie) {
      list = list.filter((a) => (a.kategorie || "") === state.filters.kategorie);
    }

    list = list.filter((a) => {
      const best = toNumber(a.bestand, 0);
      const min  = toNumber(a.mindestbestand, 0);
      const st = String(state.filters.status || "").trim();
      switch (st) {
        case "low":
        case "unterMin":
          return min > 0 && best < min;
        case "gleichMin":
          return min > 0 && best === min;
        case "ok":
          return min > 0 && best >= min;
        case "ueberMin":
          return min > 0 && best > min;
        case "zero":
        case "null":
          return best <= 0;
        case "positiv":
          return best > 0;
        default:
          return true;
      }
    });

    // Sperr-Filter (Zugang/BANF gesperrt)
    const bf = String(state.filters.blocked || "").trim();
    if (bf === "blocked") {
      list = list.filter(a => isReceiptBlocked(a));
    } else if (bf === "unblocked") {
      list = list.filter(a => !isReceiptBlocked(a));
    }

    // Sortierung: Lager, Lagerort, Bezeichnung
    list.sort((a, b) => {
      const la = (a.lager || "").localeCompare(b.lager || "", "de", { sensitivity: "base" });
      if (la !== 0) return la;
      const lo = (a.lagerort || "").localeCompare(b.lagerort || "", "de", { sensitivity: "base" });
      if (lo !== 0) return lo;
      return (a.bezeichnung || "").localeCompare(b.bezeichnung || "", "de", { sensitivity: "base" });
    });

    return list;
  }

  function renderSummary(summaryCountEl, summaryLowEl, summaryZeroEl, summaryValueEl) {
    let total = state.articles.length;
    let low = 0;
    let zero = 0;
    let valueSum = 0;

    state.articles.forEach((a) => {
      const best = toNumber(a.bestand, 0);
      const min  = toNumber(a.mindestbestand, 0);
      if (best <= 0) zero++;
      if (min > 0 && best < min) low++;
      const wert = toNumber(a.wertEinzeln, 0);
      if (wert > 0 && best > 0) valueSum += best * wert;
    });

    if (summaryCountEl) summaryCountEl.textContent = String(total);
    if (summaryLowEl)   summaryLowEl.textContent   = String(low);
    if (summaryZeroEl)  summaryZeroEl.textContent  = String(zero);

    if (summaryValueEl) {
      if (valueSum <= 0) {
        summaryValueEl.textContent = "–";
      } else {
        try {
          summaryValueEl.textContent = valueSum.toLocaleString("de-DE", {
            style: "currency",
            currency: "EUR",
            maximumFractionDigits: 0
          });
        } catch {
          summaryValueEl.textContent = String(Math.round(valueSum)) + " €";
        }
      }
    }
  }

  function updateSelectAllCheckbox(selectAllEl, currentList) {
    if (!selectAllEl) return;
    if (!currentList.length) {
      selectAllEl.checked = false;
      selectAllEl.indeterminate = false;
      return;
    }
    let selected = 0;
    currentList.forEach(a => {
      if (state.selectedIds.has(a.id)) selected++;
    });
    if (selected === 0) {
      selectAllEl.checked = false;
      selectAllEl.indeterminate = false;
    } else if (selected === currentList.length) {
      selectAllEl.checked = true;
      selectAllEl.indeterminate = false;
    } else {
      selectAllEl.checked = false;
      selectAllEl.indeterminate = true;
    }
  }

  // --- Spaltenein-/ausblenden -------------------------------------------

  function applyColumnVisibility() {
    const table = byId("invTable");
    if (!table) return;

    const baseKeys = COLUMN_CONFIG.map(c => c.key);
    const visible = state.visibleCols && state.visibleCols.length
      ? new Set(state.visibleCols)
      : new Set(baseKeys); // Standard: alles sichtbar

    COLUMN_CONFIG.forEach(col => {
      const show = col.mandatory || visible.has(col.key);
      const display = show ? "" : "none";
      const selector =
        '#invTable thead th[data-col="' + col.key + '"],' +
        '#invTable tbody td[data-col="' + col.key + '"]';
      document.querySelectorAll(selector).forEach(el => {
        el.style.display = display;
      });
    });
  }

  function buildColumnsDialog(listEl) {
    if (!listEl) return;
    listEl.innerHTML = "";

    const baseKeys = COLUMN_CONFIG.map(c => c.key);
    const visible = state.visibleCols && state.visibleCols.length
      ? new Set(state.visibleCols)
      : new Set(baseKeys);

    COLUMN_CONFIG.forEach(col => {
      const row = document.createElement("div");
      row.className = "cols-list-item";

      const label = document.createElement("label");
      label.style.display = "flex";
      label.style.alignItems = "center";
      label.style.gap = "6px";

      const cb = document.createElement("input");
      cb.type = "checkbox";
      const mandatory = !!col.mandatory;
      cb.checked = mandatory || visible.has(col.key);
      cb.disabled = mandatory;

      cb.addEventListener("change", () => {
        toggleColumnVisibility(col.key, cb.checked);
      });

      const span = document.createElement("span");
      span.textContent = getColumnLabel(col.key);

      label.appendChild(cb);
      label.appendChild(span);
      row.appendChild(label);
      listEl.appendChild(row);
    });
  }

  function toggleColumnVisibility(key, on) {
    const def = COLUMN_CONFIG.find(c => c.key === key);
    if (!def || def.mandatory) return;

    const baseKeys = COLUMN_CONFIG.map(c => c.key);
    let current = state.visibleCols && state.visibleCols.length
      ? new Set(state.visibleCols)
      : new Set(baseKeys);

    if (on) current.add(key);
    else    current.delete(key);

    state.visibleCols = Array.from(current);
    saveVisibleCols(state.visibleCols);
    applyColumnVisibility();
  }

// --- Tabelle rendern --------------------------------------------------
function renderTable(tableBodyEl, summaryEls, selectAllEl) {
  if (!tableBodyEl) return;

  const list = getFilteredArticles();

  // Falls keine Artikel gefunden werden
  if (!list.length) {
    tableBodyEl.innerHTML =
      '<tr><td colspan="23" class="tiny">Keine Artikel gefunden – Filter/Suche anpassen oder einen neuen Artikel anlegen.</td></tr>';
    renderSummary(summaryEls.count, summaryEls.low, summaryEls.zero, summaryEls.value);
    updateSelectAllCheckbox(selectAllEl, list);
    applyColumnVisibility();
    return;
  }

  let html = "";

  list.forEach((a) => {
    const best = toNumber(a.bestand, 0);
    const min  = toNumber(a.mindestbestand, 0);
    const isLow  = min > 0 && best < min;
    const isZero = best <= 0;

    const blockedReceipt = isReceiptBlocked(a);
    const receiptReason = String(a.receiptBlockedReason || "").trim();
    const receiptSince  = a.receiptBlockedAt ? formatDateTime(a.receiptBlockedAt) : "";
    const receiptBy     = String(a.receiptBlockedBy || "").trim();
    const blockedTitle = blockedReceipt
      ? ("Gesperrt für Zugang/BANF" +
          (receiptReason ? ("\nGrund: " + receiptReason) : "") +
          (receiptSince  ? ("\nSeit: "  + receiptSince)  : "") +
          (receiptBy     ? ("\nVon: "   + receiptBy)     : ""))
      : "";
    const blockedBadge = blockedReceipt
      ? (' <span class="inv-badge-blocked" title="' + escapeHtml(blockedTitle) + '" style="cursor:help;">🔒 gesperrt</span>')
      : "";
    const bezeichnungHtml = escapeHtml(a.bezeichnung || "") + blockedBadge;


    // Hintergrund-Tönung je nach Sicherheitsstufe
    const sec = (a.sicherheitsstufe || "").toLowerCase();
    let rowStyle = "";
    if (sec === "gefahrstoff") {
      rowStyle = ' style="background:rgba(248,113,113,0.08);"';
    } else if (sec === "umweltgefährlich") {
      rowStyle = ' style="background:rgba(56,189,248,0.08);"';
    } else if (sec === "sicherheitsrelevant") {
      rowStyle = ' style="background:rgba(250,204,21,0.10);"';
    }

    // Ampel-Highlight nach Bestand/Mindestbestand (hat Vorrang vor Sicherheitsstufe)
    if (isZero) {
      rowStyle =
        ' style="background:linear-gradient(90deg, rgba(248,113,113,0.18), rgba(15,23,42,0.00));' +
        ' box-shadow: inset 0 0 0 1px rgba(248,113,113,0.22);' +
        ' border-left:3px solid rgba(248,113,113,0.75);"';
    } else if (isLow) {
      rowStyle =
        ' style="background:linear-gradient(90deg, rgba(250,204,21,0.18), rgba(15,23,42,0.00));' +
        ' box-shadow: inset 0 0 0 1px rgba(250,204,21,0.22);' +
        ' border-left:3px solid rgba(250,204,21,0.80);"';
    }

    let statusText = "";
    if (isZero) statusText = "❗ Bestand = 0";
    else if (isLow) statusText = "⚠️ unter Mindestbestand";

    const statusColor = isZero ? "#fecaca" : (isLow ? "#fde68a" : "#e5e7eb");

    const statusHtml = statusText
      ? '<div class="tiny" style="color:' + statusColor + ';">'+ escapeHtml(statusText) + "</div>"
      : "";

    const bestStyle = (isLow || isZero)
      ? ' style="color:' + statusColor + ';font-weight:600;"'
      : "";

    const checked = state.selectedIds.has(a.id) ? " checked" : "";

    // Preise (aus wertEinzeln berechnen, aber als einzelpreis/gesamtpreis anzeigen)
    const einzel = toNumber(a.wertEinzeln, 0);
    const gesamt = best * einzel;

    html +=
      '<tr data-id="' + a.id + '"' + rowStyle + '>' +

      // 1. Checkbox
      '<td><input type="checkbox" class="inv-row-select" data-id="' + a.id + '"' + checked + '></td>' +
	  
    // Aktionen
      '<td data-col="aktionen">' +
        '<div class="select-wrapper inv-action-select-wrapper">' +
          '<select class="inv-action-select" data-id="' + a.id + '">' +
            '<option value="">Aktionen …</option>' +
            '<option value="edit">✏️ Bearbeiten</option>' +
            (blockedReceipt
              ? '<option value="unblock">🔓 Sperre aufheben</option>'
              : '<option value="block">🔒 Für Zugang/BANF sperren</option>') +
            // Wareneingang ist ein eigener Prozess (Lieferschein-Pflicht). Zugang (Korrektur) bleibt separat.
            '<option value="goods-in"' + (blockedReceipt ? ' disabled' : '') + '>📥 Wareneingang</option>' +
            '<option value="book-in"' + (blockedReceipt ? ' disabled' : '') + '>➕ Zugang (Korrektur)</option>' +
            '<option value="goods-out">📤 Warenausgang</option>' +
            '<option value="book-out">➖ Entnahme</option>' +
            '<option value="inventory">📦 Inventur</option>' +
            '<option value="details">📄 Details</option>' +
            '<option value="banf"' + (blockedReceipt ? ' disabled' : '') + '>🧾 BANF / Bestellung</option>' +
          '</select>' +
        '</div>' +
      '</td>' +

      // 2–22: Spalten (Reihenfolge wie in deiner Überschrift/HTML)
      '<td data-col="artikelnummer">'        + escapeHtml(a.artikelnummer || "")         + '</td>' +
      '<td data-col="interneArtikelnummer">' + escapeHtml(a.interneArtikelnummer || "")  + '</td>' +
      '<td data-col="bezeichnung">'          + bezeichnungHtml                            + '</td>' +
      '<td data-col="hersteller">'           + escapeHtml(a.hersteller || "")            + '</td>' +
      '<td data-col="lieferant">'            + escapeHtml(a.lieferant || "")             + '</td>' +
      '<td data-col="kategorie">'            + escapeHtml(a.kategorie || "")             + '</td>' +
      '<td data-col="lager">'                + escapeHtml(a.lager || "")                 + '</td>' +
      '<td data-col="mindestbestand">'       + min                                        + '</td>' +
      '<td data-col="bestand"' + bestStyle + '>' + best + statusHtml                      + '</td>' +
      '<td data-col="einheit">'              + escapeHtml(a.einheit || "")                + '</td>' +

      '<td data-col="einzelpreis" style="text-align:right;">' + (einzel ? einzel.toFixed(2) : "") + '</td>' +
      '<td data-col="gesamtpreis" style="text-align:right;">' + (gesamt ? gesamt.toFixed(2) : "") + '</td>' +

      '<td data-col="notizen">'              + escapeHtml(a.notizen || "")                + '</td>' +
      '<td data-col="technischerPlatz">'     + escapeHtml(a.technischerPlatz || "")       + '</td>' +
      '<td data-col="equipment">'            + escapeHtml(a.equipment || "")              + '</td>' +
      '<td data-col="anlage">'               + escapeHtml(a.anlage || "")                 + '</td>' +
      '<td data-col="regal">'                + escapeHtml(a.regal || "")                  + '</td>' +
      '<td data-col="fach">'                 + escapeHtml(a.fach || "")                   + '</td>' +
      '<td data-col="position">'             + escapeHtml(a.position || "")               + '</td>' +
      '<td data-col="niveau">'               + escapeHtml(a.niveau || "")                 + '</td>' +
      '<td data-col="lagerort">'             + escapeHtml(a.lagerort || "")               + '</td>' +

  

      '</tr>';
  });

  tableBodyEl.innerHTML = html;

  // (falls vorhanden)
  if (typeof enhanceActionMenus === "function") {
    enhanceActionMenus(tableBodyEl);
  }

  renderSummary(summaryEls.count, summaryEls.low, summaryEls.zero, summaryEls.value);
  updateSelectAllCheckbox(selectAllEl, list);
  applyColumnVisibility();
}


  

  // --- Soft-Sperre (SAP-ähnlich): Zugang/BANF sperren, Entnahme erlaubt ----
  function ensureInlineCss(id, cssText) {
    if (document.getElementById(id)) return;
    const style = document.createElement("style");
    style.id = id;
    style.textContent = cssText;
    document.head.appendChild(style);
  }

  function ensureReceiptBlockStyles() {
    ensureInlineCss("inv-receipt-block-css", `
      .inv-badge-blocked{
        display:inline-flex;
        align-items:center;
        gap:6px;
        font-size:12px;
        line-height:1;
        padding:2px 8px;
        border-radius:999px;
        margin-left:6px;
        background:rgba(148,163,184,0.14);
        border:1px solid rgba(148,163,184,0.22);
        color:rgba(226,232,240,0.95);
        white-space:nowrap;
      }
    `);
  }

  function currentEmployeeName() {
    const el = document.getElementById("invBookingEmployee");
    return el ? String(el.value || "").trim() : "";
  }

  function promptRequired(message, defaultValue) {
    while (true) {
      const val = prompt(message, defaultValue || "");
      if (val === null) return null;
      const t = String(val).trim();
      if (t) return t;
            uiAlert(
        "Bitte einen Grund eingeben (Pflichtfeld).",
        "Please enter a reason (required).",
        "\u26a0\ufe0f Hinweis",
        "\u26a0\ufe0f Note",
        "OK",
        "OK"
      );
    }
  }

  function refreshUI() {
    const body =
      byId("invTableBody") ||
      (byId("invTable") ? byId("invTable").querySelector("tbody") : null);
    const selectAllEl = byId("invSelectAll");
    const summaryEls = {
      count: byId("invSummaryCount"),
      low:   byId("invSummaryLow"),
      zero:  byId("invSummaryZero"),
      value: byId("invSummaryValue")
    };
    renderTable(body, summaryEls, selectAllEl);
  }

  function isReceiptBlocked(article) {
    return !!(article && article.receiptBlocked);
  }

  // Modern (Dialog) – Sperren/Entsperren mit Pflicht-Grund, ohne prompt().
  function blockReceiptsDialog(article) {
    if (!article) return;
    if (isReceiptBlocked(article)) {
            uiAlert(
        "Dieser Artikel ist bereits für Zugang/BANF gesperrt.",
        "This item is already blocked for receiving/requisition.",
        "\ud83d\udd12 Sperre",
        "\ud83d\udd12 Block",
        "OK",
        "OK"
      );
      return;
    }
    const ok = openReasonDialogRequired({
      title: tLangSafe("🔒 Artikel sperren", "🔒 Block item"),
      intro: tLangSafe(
        "➡️ Entnahme bleibt erlaubt (Restbestand kann verbraucht werden).\n\nZugang und BANF/Bestellung werden blockiert.",
        "➡️ Withdrawal remains allowed (existing stock can still be used).\n\nReceiving and requisitions/orders will be blocked."
      ),
      okText: tLangSafe("Sperren", "Block"),
      onOk: (reason) => {
        const now = nowIso();
        article.receiptBlocked = true;
        article.receiptBlockedReason = reason;
        article.receiptBlockedAt = now;
        article.receiptBlockedBy = currentEmployeeName();
        article.updatedAt = now;

        saveArticles(state.articles);

        addAuditEvent("article.blocked_receipt", {
          articleId: article.id,
          artikelnummer: article.artikelnummer || "",
          bezeichnung: article.bezeichnung || "",
          reason,
          employee: article.receiptBlockedBy || ""
        });

        refreshUI();
      }
    });
    if (!ok) blockReceipts(article); // Fallback
  }

  function unblockReceiptsDialog(article) {
    if (!article) return;
    if (!isReceiptBlocked(article)) {
            uiAlert(
        "Dieser Artikel ist nicht gesperrt.",
        "This item is not blocked.",
        "\u2139\ufe0f Hinweis",
        "\u2139\ufe0f Info",
        "OK",
        "OK"
      );
      return;
    }
    const prevReason = String(article.receiptBlockedReason || "").trim();
    const prevBy = String(article.receiptBlockedBy || "").trim();

    const ok = openReasonDialogRequired({
      title: "🔓 Sperre aufheben",
      intro:
        "Der Artikel kann danach wieder Zugänge erhalten und per BANF bestellt werden.\n\n" +
        "Grund (Pflicht):",
      okText: "Entsperren",
      onOk: (reason) => {
        const now = nowIso();

        article.receiptBlocked = false;
        article.receiptBlockedReason = "";
        article.receiptBlockedAt = "";
        article.receiptBlockedBy = "";
        article.receiptUnblockedReason = reason;
        article.receiptUnblockedAt = now;
        article.receiptUnblockedBy = currentEmployeeName();
        article.updatedAt = now;

        saveArticles(state.articles);

        addAuditEvent("article.unblocked_receipt", {
          articleId: article.id,
          artikelnummer: article.artikelnummer || "",
          bezeichnung: article.bezeichnung || "",
          prevReason,
          prevBy,
          reason,
          employee: currentEmployeeName()
        });

        refreshUI();
      }
    });
    if (!ok) unblockReceipts(article); // Fallback (prompt/confirm)
  }

  function blockReceipts(article) {
    if (!article) return;
    if (isReceiptBlocked(article)) {
            uiAlert(
        "Dieser Artikel ist bereits für Zugang/BANF gesperrt.",
        "This item is already blocked for receiving/requisition.",
        "\ud83d\udd12 Sperre",
        "\ud83d\udd12 Block",
        "OK",
        "OK"
      );
      return;
    }

    const reason = promptRequired(
      "Artikel für Zugang/BANF sperren.\n\n" +
        "➡️ Entnahme bleibt erlaubt (Restbestand kann verbraucht werden).\n\n" +
        "Grund (Pflicht):"
    );
    if (reason === null) return;

    const now = nowIso();
    article.receiptBlocked = true;
    article.receiptBlockedReason = reason;
    article.receiptBlockedAt = now;
    article.receiptBlockedBy = currentEmployeeName();
    article.updatedAt = now;

    saveArticles(state.articles);

    addAuditEvent("article.blocked_receipt", {
      articleId: article.id,
      artikelnummer: article.artikelnummer || "",
      bezeichnung: article.bezeichnung || "",
      reason,
      employee: article.receiptBlockedBy || ""
    });

    refreshUI();
  }

  async function unblockReceipts(article) {
    if (!article) return;
    if (!isReceiptBlocked(article)) {
            uiAlert(
        "Dieser Artikel ist nicht gesperrt.",
        "This item is not blocked.",
        "\u2139\ufe0f Hinweis",
        "\u2139\ufe0f Info",
        "OK",
        "OK"
      );
      return;
    }

    const ok = await uiConfirm(
      "Sperre für Zugang/BANF aufheben?\n\nDer Artikel kann danach wieder Zugänge erhalten und per BANF bestellt werden.",
      "Unblock receiving/requisitions?\n\nAfterwards, the item can receive stock again and be ordered via requisition.",
      "🔓 Sperre aufheben",
      "🔓 Unblock",
      "OK",
      "OK",
      "Abbrechen",
      "Cancel"
    );
    if (!ok) return;

    const now = nowIso();
    const prevReason = String(article.receiptBlockedReason || "").trim();
    const prevBy = String(article.receiptBlockedBy || "").trim();

    article.receiptBlocked = false;
    article.receiptBlockedReason = "";
    article.receiptBlockedAt = "";
    article.receiptBlockedBy = "";
    article.receiptUnblockedReason = reason;
    article.receiptUnblockedAt = now;
    article.receiptUnblockedBy = currentEmployeeName();
    article.updatedAt = now;

    saveArticles(state.articles);

    addAuditEvent("article.unblocked_receipt", {
      articleId: article.id,
      artikelnummer: article.artikelnummer || "",
      bezeichnung: article.bezeichnung || "",
      prevReason,
      prevBy,
      reason,
      employee: currentEmployeeName()
    });

    refreshUI();
  }
// --- Artikel-Formular -------------------------------------------------

  function buildArticleFromForm(formEls, existing) {
    const now = nowIso();
    const id = existing && existing.id ? existing.id : ("a" + Math.random().toString(36).slice(2));

    return {
      id: id,
      artikelnummer: (formEls.artikelnummer.value || "").trim(),
	  interneArtikelnummer: (formEls.interneArtikelnummer?.value || "").trim(),
      bezeichnung: (formEls.bezeichnung.value || "").trim(),
      hersteller: (formEls.hersteller.value || "").trim(),
	  lieferant: (formEls.lieferant?.value || "").trim(),
      kategorie: (formEls.kategorie.value || "").trim(),
      lager: (formEls.lager.value || "").trim(),
      lagerart: (formEls.lagerart.value || "").trim(),
	  regal: (formEls.regal.value || "").trim(),
      fach: (formEls.fach.value || "").trim(),
      position: (formEls.position.value || "").trim(),
      niveau: (formEls.niveau.value || "").trim(),
      lagerplatztyp: (formEls.lagerplatztyp.value || "").trim(),
      sicherheitsstufe: (formEls.sicherheitsstufe.value || "").trim(),
      lagerort: (formEls.lagerort.value || "").trim(),
      mindestbestand: toNumber(formEls.mindestbestand.value, 0),
      bestand: toNumber(formEls.bestand.value, 0),
      einheit: (formEls.einheit.value || "").trim(),
      notizen: (formEls.notizen.value || "").trim(),
      technischerPlatz: (formEls.technPlatz.value || "").trim(),
      equipment: (formEls.equipment.value || "").trim(),
      anlage: (formEls.anlage.value || "").trim(),
      wertEinzeln: toNumber(formEls.wertEinzeln ? formEls.wertEinzeln.value : 0, 0),
      waehrung: formEls.waehrung ? (formEls.waehrung.value || "").trim() : "",
      wareneingang: formEls.wareneingang ? (formEls.wareneingang.value || "").trim() :
                    (existing && existing.wareneingang) || "",
      // 📎 Anlagen (z.B. Lieferscheine) – beim Bearbeiten beibehalten
      attachments: existing && Array.isArray(existing.attachments) ? existing.attachments : [],

      // 🔒 Sperrstatus / Archiv – beim Bearbeiten beibehalten
      receiptBlocked: existing ? !!existing.receiptBlocked : false,
      receiptBlockedReason: existing ? (existing.receiptBlockedReason || "") : "",
      receiptBlockedAt: existing ? (existing.receiptBlockedAt || "") : "",
      receiptBlockedBy: existing ? (existing.receiptBlockedBy || "") : "",
      receiptUnblockedReason: existing ? (existing.receiptUnblockedReason || "") : "",
      receiptUnblockedAt: existing ? (existing.receiptUnblockedAt || "") : "",
      receiptUnblockedBy: existing ? (existing.receiptUnblockedBy || "") : "",
      archived: existing ? !!existing.archived : false,
      archivedReason: existing ? (existing.archivedReason || "") : "",
      archivedAt: existing ? (existing.archivedAt || "") : "",
      archivedBy: existing ? (existing.archivedBy || "") : "",

      createdAt: existing && existing.createdAt ? existing.createdAt : now,
      updatedAt: now,
      lastBooking: existing && existing.lastBooking ? existing.lastBooking : null
    };
  }


	function fillArticleForm(formEls, article, metaEls) {
	if (!article) {
      formEls.id.value = "";
      formEls.artikelnummer.value = "";
      if (formEls.interneArtikelnummer) formEls.interneArtikelnummer.value = "";
      if (formEls.lieferant) formEls.lieferant.value = "";
      formEls.bezeichnung.value = "";
      formEls.hersteller.value = "";
      formEls.kategorie.value = "";
      formEls.lager.value = "";
      formEls.lagerart.value = "";
      formEls.lagerplatztyp.value = "";
      formEls.sicherheitsstufe.value = "";
      formEls.lagerort.value = "";
      formEls.mindestbestand.value = "0";
      formEls.bestand.value = "0";
      formEls.einheit.value = "";
      formEls.notizen.value = "";
      formEls.technPlatz.value = "";
      formEls.equipment.value = "";
      formEls.anlage.value = "";
      // Regalfelder leeren
      if (formEls.regal)    formEls.regal.value = "";
      if (formEls.fach)     formEls.fach.value = "";
      if (formEls.position) formEls.position.value = "";
      if (formEls.niveau)   formEls.niveau.value = "";
      if (formEls.wertEinzeln) formEls.wertEinzeln.value = "";
      if (formEls.waehrung)   formEls.waehrung.value   = "";
      if (formEls.wareneingang) formEls.wareneingang.value = "";
      if (metaEls && metaEls.meta)    metaEls.meta.textContent    = "";
      if (metaEls && metaEls.booking) metaEls.booking.textContent = "";
      return;
    }

    formEls.id.value = article.id;
    formEls.artikelnummer.value = article.artikelnummer || "";
    if (formEls.interneArtikelnummer) formEls.interneArtikelnummer.value = article.interneArtikelnummer || "";
    if (formEls.lieferant) formEls.lieferant.value = article.lieferant || "";
    formEls.bezeichnung.value   = article.bezeichnung   || "";
    formEls.hersteller.value    = article.hersteller    || "";
    formEls.kategorie.value     = article.kategorie     || "";
    formEls.lager.value         = article.lager         || "";
    formEls.lagerart.value      = article.lagerart      || "";
    formEls.lagerplatztyp.value = article.lagerplatztyp || "";
    formEls.sicherheitsstufe.value = article.sicherheitsstufe || "";
    formEls.lagerort.value      = article.lagerort      || "";
    // Regalfelder setzen
    if (formEls.regal)    formEls.regal.value    = article.regal    || "";
    if (formEls.fach)     formEls.fach.value     = article.fach     || "";
    if (formEls.position) formEls.position.value = article.position || "";
    if (formEls.niveau)   formEls.niveau.value   = article.niveau   || "";
    formEls.lagerort.value      = article.lagerort      || "";
    formEls.mindestbestand.value = String(article.mindestbestand != null ? article.mindestbestand : 0);
    formEls.bestand.value        = String(article.bestand != null ? article.bestand : 0);
    formEls.einheit.value        = article.einheit || "";
    formEls.notizen.value        = article.notizen || "";
    formEls.technPlatz.value     = article.technischerPlatz || "";
    formEls.equipment.value      = article.equipment || "";
    formEls.anlage.value         = article.anlage || "";
    if (formEls.wertEinzeln)     formEls.wertEinzeln.value = article.wertEinzeln ? String(article.wertEinzeln) : "";
    if (formEls.waehrung)        formEls.waehrung.value    = article.waehrung || "";
    if (formEls.wareneingang)    formEls.wareneingang.value = article.wareneingang || "";

    if (metaEls && metaEls.meta) {
      metaEls.meta.textContent =
        "Erstellt: " + (formatDateTime(article.createdAt) || "–") +
        " · Zuletzt geändert: " + (formatDateTime(article.updatedAt) || "–");
    }

    if (metaEls && metaEls.booking) {
      const lb = article.lastBooking;
      if (lb && lb.ts) {
        const dirText    = lb.direction === "in" ? "Zugang" : "Entnahme";
        const menge      = lb.amount != null ? lb.amount : "?";
        const einheit    = article.einheit || "";
        const kostenst   = lb.kostenstelle || "–";
        const mitarbeiter= lb.mitarbeiter || "–";
        metaEls.booking.textContent =
          "Letzte Buchung: " + formatDateTime(lb.ts) +
          " · " + dirText + " " + menge + " " + einheit +
          " · Kostenstelle: " + kostenst +
          " · Mitarbeiter: " + mitarbeiter;
      } else {
        metaEls.booking.textContent = "Noch keine Buchung erfasst.";
      }
    }
  }

  // --- Buchungen --------------------------------------------------------

  function addBooking(article, params, bookingsArr) {
    const before = toNumber(article.bestand, 0);
    const after  = params.newBestand;
    const now    = nowIso();

    const booking = {
      id: "b" + Math.random().toString(36).slice(2),
      ts: now,
      articleId: article.id,
      artikelnummer: article.artikelnummer,
      bezeichnung: article.bezeichnung,
      lager: article.lager,
      lagerort: article.lagerort,
      direction: params.direction, // "in" | "out"
      amount: params.amount,
      reason: params.reason || "",
      note: params.note || "",
      kostenstelle: params.kostenstelle || "",
      mitarbeiter: params.mitarbeiter || "",
      bestandVorher: before,
      bestandNachher: after,
      supplier: params.supplier || "",
      deliveryNoteNo: params.deliveryNoteNo || "",
      deliveryNoteDate: params.deliveryNoteDate || "",
      attachments: Array.isArray(params.attachments) ? params.attachments : []
    };

    bookingsArr.push(booking);

    const updatedArticle = Object.assign({}, article, {
      bestand: after,
      updatedAt: now,
      lastBooking: {
        ts: now,
        direction: params.direction,
        amount: params.amount,
        reason: params.reason || "",
        note: params.note || "",
        kostenstelle: params.kostenstelle || "",
        mitarbeiter: params.mitarbeiter || "",
        supplier: params.supplier || "",
        deliveryNoteNo: params.deliveryNoteNo || "",
        deliveryNoteDate: params.deliveryNoteDate || "",
        attachments: Array.isArray(params.attachments) ? params.attachments : []
      }
    });

    return { updatedArticle, booking };
  }

  // --- CSV Export -------------------------------------------------------

  function exportCsv() {
    const list = state.articles;
    if (!list.length) {
            uiAlert(
        "Es sind noch keine Artikel angelegt.",
        "No items have been created yet.",
        "\u2139\ufe0f Hinweis",
        "\u2139\ufe0f Info",
        "OK",
        "OK"
      );
      return;
    }

const header = [
  getColumnLabel("artikelnummer"),
  getColumnLabel("bezeichnung"),
  getColumnLabel("hersteller"),
  getColumnLabel("kategorie"),
  getColumnLabel("lager"),
  "Lagerart",
  "Lagerplatztyp",
  "Sicherheitsstufe",
  getColumnLabel("lagerort"),
  getColumnLabel("mindestbestand"),
  getColumnLabel("bestand"),
  getColumnLabel("einheit"),
  getColumnLabel("notizen"),
  "Technischer Platz",
  "Equipment",
  "Anlage",
  getColumnLabel("wertEinzeln"),
  getColumnLabel("waehrung"),
  "Wareneingang",
  getColumnLabel("createdAt"),
  getColumnLabel("updatedAt")
];



    function csvVal(v) {
      if (v === null || v === undefined) return "";
      let s = String(v);
      if (s.indexOf('"') !== -1) s = s.replace(/"/g, '""');
      if (s.indexOf(";") !== -1) s = '"' + s + '"';
      return s;
    }

    const lines = [header.join(";")];

    list.forEach((a) => {
      const row = [
        a.artikelnummer,
        a.bezeichnung,
        a.hersteller,
        a.kategorie,
        a.lager,
        a.lagerart,
        a.lagerplatztyp,
        a.sicherheitsstufe,
        a.lagerort,
        String(a.mindestbestand),
        String(a.bestand),
        a.einheit,	
        a.notizen,
        a.technischerPlatz,
        a.equipment,
        a.anlage,
        a.wertEinzeln != null ? String(a.wertEinzeln) : "",
        a.waehrung || "",
        a.wareneingang || "",
        a.createdAt || "",
        a.updatedAt || ""
      ];
      lines.push(row.map(csvVal).join(";"));
    });

    const csv = lines.join("\n");
    const csvWithBom = "\uFEFF" + csv; // BOM für Excel & Umlaute

    try {
      const blob = new Blob([csvWithBom], { type: "text/csv;charset=utf-8;" });
      const url = URL.createObjectURL(blob);
      const a = document.createElement("a");
      const d = new Date();
      const dateStr = d.getFullYear() + "-" + pad2(d.getMonth() + 1) + "-" + pad2(d.getDate());
      a.href = url;
      a.download = "lagerbestand-" + dateStr + ".csv";
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
    } catch (e) {
      console.warn("[lager] CSV-Export fehlgeschlagen:", e);
    }
  }

  // --- CSV Import -------------------------------------------------------

function normalizeHeaderName(name) {
  let s = stripBom(name).trim().toLowerCase();
  s = s
    .replace(/ä/g, "ae")
      .replace(/ö/g, "oe")
      .replace(/ü/g, "ue")
      .replace(/ß/g, "ss");
    s = s.replace(/[^a-z0-9]/g, "");
    return s;
  }

const HEADER_MAP = {
  // Pflicht / Basis
  artikelnummer: "artikelnummer",
  artikelnr: "artikelnummer",
  artnr: "artikelnummer",

  interneartikelnr: "interneArtikelnummer",
  interneartikelnummer: "interneArtikelnummer",

  bezeichnung: "bezeichnung",
  kurztext: "bezeichnung",

  hersteller: "hersteller",
  lieferant: "lieferant",

  kategorie: "kategorie",
  lager: "lager",

  mindestbestand: "mindestbestand",
  mindestbestaende: "mindestbestand",

  bestand: "bestand",
  einheit: "einheit",

  // Preise: deine Tabelle heißt „Einzelpreis/Gesamtpreis“,
  // intern nutzt du aber wertEinzeln + (gesamt wird berechnet)
  einzelpreis: "wertEinzeln",
  wertproeinheit: "wertEinzeln",
  wert: "wertEinzeln",

  gesamtpreis: "gesamtpreis", // optional (kannst du beim Import ignorieren/überschreiben)

  notizen: "notizen",
  bemerkung: "notizen",

  technischerplatz: "technischerPlatz",
  technischerplatz: "technischerPlatz", // falls doppelt ok
  technischerplatz: "technischerPlatz",

  equipment: "equipment",
  anlage: "anlage",

  regal: "regal",
  fach: "fach",

  // dein Normalizer macht aus "Position / Fachnummer" -> "positionfachnummer"
  positionfachnummer: "position",
  position: "position",

  niveau: "niveau",
  lagerort: "lagerort",

  // optional falls vorhanden
  waehrung: "waehrung",
  wareneingang: "wareneingang",
  erstelltam: "createdAt",
  geaendertam: "updatedAt"
};



  // CSV Parser, der Semikolon/Komma/Tab/Pipe automatisch erkennt
  // - unterstützt Quotes ("...") inkl. Zeilenumbrüchen im Feld
  // - unterstützt escaped Quotes ("" -> ")
  function parseCsvFlexible(text) {
    const t = stripBom(String(text || ""));

    function detectDelimiter(sampleText) {
      const candidates = [";", ",", "\t", "|"];
      const lines = sampleText
        .split(/\r?\n/)
        .map(l => l.trim())
        .filter(l => l !== "")
        .slice(0, 25);
      const joined = lines.join("\n");

      // Zähle Trennzeichen nur außerhalb von Quotes
      function countOutsideQuotes(s, ch) {
        let inQ = false;
        let cnt = 0;
        for (let i = 0; i < s.length; i++) {
          const c = s[i];
          if (c === '"') {
            const next = s[i + 1];
            if (inQ && next === '"') {
              i++; // escaped quote
            } else {
              inQ = !inQ;
            }
          } else if (!inQ && c === ch) {
            cnt++;
          }
        }
        return cnt;
      }

      let best = ";";
      let bestCount = -1;
      for (const ch of candidates) {
        const c = countOutsideQuotes(joined, ch);
        if (c > bestCount) {
          bestCount = c;
          best = ch;
        }
      }
      return best;
    }

    const delimiter = detectDelimiter(t);
    const rows = [];
    let row = [];
    let field = "";
    let inQuotes = false;

    for (let i = 0; i < t.length; i++) {
      const c = t[i];

      if (inQuotes) {
        if (c === '"') {
          const next = t[i + 1];
          if (next === '"') {
            field += '"';
            i++;
          } else {
            inQuotes = false;
          }
        } else {
          field += c;
        }
      } else {
        if (c === '"') {
          inQuotes = true;
        } else if (c === delimiter) {
          row.push(field);
          field = "";
        } else if (c === "\r") {
          // ignore
        } else if (c === "\n") {
          row.push(field);
          rows.push(row);
          row = [];
          field = "";
        } else {
          field += c;
        }
      }
    }

    if (field.length > 0 || row.length) {
      row.push(field);
      rows.push(row);
    }

    // leere Zeilen am Ende raus
    return rows.filter(r => r.some(cell => (cell || "").trim() !== ""));
  }

  // Backward-compat: alter Name (wird intern weiter genutzt)
  function parseCsvSemicolon(text) {
    return parseCsvFlexible(text);
  }

  async function importCsvText(text) {
    const rows = parseCsvFlexible(text);
    if (!rows.length) {
            uiAlert(
        "CSV-Datei enthält keine Daten.",
        "The CSV file contains no data.",
        "\u274c Fehler",
        "\u274c Error",
        "OK",
        "OK"
      );
      return;
    }

    // Standard: erste Zeile ist Header
    const headerRow = rows[0];
    let startRow = 1;

    const fieldIndex = {};

    // Bisherige Labels holen (oder leeres Objekt)
    const currentLabels = Object.assign({}, state.columnLabels || {});

    let hasHeaderMatch = false;

    // 1) Versuch: bekannte Überschriften per HEADER_MAP zuordnen
    headerRow.forEach((h, idx) => {
      const norm = normalizeHeaderName(h);
      const mapped = HEADER_MAP[norm];
      if (mapped && fieldIndex[mapped] === undefined) {
        fieldIndex[mapped] = idx;
        hasHeaderMatch = true;

        const cleanLabel = stripBom(h).trim();
        if (cleanLabel) currentLabels[mapped] = cleanLabel;
      }
    });

    // 2) Fallback: interne Artikelnummer als Artikelnummer nutzen, wenn nötig
    if (fieldIndex.artikelnummer === undefined && fieldIndex.interneArtikelnummer !== undefined) {
      fieldIndex.artikelnummer = fieldIndex.interneArtikelnummer;
      const h = stripBom(headerRow[fieldIndex.interneArtikelnummer]).trim();
      if (h) currentLabels.artikelnummer = h;
    }

    // 3) Wenn KEIN Header-Match möglich war, versuchen wir "ohne Header" zu importieren:
    //    Spalte 0 = Artikelnummer, Spalte 1 = Bezeichnung (wenn vorhanden)
    if (!hasHeaderMatch &&
        fieldIndex.artikelnummer === undefined &&
        fieldIndex.bezeichnung === undefined) {

      const colCount = Array.isArray(rows[0]) ? rows[0].length : 0;
      if (colCount >= 1) fieldIndex.artikelnummer = 0;
      if (colCount >= 2) fieldIndex.bezeichnung = 1;

      // Dann ist Zeile 0 Daten, nicht Header
      startRow = 0;
    }

    // 4) Labels im State / LocalStorage aktualisieren
    state.columnLabels = currentLabels;
    saveColumnLabels(currentLabels);
    applyColumnLabels();

    // 5) Wenn selbst nach Fallback weder Artikelnummer noch Bezeichnung zugeordnet werden konnten, abbrechen
    if (fieldIndex.artikelnummer === undefined && fieldIndex.bezeichnung === undefined) {
            uiAlert(
        "CSV-Kopfzeile konnte nicht zugeordnet werden.\n\n" +
                "Bitte entweder bekannte Spaltennamen verwenden (z.B. Artikelnummer, Bezeichnung)\n" +
                "oder ohne Kopfzeile importieren (Spalte 1 = Artikelnummer, Spalte 2 = Bezeichnung).",
        "CSV header could not be mapped.\n\n" +
                "Please use known column names (e.g. ItemNumber, Description)\n" +
                "or import without a header (Column 1 = item number, Column 2 = description).",
        "\u274c Fehler",
        "\u274c Error",
        "OK",
        "OK"
      );
      return;
    }

    const now = nowIso();
    const newArticles = [];
    const updateByArtNr = new Map();

    function getField(row, fieldName) {
      const idx = fieldIndex[fieldName];
      if (idx === undefined) return "";
      return (row[idx] || "").trim();
    }

    for (let i = startRow; i < rows.length; i++) {
      const row = rows[i];
      if (!row || !row.length) continue;

      const artikelnummer = getField(row, "artikelnummer");
      const bezeichnung   = getField(row, "bezeichnung");
      if (!artikelnummer && !bezeichnung) continue;

      const csvObj = {
        artikelnummer,
        interneArtikelnummer: getField(row, "interneArtikelnummer"),
        bezeichnung,
        hersteller: getField(row, "hersteller"),
        lieferant:  getField(row, "lieferant"),
        kategorie:  getField(row, "kategorie"),
        lager:      getField(row, "lager"),
        lagerart:   getField(row, "lagerart"),
        lagerplatztyp:    getField(row, "lagerplatztyp"),
        sicherheitsstufe: getField(row, "sicherheitsstufe"),
        lagerort:   getField(row, "lagerort"),
        mindestbestand: toNumber(getField(row, "mindestbestand"), 0),
        bestand:        toNumber(getField(row, "bestand"), 0),
        einheit:   getField(row, "einheit"),
        notizen:   getField(row, "notizen"),
        technischerPlatz: getField(row, "technischerPlatz"),
        equipment: getField(row, "equipment"),
        anlage:    getField(row, "anlage"),
        regal:     getField(row, "regal"),
        fach:      getField(row, "fach"),
        position:  getField(row, "position"),
        niveau:    getField(row, "niveau"),
        wertEinzeln: toNumber(getField(row, "wertEinzeln"), 0),
        waehrung:    getField(row, "waehrung"),
        wareneingang: getField(row, "wareneingang")
      };

      const createdAt = getField(row, "createdAt") || now;
      const updatedAt = getField(row, "updatedAt") || now;

      const existing = state.articles.find(a =>
        a.artikelnummer && a.artikelnummer === artikelnummer
      );

      if (existing) {
        updateByArtNr.set(existing.artikelnummer, Object.assign({}, csvObj, {
          createdAt: existing.createdAt || createdAt,
          updatedAt: updatedAt
        }));
      } else {
        newArticles.push({
          id: "a" + Math.random().toString(36).slice(2),
          ...csvObj,
          createdAt,
          updatedAt,
          lastBooking: null
        });
      }
    }

    const newCount      = newArticles.length;
    const existingCount = updateByArtNr.size;

    let overwriteExisting = true;
    if (existingCount > 0) {
      overwriteExisting = await uiConfirm(
        "CSV-Import:\n\n" +
          "Neue Artikel: " + newCount + "\n" +
          "Vorhandene Artikel (gleiche Artikelnummer): " + existingCount + "\n\n" +
          "OK = vorhandene Artikel mit CSV-Daten aktualisieren\n" +
          "Abbrechen = vorhandene Artikel behalten (nur neue hinzufügen)",
        "CSV import:\n\n" +
          "New items: " + newCount + "\n" +
          "Existing items (same article number): " + existingCount + "\n\n" +
          "OK = update existing items with CSV data\n" +
          "Cancel = keep existing items (add new only)",
        "CSV-Import",
        "CSV import",
        "OK",
        "OK",
        "Abbrechen",
        "Cancel"
      );
}

    if (overwriteExisting) {
      const nowUpdate = nowIso();
      state.articles = state.articles.map((a) => {
        const upd = a.artikelnummer ? updateByArtNr.get(a.artikelnummer) : null;
        if (!upd) return a;
        return {
          ...a,
          ...upd,
          createdAt: a.createdAt || upd.createdAt || nowUpdate,
          updatedAt: nowUpdate
        };
      });
    }

    if (newArticles.length) {
      state.articles.push(...newArticles);
    }

        uiAlert(
      "CSV-Import abgeschlossen:\n\n" +
            "Neu hinzugefügt: " + newArticles.length + "\n" +
            "Aktualisiert: " + (overwriteExisting ? existingCount : 0),
      "CSV import completed:\n\n" +
            "Added: " + newArticles.length + "\n" +
            "Updated: " + (overwriteExisting ? existingCount : 0),
      "\u2705 Fertig",
      "\u2705 Done",
      "OK",
      "OK"
    );
  }

  function handleCsvImport(file) {
    if (!file) return;

    const reader = new FileReader();

    // Moderne Browser: erst UTF-8, bei kaputten Umlauten auf windows-1252 wechseln
    if (window.TextDecoder) {
      reader.onload = async (ev) => {
        const buffer = ev.target.result;
        let textUtf8 = "";
        let text1252 = "";
        let finalText = "";

        try {
          const decUtf8 = new TextDecoder("utf-8", { fatal: false });
          textUtf8 = decUtf8.decode(buffer);
        } catch (e) {
          console.warn("[lager] UTF-8 Dekodierung fehlgeschlagen:", e);
        }

        const hasMojibake =
          textUtf8.indexOf("Ã¤") !== -1 ||
          textUtf8.indexOf("Ã¶") !== -1 ||
          textUtf8.indexOf("Ã¼") !== -1 ||
          textUtf8.indexOf("Ã„") !== -1 ||
          textUtf8.indexOf("Ã–") !== -1 ||
          textUtf8.indexOf("Ãœ") !== -1 ||
          textUtf8.indexOf("ÃŸ") !== -1 ||
          textUtf8.indexOf("Â")  !== -1;

        if (hasMojibake) {
          try {
            const dec1252 = new TextDecoder("windows-1252", { fatal: false });
            text1252 = dec1252.decode(buffer);
            finalText = text1252;
          } catch (e) {
            console.warn("[lager] windows-1252 Dekodierung fehlgeschlagen, nutze UTF-8:", e);
            finalText = textUtf8;
          }
        } else {
          finalText = textUtf8;
        }

        try {
          await importCsvText(finalText);
        } catch (e) {
          console.warn("[lager] Fehler beim CSV-Import:", e);
                    uiAlert(
            "Fehler beim CSV-Import. Details in der Konsole.",
            "CSV import failed. See console for details.",
            "\u274c Fehler",
            "\u274c Error",
            "OK",
            "OK"
          );
        }
      };

      reader.onerror = () => {
                uiAlert(
          "CSV-Datei konnte nicht gelesen werden.",
          "Could not read the CSV file.",
          "\u274c Fehler",
          "\u274c Error",
          "OK",
          "OK"
        );
      };

      reader.readAsArrayBuffer(file);
    } else {
      // Fallback für sehr alte Browser
      reader.onload = async (ev) => {
        const text = ev.target.result;
        try {
          await importCsvText(text);
        } catch (e) {
          console.warn("[lager] Fehler beim CSV-Import:", e);
                    uiAlert(
            "Fehler beim CSV-Import. Details in der Konsole.",
            "CSV import failed. See console for details.",
            "\u274c Fehler",
            "\u274c Error",
            "OK",
            "OK"
          );
        }
      };
      reader.onerror = () => {
                uiAlert(
          "CSV-Datei konnte nicht gelesen werden.",
          "Could not read the CSV file.",
          "\u274c Fehler",
          "\u274c Error",
          "OK",
          "OK"
        );
      };
      reader.readAsText(file, "UTF-8");
    }
  }

  // --- Init / Event-Handling --------------------------------------------

  function init() {
    const root = byId("lagerverwaltung");
    if (!root) return;
	// DEMO: Welcome-Popup nicht anzeigen (weil wir CSV automatisch laden)
if (window.OPS_DEMO) {
  try { localStorage.setItem(STORAGE_INITIAL, "yes"); } catch (_) {}
}


    // Filter & Suche
    const searchEl          = byId("invSearch");
    const filterLagerEl     = byId("invFilterLager");
    const filterKategorieEl = byId("invFilterKategorie");
    const filterStatusEl    = byId("invFilterStatus") || byId("invFilterBestand");
    const filterBlockedEl   = ensureBlockedFilterEl();
    const resetFiltersBtn   = byId("invResetFilters");


	// Buttons
	const newBtn            = byId("invNewArticleBtn");
	const weTopBtn          = byId("invWeTopBtn");
	const woTopBtn          = byId("invWoTopBtn");
	const receiptsBtn       = byId("invReceiptsBtn");
	const printTableBtn     = byId("invPrintTableBtn");  
	const exportBtn         = byId("invExportBtn");
	const importBtn         = byId("invImportBtn");
	const importFileInput   = byId("invImportFile");
	const inventoryBtn      = byId("invInventoryBtn");
	const deleteAllBtn      = byId("invDeleteAllBtn");
	const columnsBtn        = byId("invColumnsBtn");

    // UI: Tabelle groß/normal umschalten (nur Optik)
    ensureTableSizeToggleButton(root, printTableBtn);

    // Historie/Audit Button (ID: invAuditBtn oder invHistoryBtn, Fallback: Button-Text enthält "Historie")
    let auditHistoryBtn = byId("btnAuditHistory") || byId("invAuditBtn") || byId("invHistoryBtn");
    if (!auditHistoryBtn) {
      try {
        const headerActions = root.querySelector(".card-header > div:last-child");
        if (headerActions) {
          const btns = Array.from(headerActions.querySelectorAll("button"));
          auditHistoryBtn = btns.find(b => {
            const txt = (b.textContent || "").toLowerCase();
            return txt.includes("historie") || txt.includes("history") || (b.dataset && b.dataset.i18n === "btnHistory");
          }) || null;
        }

// Seal / Monatsabschluss + Monatsmappe Buttons (optional – werden ergänzt, falls nicht in HTML)
let sealBtn = byId("invSealBtn");
let monthExportBtn = byId("invMonthExportBtn");
let monthProtocolBtn = byId("invMonthProtocolBtn");
let monthExcelTemplateBtn = byId("invMonthExcelTemplateBtn");
let reportingBtn = byId("invReportingBtn");

try {
  const headerActions = root.querySelector(".card-header > div:last-child");
  if (headerActions) {
    if (!reportingBtn) {
      reportingBtn = document.createElement("button");
      reportingBtn.id = "invReportingBtn";
      reportingBtn.className = "btn btn-ghost";
      reportingBtn.textContent = "📊 Reporting";
      headerActions.insertBefore(reportingBtn, headerActions.firstChild);
    }
    if (!monthExportBtn) {
      monthExportBtn = document.createElement("button");
      monthExportBtn.id = "invMonthExportBtn";
      monthExportBtn.className = "btn btn-ghost";
      monthExportBtn.textContent = "📦 Monatsmappe exportieren";
      headerActions.insertBefore(monthExportBtn, headerActions.firstChild);
    }
    if (!monthProtocolBtn) {
      monthProtocolBtn = document.createElement("button");
      monthProtocolBtn.id = "invMonthProtocolBtn";
      monthProtocolBtn.className = "btn btn-ghost";
      monthProtocolBtn.textContent = "🧾 Monatsabschluss-Protokoll";
      headerActions.insertBefore(monthProtocolBtn, headerActions.firstChild);
    }

    if (!monthExcelTemplateBtn) {
      monthExcelTemplateBtn = document.createElement("button");
      monthExcelTemplateBtn.id = "invMonthExcelTemplateBtn";
      monthExcelTemplateBtn.className = "btn btn-ghost";
      monthExcelTemplateBtn.textContent = "📄 Excel-Template (Monatsabschluss)";
      headerActions.insertBefore(monthExcelTemplateBtn, headerActions.firstChild);
    }
    if (!sealBtn) {
      sealBtn = document.createElement("button");
      sealBtn.id = "invSealBtn";
      sealBtn.className = "btn btn-ghost";
      sealBtn.textContent = "🔐 Monatsabschluss / Seal";
      headerActions.insertBefore(sealBtn, headerActions.firstChild);
    }
  }
} catch (e) {
  console.warn("[lager] Monatsabschluss/Export-Buttons konnten nicht ergänzt werden:", e);
}

      } catch (e) {}
    }



    // BANF/Bestellung (wird ergänzt, falls nicht in HTML) (wird ergänzt, falls nicht in HTML)
    let banfBtn = byId("invBanfBtn");
    if (!banfBtn) {
      try {
        const headerActions = root.querySelector(".card-header > div:last-child");
        if (headerActions) {
          banfBtn = document.createElement("button");
          banfBtn.id = "invBanfBtn";
          banfBtn.className = "btn btn-ghost";
          banfBtn.textContent = "🧾 BANF / Bestellung";
          headerActions.insertBefore(banfBtn, headerActions.firstChild);
        }
      } catch (e) {
        console.warn("[lager] BANF-Button konnte nicht ergänzt werden:", e);
      }
    }


    // Spalten-Dialog
    const columnsDialog     = byId("invColumnsDialog");
    const columnsList       = byId("invColumnsList");
    const columnsCloseBtn   = byId("invColumnsClose");
    const columnsResetBtn   = byId("invColumnsReset");

    // Tabelle & Summary
    const tableBodyEl       = byId("invTableBody");
    const selectAllEl       = byId("invSelectAll");
    const summaryCountEl    = byId("invSummaryCount");
    const summaryLowEl      = byId("invSummaryLow");
    const summaryZeroEl     = byId("invSummaryZero");
    const summaryValueEl    = byId("invSummaryValue");

	// Dialog Artikel
	const articleDialog        = byId("invArticleDialog");
	const articleForm          = byId("invArticleForm");
	const articleIdEl          = byId("invArticleId");
	const artikelnummerEl      = byId("invArtikelnummer");
	const interneArtikelnummerEl = byId("invInterneArtikelnummer");
	const lieferantEl            = byId("invLieferant");
	const bezeichnungEl        = byId("invBezeichnung");
	const herstellerEl         = byId("invHersteller");
	const kategorieEl          = byId("invKategorie");
	const lagerEl              = byId("invLager");
	const lagerartEl           = byId("invLagerart");

	// Regalfelder
	const articleRegalEl       = byId("invArticleRegal");
	const articleFachEl        = byId("invArticleFach");
	const articlePositionEl    = byId("invArticlePosition");
	const articleNiveauEl      = byId("invArticleNiveau");


    const lagerplatztypEl      = byId("invLagerplatztyp");
    const sicherheitsstufeEl   = byId("invSicherheitsstufe");
    const lagerortEl           = byId("invLagerort");
    const mindestbestandEl     = byId("invMindestbestand");
    const bestandEl            = byId("invBestand");
    const einheitEl            = byId("invEinheit");
    const notizenEl            = byId("invNotizen");
    const technPlatzEl         = byId("invTechnischerPlatz");
    const equipmentEl          = byId("invEquipment");
    const anlageEl             = byId("invAnlage");
    const wertEinzelnEl        = byId("invWertEinzeln");
    const waehrungEl           = byId("invWaehrung");
    const wareneingangEl       = byId("invWareneingang");
    const articleMetaEl        = byId("invArticleMeta");
    const articleBookingMetaEl = byId("invArticleBookingMeta");
    const articleCancelBtn     = byId("invArticleCancel");


    // Dialog Buchung
    const bookingDialog          = byId("invBookingDialog");
    const bookingForm            = byId("invBookingForm");
    const bookingArticleIdEl     = byId("invBookingArticleId");
    const bookingDirectionEl     = byId("invBookingDirection");
    const bookingAmountEl        = byId("invBookingAmount");
    const bookingReasonEl        = byId("invBookingReason");
    const bookingTransferFieldsEl= byId("invBookingTransferFields");
    const bookingTargetLagerEl   = byId("invBookingTargetLager");
    const bookingTargetLagerortEl= byId("invBookingTargetLagerort");
    const bookingCostCenterEl    = byId("invBookingCostCenter");
    const bookingEmployeeEl      = byId("invBookingEmployee");
    const bookingNoteEl          = byId("invBookingNote");
    const bookingPriceFieldsEl   = byId("invBookingPriceFields");
    const bookingUnitPriceEl     = byId("invBookingUnitPrice");
    const bookingCurrencyEl      = byId("invBookingCurrency");
    const bookingLabelEl         = byId("invBookingArticleLabel");
    const bookingCancelBtn       = byId("invBookingCancel");

    // Wareneingang / Lieferschein (wird per JS injiziert, damit HTML unverändert bleiben kann)
    let bookingDeliveryFieldsEl     = byId("invBookingDeliveryFields");
    let bookingDeliveryNoteNoEl     = byId("invBookingDeliveryNoteNo");
    let bookingDeliveryNoteDateEl   = byId("invBookingDeliveryNoteDate");
    let bookingSupplierEl           = byId("invBookingSupplier");
    let bookingDeliveryFilesEl      = byId("invBookingDeliveryFiles");
    let bookingDeliveryFilesInfoEl  = byId("invBookingDeliveryFilesInfo");

    ensureBookingDeliveryFields();

    // Dialog Inventur
    const inventoryDialog        = byId("invInventoryDialog");
    const inventoryForm          = byId("invInventoryForm");
    const inventoryArticleIdEl   = byId("invInventoryArticleId");
    const inventoryCurrentEl     = byId("invInventoryCurrent");
    const inventoryCountedEl     = byId("invInventoryCounted");
    const inventoryNoteEl        = byId("invInventoryNote");
    const inventoryLabelEl       = byId("invInventoryArticleLabel");
    const inventoryCancelBtn     = byId("invInventoryCancel");

    const summaryEls = {
      count: summaryCountEl,
      low:   summaryLowEl,
      zero:  summaryZeroEl,
      value: summaryValueEl
    };

    // Daten laden
    // Daten laden
    state.articles     = loadArticles();
    state.bookings     = loadBookings();
    state.banf         = loadBanf();
    state.audit        = loadAudit();
    
    state.seals        = loadSeals();
state.visibleCols  = loadVisibleCols();
    state.columnLabels = loadColumnLabels();



    try {
      if (!state.articles.length 
          && !localStorage.getItem(STORAGE_INITIAL)) {
                uiAlert(
          "Willkommen in der Lagerverwaltung.\n\nSpäter können wir hier eine kurze Einführung/FAQ anzeigen.\nFür jetzt: Bitte einfach mit dem ersten Artikel beginnen.",
          "Welcome to Inventory Management.\n\nLater we can show a short intro/FAQ here.\nFor now: Just start by creating your first item.",
          "\ud83d\udc4b Willkommen",
          "\ud83d\udc4b Welcome",
          "OK",
          "OK"
        );
        localStorage.setItem(STORAGE_INITIAL, "yes");
      }
    } catch {
      // egal
    }

    // NEU: Lager-Filter füllen
    renderLagerFilterOptions(filterLagerEl);
    // wie bisher: Kategorie-Filter füllen
    renderCategoryFilterOptions(filterKategorieEl);
    // Tabelle zeichnen
    renderTable(tableBodyEl, summaryEls, selectAllEl);


    // Seal-Status prüfen (asynchron)
    setTimeout(() => { try { renderSealStatus(); } catch (e) {} }, 50);
    function refresh() {
      // bei jeder Änderung Filter neu aufbauen
      renderLagerFilterOptions(filterLagerEl);
      renderCategoryFilterOptions(filterKategorieEl);
      renderTable(tableBodyEl, summaryEls, selectAllEl);
      saveArticles(state.articles);
      saveBookings(state.bookings);
      // Seal-Status aktualisieren
      setTimeout(() => { try { renderSealStatus(); } catch (e) {} }, 10);
    }
// ---------------- DEMO: CSV Auto-Load + Reset ohne Reload ----------------
if (!window.OpsDeck360Demo) {
  const DEMO_CSV_URL = "./Lagerverwaltung_Produktion.csv";

  // Entfernt doppelte Header-Zeilen (deine CSV hat 2 Kopfzeilen)
  function removeExtraHeaderLines(csvText) {
    const lines = String(csvText || "").split(/\r?\n/);

    // Erste nicht-leere Zeile = Header
    let headerLineIndex = -1;
    for (let i = 0; i < lines.length; i++) {
      if (lines[i].trim() !== "") { headerLineIndex = i; break; }
    }
    if (headerLineIndex < 0) return csvText;

    const headerLine = lines[headerLineIndex].replace(/^\uFEFF/, "");
    const headerFirstCell = (headerLine.split(";")[0] || "").trim().toLowerCase(); // "artikelnummer"

    const out = [];
    for (let i = 0; i < lines.length; i++) {
      const line = lines[i];
      if (!line || line.trim() === "") continue;

      // Wenn es nach der ersten Headerline nochmal mit "Artikelnummer;" anfängt -> weg damit
      if (i > headerLineIndex) {
        const firstCell = (line.replace(/^\uFEFF/, "").split(";")[0] || "").trim().toLowerCase();
        if (firstCell === headerFirstCell) continue;
      }

      out.push(line);
    }

    return out.join("\n") + "\n";
  }

  async function fetchTextSmart(url) {
    const res = await fetch(url + (url.includes("?") ? "&" : "?") + "cb=" + Date.now(), { cache: "no-store" });
    const buf = await res.arrayBuffer();

    // UTF-8 (mit BOM) -> passt zu deiner Datei
    if (window.TextDecoder) {
      const decUtf8 = new TextDecoder("utf-8", { fatal: false });
      const tUtf8 = decUtf8.decode(buf);

      // Falls Windows-1252 (Umlaute kaputt), fallback
      const mojibake = /Ã¤|Ã¶|Ã¼|Ã„|Ã–|Ãœ|ÃŸ|Â/.test(tUtf8);
      if (mojibake) {
        try {
          const dec1252 = new TextDecoder("windows-1252", { fatal: false });
          return dec1252.decode(buf);
        } catch (_) {
          return tUtf8;
        }
      }
      return tUtf8;
    }

    return new TextDecoder().decode(buf);
  }

  window.OpsDeck360Demo = {
    async seedFromCsv(url = DEMO_CSV_URL) {
      const raw = await fetchTextSmart(url);
      const cleaned = removeExtraHeaderLines(raw);
      await importCsvText(cleaned);     // nutzt deine bestehende Import-Logik
      try { refresh(); } catch (_) {}   // aktualisiert die Tabelle
    },

    async reset() {
      // 1) In-Memory Zustand zurücksetzen
      try { state.selectedIds && state.selectedIds.clear && state.selectedIds.clear(); } catch (_) {}
      state.articles = [];
      state.bookings = [];
      state.banf = [];
      state.audit = [];
      state.seals = [];

      // Filter zurück (optional)
      try {
        state.filters.search = "";
        state.filters.lager = "";
        state.filters.kategorie = "";
        state.filters.status = "";
        state.filters.blocked = "";
      } catch (_) {}

      // 2) LocalStorage leeren (falls irgendwo doch gespeichert wurde)
      try {
        localStorage.removeItem(STORAGE_ARTICLES);
        localStorage.removeItem(STORAGE_BOOKINGS);
        localStorage.removeItem(STORAGE_BANF);
        localStorage.removeItem(STORAGE_AUDIT);
        localStorage.removeItem(STORAGE_SEALS);
        localStorage.removeItem(STORAGE_VISIBLE_COLS);
        localStorage.removeItem(STORAGE_COL_LABELS);
      } catch (_) {}

      // 3) Optional: IndexedDB Anhänge löschen
      try {
        if (window.indexedDB && typeof indexedDB.deleteDatabase === "function") {
          indexedDB.deleteDatabase(FILES_DB_NAME);
        }
      } catch (_) {}

      // 4) CSV neu laden
      await this.seedFromCsv(DEMO_CSV_URL);
    }
  };

  // Auto-Load beim Start (nur wenn noch keine Artikel da sind)
  if (window.OPS_DEMO) {
    setTimeout(async () => {
      try {
        if (!state.articles || state.articles.length === 0) {
          await window.OpsDeck360Demo.seedFromCsv(DEMO_CSV_URL);
        }
      } catch (e) {
        console.warn("[DEMO] CSV Auto-Load fehlgeschlagen:", e);
      }
    }, 0);
  }
}
// ------------------------------------------------------------------------

    // --- Artikel-Dialog öffnen -----------------------------------------
function openNewArticle() {
  state.selectedIds.clear();
  fillArticleForm({
    id: articleIdEl,
    artikelnummer: artikelnummerEl,
    interneArtikelnummer: interneArtikelnummerEl,
    lieferant: lieferantEl,
    bezeichnung:   bezeichnungEl,
    hersteller:    herstellerEl,
    kategorie:     kategorieEl,
    lager:         lagerEl,
    lagerart:      lagerartEl,

    // ➕ NEU: Regalfelder
    regal:         articleRegalEl,
    fach:          articleFachEl,
    position:      articlePositionEl,
    niveau:        articleNiveauEl,

    lagerplatztyp: lagerplatztypEl,
    sicherheitsstufe: sicherheitsstufeEl,
    lagerort:      lagerortEl,
    mindestbestand:mindestbestandEl,
    bestand:       bestandEl,
    einheit:       einheitEl,
    notizen:       notizenEl,
    technPlatz:    technPlatzEl,
    equipment:     equipmentEl,
    anlage:        anlageEl,
    wertEinzeln:   wertEinzelnEl,
    waehrung:      waehrungEl,
    wareneingang:  wareneingangEl
  }, null, { meta: articleMetaEl, booking: articleBookingMetaEl });
  openDialog(articleDialog);
}

function openEditArticle(id) {
  const art = state.articles.find(a => String(a.id) === String(id));
  if (!art) return;
  fillArticleForm({
    id: articleIdEl,
    artikelnummer: artikelnummerEl,
    interneArtikelnummer: interneArtikelnummerEl,
    lieferant: lieferantEl,
    bezeichnung:   bezeichnungEl,
    hersteller:    herstellerEl,
    kategorie:     kategorieEl,
    lager:         lagerEl,
    lagerart:      lagerartEl,

    // ➕ NEU: Regalfelder
    regal:         articleRegalEl,
    fach:          articleFachEl,
    position:      articlePositionEl,
    niveau:        articleNiveauEl,

    lagerplatztyp: lagerplatztypEl,
    sicherheitsstufe: sicherheitsstufeEl,
    lagerort:      lagerortEl,
    mindestbestand:mindestbestandEl,
    bestand:       bestandEl,
    einheit:       einheitEl,
    notizen:       notizenEl,
    technPlatz:    technPlatzEl,
    equipment:     equipmentEl,
    anlage:        anlageEl,
    wertEinzeln:   wertEinzelnEl,
    waehrung:      waehrungEl,
    wareneingang:  wareneingangEl
  }, art, { meta: articleMetaEl, booking: articleBookingMetaEl });
  openDialog(articleDialog);
}


    // direction: "in" | "out"
    // presetReason: optional (e.g. "wareneingang", "korrektur", "entnahme", "warenausgang")
    function openBooking(article, direction, presetReason) {
      if (!article || !bookingDialog) return;

      if (direction === "in" && isReceiptBlocked(article)) {
        const reason = String(article.receiptBlockedReason || "").trim();
                uiAlert(
          "Dieser Artikel ist für Zugang/BANF gesperrt.\n\n" +
                    (reason ? ("Grund: " + reason + "\n\n") : "") +
                    "Entnahme ist weiterhin möglich.",
          "This item is blocked for receiving/requisition.\n\n" +
                    (reason ? ("Reason: " + reason + "\n\n") : "") +
                    "Issue/consumption is still allowed.",
          "\ud83d\udd12 Gesperrt",
          "\ud83d\udd12 Blocked",
          "OK",
          "OK"
        );
        return;
      }

      bookingArticleIdEl.value = article.id;
      bookingDirectionEl.value = direction;
      bookingAmountEl.value    = "1";
      // Default: Zugang = Korrektur (ohne Lieferschein-Pflicht), Entnahme = intern
      // Wareneingang / Warenausgang werden über eigene Aktionen/Buttons gesetzt.
      const defReason = (direction === "in") ? "korrektur" : "entnahme";
      bookingReasonEl.value    = presetReason || defReason;
      bookingCostCenterEl.value= "";
      bookingEmployeeEl.value  = "";
      bookingNoteEl.value      = "";

      // Wareneingang: Lieferschein-Felder zurücksetzen / vorbelegen
      if (bookingDeliveryNoteNoEl)   bookingDeliveryNoteNoEl.value = "";
      if (bookingDeliveryNoteDateEl) bookingDeliveryNoteDateEl.value = "";
      if (bookingSupplierEl)         bookingSupplierEl.value = (direction === "in") ? (article.lieferant || "") : "";
      if (bookingDeliveryFilesEl)    bookingDeliveryFilesEl.value = "";
      if (bookingDeliveryFilesInfoEl) bookingDeliveryFilesInfoEl.textContent = "";

      syncBookingDeliveryLabels();
      syncBookingConditionalFields();

      // Preis + Währung nur bei "Zugang" anzeigen
      if (bookingPriceFieldsEl) {
        bookingPriceFieldsEl.style.display = (direction === "in") ? "flex" : "none";
      }
      if (bookingUnitPriceEl) {
        if (direction === "in") {
          const p = toNumber(article.wertEinzeln ?? article.einzelpreis ?? "", 0);
          bookingUnitPriceEl.value = p ? String(p).replace(".", ",") : "";
        } else {
          bookingUnitPriceEl.value = "";
        }
      }
      if (bookingCurrencyEl) {
        if (direction === "in") {
          bookingCurrencyEl.value = (article.waehrung || "EUR");
        } else {
          bookingCurrencyEl.value = "";
        }
      }

      if (bookingTransferFieldsEl) bookingTransferFieldsEl.style.display = "none";
      if (bookingTargetLagerEl)    bookingTargetLagerEl.value = "";
      if (bookingTargetLagerortEl) bookingTargetLagerortEl.value = "";

      const reasonNow = bookingReasonEl ? String(bookingReasonEl.value || "").trim() : "";
      const dirText = (direction === "in")
        ? (reasonNow === "wareneingang" ? "Wareneingang" : "Zugang")
        : (reasonNow === "warenausgang" ? "Warenausgang" : "Entnahme");
      bookingLabelEl.textContent =
        dirText + " für " +
        (article.bezeichnung || article.artikelnummer || "Artikel") +
        " – Lager: " + (article.lager || "unbekannt") +
        " · aktueller Bestand: " + toNumber(article.bestand, 0);

      openDialog(bookingDialog);
    }

       function openInventory(article) {
      if (!article || !inventoryDialog) return;

      inventoryArticleIdEl.value = article.id;
      inventoryCurrentEl.textContent =
        String(toNumber(article.bestand, 0)) + " " + (article.einheit || "");
      inventoryCountedEl.value = String(toNumber(article.bestand, 0));
      inventoryNoteEl.value = "";

      inventoryLabelEl.textContent =
        tLangSafe("Inventur für ", "Inventory count for ") +
        (article.bezeichnung || article.artikelnummer || tLangSafe("Artikel", "Item")) +
        " – " +
        tLangSafe("Lager: ", "Warehouse: ") + (article.lager || "-") +
        ", " +
        tLangSafe("Lagerort: ", "Location: ") + (article.lagerort || "-");

      openDialog(inventoryDialog);
    }



// --- Events --------------------------------------------------------

		if (newBtn) {
		  newBtn.addEventListener("click", (e) => {
			e.preventDefault();
			openNewArticle();
		  });
		}

		// Fester Button: Wareneingang buchen (öffnet Add-on Dialog)
		if (weTopBtn) {
		  weTopBtn.addEventListener("click", (e) => {
			e.preventDefault();
			let prefillId = null;
			try {
			  const ids = Array.from(state.selectedIds || []);
			  if (ids.length === 1) prefillId = ids[0];
			} catch (_) {}
			try {
			  window.dispatchEvent(new CustomEvent("opsWeOpenDialog", { detail: { prefillArticleId: prefillId } }));
			} catch (_) {
			  // fallback
			  window.dispatchEvent(new Event("opsWeOpenDialog"));
			}
		  });
		}

		// Fester Button: Warenausgang buchen (öffnet Add-on Dialog)
		if (woTopBtn) {
		  woTopBtn.addEventListener("click", (e) => {
			e.preventDefault();
			let prefillId = null;
			try {
			  const ids = Array.from(state.selectedIds || []);
			  if (ids.length === 1) prefillId = ids[0];
			} catch (_) {}
			try {
			  window.dispatchEvent(new CustomEvent("opsWoOpenDialog", { detail: { prefillArticleId: prefillId } }));
			} catch (_) {
			  window.dispatchEvent(new Event("opsWoOpenDialog"));
			}
		  });
		}

		// Fester Button: Belegliste
		if (receiptsBtn) {
		  receiptsBtn.addEventListener("click", (e) => {
			e.preventDefault();
			try {
			  window.dispatchEvent(new CustomEvent("opsReceiptsOpenDialog"));
			} catch (_) {
			  window.dispatchEvent(new Event("opsReceiptsOpenDialog"));
			}
		  });
		}

		// Tabellen-Druck
		if (printTableBtn) {
		  printTableBtn.addEventListener("click", (e) => {
			e.preventDefault();
			printSelectedArticles();
		});
		}

		// BANF / Bestellung (aus Auswahl)
		if (banfBtn) {
		  banfBtn.addEventListener("click", (e) => {
			e.preventDefault();
			const selected = state.articles.filter(a => state.selectedIds && state.selectedIds.has(a.id));
			openBanfFromArticles(selected.length ? selected : null);
		  });
		}

		if (exportBtn) {
		  exportBtn.addEventListener("click", (e) => {
			e.preventDefault();
			exportCsv();
		  });
		}

    if (importBtn && importFileInput) {
      importBtn.addEventListener("click", (e) => {
        e.preventDefault();
        importFileInput.value = "";
        importFileInput.click();
      });
      importFileInput.addEventListener("change", () => {
        const file = importFileInput.files && importFileInput.files[0];
        if (file) {
          handleCsvImport(file);
          setTimeout(refresh, 50);
        }
      });
    }

    if (inventoryBtn) {
      inventoryBtn.addEventListener("click", (e) => {
        e.preventDefault();
                uiAlert(
          "Globaler Inventur-Button:\n\n" +
                      "Aktuell ist Inventur artikelbezogen über die Schaltfläche in jeder Zeile möglich.\n" +
                      "Später können wir hier eine Gesamt-Inventur mit Zähllisten implementieren.",
          "Global inventory button:\n\n" +
                      "Currently, inventory is item-based via the button in each row.\n" +
                      "Later we can implement a full inventory with count lists here.",
          "\u2139\ufe0f Hinweis",
          "\u2139\ufe0f Info",
          "OK",
          "OK"
        );
      });
    }

    if (deleteAllBtn) {
      deleteAllBtn.addEventListener("click", async (e) => {
        e.preventDefault();

        if (!state.articles.length && !state.bookings.length) {
                    uiAlert(
            "Es sind keine Daten vorhanden.",
            "No data available.",
            "\u2139\ufe0f Hinweis",
            "\u2139\ufe0f Info",
            "OK",
            "OK"
          );
          return;
        }

        const proceed = await uiConfirm(
          "Tabelle wirklich löschen?\n\n" +
            "Vor dem Löschen wird automatisch eine Sicherung heruntergeladen (CSV + JSON).\n" +
            "Diese Aktion kann nicht rückgängig gemacht werden.",
          "Delete the table data?\n\n" +
            "Before deletion, a backup will be downloaded automatically (CSV + JSON).\n" +
            "This action cannot be undone.",
          "⚠️ Tabelle löschen",
          "⚠️ Delete table",
          "OK",
          "OK",
          "Abbrechen",
          "Cancel"
        );
        if (!proceed) return;

        // --- Backup: CSV (Artikel) + JSON (komplett) -----------------------
        let backupInfo = null;
        try {
          const d = new Date();
          const stamp =
            d.getFullYear() +
            "-" +
            pad2(d.getMonth() + 1) +
            "-" +
            pad2(d.getDate()) +
            "_" +
            pad2(d.getHours()) +
            pad2(d.getMinutes()) +
            pad2(d.getSeconds());

          const note = "Tabelle gesichert vor dem Löschen: " + d.toLocaleString("de-DE");
          const csvName = "lagerbestand-SICHERUNG-vor-loeschen-" + stamp + ".csv";
          const jsonName = "lager-backup-SICHERUNG-vor-loeschen-" + stamp + ".json";

          // CSV (nur Artikel, importierbar wie gewohnt)
          try {
            const csv = csvFromArticles(state.articles || []);
            downloadText(csvName, csv, "text/csv;charset=utf-8;");
          } catch (e2) {
            console.warn("[lager] Backup CSV fehlgeschlagen:", e2);
          }

          // JSON (vollständig: Artikel + Buchungen + Audit + BANF)
          try {
            const payload = {
              note,
              exportedAt: d.toISOString(),
              articles: state.articles || [],
              bookings: state.bookings || [],
              audit: state.audit || [],
              banf: state.banf || []
            };
            downloadText(jsonName, JSON.stringify(payload, null, 2), "application/json;charset=utf-8;");
          } catch (e3) {
            console.warn("[lager] Backup JSON fehlgeschlagen:", e3);
          }

          backupInfo = { csvName, jsonName };
        } catch (err) {
          console.warn("[lager] Backup vor Löschen fehlgeschlagen:", err);
          const cont = await uiConfirm(
            "Sicherung konnte nicht gestartet werden.\n\nTrotzdem löschen?",
            "Backup could not be started.\n\nDelete anyway?",
            "⚠️ Tabelle löschen",
            "⚠️ Delete table",
            "OK",
            "OK",
            "Abbrechen",
            "Cancel"
          );
          if (!cont) return;
        }

                if (backupInfo) {
          const ok = await uiTypeToConfirm(
            "Sicherung wurde gestartet:\n" +
              "• " + backupInfo.csvName + "\n" +
              "• " + backupInfo.jsonName + "\n\n" +
              "Wenn du keinen Download siehst, klicke jetzt auf „Abbrechen“.",
            "Backup started:\n" +
              "• " + backupInfo.csvName + "\n" +
              "• " + backupInfo.jsonName + "\n\n" +
              "If you don't see a download, click “Cancel”.",
            "⚠️ Endgültig löschen",
            "⚠️ Delete permanently",
            "LÖSCHEN",
            "DELETE",
            "Endgültig löschen",
            "Delete",
            "Abbrechen",
            "Cancel"
          );
          if (!ok) return;
        }

        const delMeta = await openDeleteMetaDialogRequired({
          title: tLangSafe("⚠️ Endgültig löschen", "⚠️ Delete permanently"),
          intro: tLangSafe(
            "Bitte Name/Kürzel und Grund angeben.\nDies wird in der Audit-Historie protokolliert.",
            "Please enter your name/initials and a reason.\nThis will be recorded in the audit history."
          ),
          defaultName: currentEmployeeName()
        });
        if (!delMeta) return;

        // --- Löschen -------------------------------------------------------
        // Audit: Löschvorgang protokollieren (vor dem Leeren!)
        try {
          addAuditEvent("table.cleared", {
            employee: (delMeta && delMeta.name) ? delMeta.name : currentEmployeeName(),
            reason: (delMeta && delMeta.reason) ? delMeta.reason : "",
            countsBefore: {
              articles: (state.articles || []).length,
              bookings: (state.bookings || []).length
            },
            backup: backupInfo || null,
            note: "Tabelle gesichert vor dem Löschen"
          });
        } catch (e) {
          console.warn("[lager] Audit für table.cleared fehlgeschlagen:", e);
        }

        state.articles = [];
        state.bookings = [];
        state.selectedIds.clear();
        saveArticles(state.articles);
        saveBookings(state.bookings);
        try { await clearFilesDb(); } catch (e) {}
        renderCategoryFilterOptions(filterKategorieEl);
        renderTable(tableBodyEl, summaryEls, selectAllEl);
      });
    }

    if (auditHistoryBtn) {
      auditHistoryBtn.addEventListener("click", (e) => {
        e.preventDefault();
        openAuditHistory();
      });
    }





// ✅ Robust: Listener an ALLE Buttons hängen (falls IDs doppelt vorkommen / Buttons dynamisch ergänzt werden)
document.querySelectorAll("#invReportingBtn").forEach((btn) => {
  btn.addEventListener("click", (e) => {
    e.preventDefault();
    const dlg = ensureReportingDialog();
    openDialog(dlg);
  });
});

document.querySelectorAll("#invSealBtn").forEach((btn) => {
  btn.addEventListener("click", async (e) => {
    e.preventDefault();
    await sealMonthFlow();
  });
});

document.querySelectorAll("#invMonthExportBtn").forEach((btn) => {
  btn.addEventListener("click", async (e) => {
    e.preventDefault();
    await exportMonthlyBundleFlow();
  });
});

document.querySelectorAll("#invMonthProtocolBtn").forEach((btn) => {
  btn.addEventListener("click", async (e) => {
    e.preventDefault();
    await printMonthProtocolFlow();
  });
});


document.querySelectorAll("#invMonthExcelTemplateBtn").forEach((btn) => {
  btn.addEventListener("click", (e) => {
    e.preventDefault();
    downloadMonthCloseExcelTemplate();
  });
});

// UI vereinfachen: Monatsabschluss-Buttons in "Reporting" bündeln (Funktionen bleiben erhalten)
try {
  document.querySelectorAll("#invSealBtn, #invMonthExportBtn, #invMonthProtocolBtn, #invMonthExcelTemplateBtn").forEach((btn) => {
    if (btn) btn.style.display = "none";
  });
} catch (e) {}

    if (columnsBtn && columnsDialog) {
      columnsBtn.addEventListener("click", (e) => {
        e.preventDefault();
        buildColumnsDialog(columnsList);
        openDialog(columnsDialog);
      });
    }

    if (columnsCloseBtn && columnsDialog) {
      columnsCloseBtn.addEventListener("click", (e) => {
        e.preventDefault();
        closeDialog(columnsDialog);
      });
    }

    if (columnsDialog) {
      columnsDialog.addEventListener("click", (e) => {
        if (e.target === columnsDialog) {
          closeDialog(columnsDialog);
        }
      });
    }

    if (columnsResetBtn) {
      columnsResetBtn.addEventListener("click", (e) => {
        e.preventDefault();
        state.visibleCols = null;
        try {
          localStorage.removeItem(STORAGE_VISIBLE_COLS);
        } catch (e) {}
        applyColumnVisibility();
        buildColumnsDialog(columnsList);
      });
    }

    if (resetFiltersBtn) {
      resetFiltersBtn.addEventListener("click", (e) => {
        e.preventDefault();
        if (searchEl) searchEl.value = "";
        if (filterLagerEl) filterLagerEl.value = "";
        if (filterKategorieEl) filterKategorieEl.value = "";
        if (filterStatusEl) filterStatusEl.value = "";
        if (filterBlockedEl) filterBlockedEl.value = "";
        state.filters.search = "";
        state.filters.lager = "";
        state.filters.kategorie = "";
        state.filters.status = "";
        state.filters.blocked = "";
        renderTable(tableBodyEl, summaryEls, selectAllEl);
      });
    }

    if (searchEl) {
      searchEl.addEventListener("input", () => {
        state.filters.search = searchEl.value || "";
        renderTable(tableBodyEl, summaryEls, selectAllEl);
      });
    }

    if (filterLagerEl) {
      filterLagerEl.addEventListener("change", () => {
        state.filters.lager = filterLagerEl.value || "";
        renderTable(tableBodyEl, summaryEls, selectAllEl);
      });
    }

    if (filterKategorieEl) {
      filterKategorieEl.addEventListener("change", () => {
        state.filters.kategorie = filterKategorieEl.value || "";
        renderTable(tableBodyEl, summaryEls, selectAllEl);
      });
    }

    if (filterStatusEl) {
      filterStatusEl.addEventListener("change", () => {
        state.filters.status = filterStatusEl.value || "";
        renderTable(tableBodyEl, summaryEls, selectAllEl);
      });
    }
    if (filterBlockedEl) {
      filterBlockedEl.addEventListener("change", () => {
        state.filters.blocked = filterBlockedEl.value || "";
        renderTable(tableBodyEl, summaryEls, selectAllEl);
      });
    }


    if (articleCancelBtn) {
      articleCancelBtn.addEventListener("click", (e) => {
        e.preventDefault();
        closeDialog(articleDialog);
      });
    }

    if (articleForm) {
      articleForm.addEventListener("submit", (e) => {
        e.preventDefault();

        if (!artikelnummerEl.value.trim() && !bezeichnungEl.value.trim()) {
                    uiAlert(
            "Bitte mindestens Artikelnummer oder Bezeichnung angeben.",
            "Please enter at least an item number or description.",
            "\u26a0\ufe0f Hinweis",
            "\u26a0\ufe0f Note",
            "OK",
            "OK"
          );
          return;
        }

        const existingId = articleIdEl.value || null;
        const existing = existingId
          ? state.articles.find((a) => String(a.id) === String(existingId))
          : null;

        const art = buildArticleFromForm(
          {
            id: articleIdEl,
            artikelnummer: artikelnummerEl,
    interneArtikelnummer: interneArtikelnummerEl,
    lieferant: lieferantEl,
            bezeichnung: bezeichnungEl,
            hersteller: herstellerEl,
            kategorie: kategorieEl,
            lager: lagerEl,
            lagerart: lagerartEl,
			regal: articleRegalEl,
            fach: articleFachEl,
            position: articlePositionEl,
            niveau: articleNiveauEl,
            lagerplatztyp: lagerplatztypEl,
            sicherheitsstufe: sicherheitsstufeEl,
            lagerort: lagerortEl,
            mindestbestand: mindestbestandEl,
            bestand: bestandEl,
            einheit: einheitEl,
            notizen: notizenEl,
            technPlatz: technPlatzEl,
            equipment: equipmentEl,
            anlage: anlageEl,
            wertEinzeln: wertEinzelnEl,
            waehrung: waehrungEl,
            wareneingang: wareneingangEl,
          },
          existing || null
        );

        if (existing) {
          const idx = state.articles.findIndex((a) => String(a.id) === String(existing.id));
          if (idx !== -1) state.articles[idx] = art;
        } else {
          state.articles.push(art);
        }

        // --- Audit: Artikel angelegt / geändert --------------------------------
        try {
          const before = existing ? Object.assign({}, existing) : null;
          const after  = Object.assign({}, art);
          addAuditEvent(existing ? "article.updated" : "article.created", {
            articleId: after.id,
            artikelnummer: after.artikelnummer || "",
            bezeichnung: after.bezeichnung || "",
            before: before ? { bestand: before.bestand, wertEinzeln: before.wertEinzeln, waehrung: before.waehrung } : null,
            after:  { bestand: after.bestand,  wertEinzeln: after.wertEinzeln,  waehrung: after.waehrung }
          });
        } catch (e) {}

        refresh();
        closeDialog(articleDialog);
      });
    }


    // --- Wareneingang-Felder (Lieferschein / Anlagen) --------------------
    function ensureBookingDeliveryFields() {
      if (!bookingForm) return;

      // Falls nicht in HTML vorhanden: dynamisch anlegen
      let wrap = byId("invBookingDeliveryFields");
      if (!wrap) {
        wrap = document.createElement("div");
        wrap.id = "invBookingDeliveryFields";
        wrap.style.display = "none";
        wrap.style.marginTop = "10px";
        wrap.style.paddingTop = "8px";
        wrap.style.borderTop = "1px solid rgba(0,0,0,.08)";

        wrap.innerHTML = `
          <div style="display:flex; gap:10px; flex-wrap:wrap; align-items:flex-end;">
            <label style="display:flex; flex-direction:column; gap:4px; min-width:180px;">
              <span id="invBookingDeliveryNoteNoLabel">Lieferschein-Nr. (Pflicht)</span>
              <input id="invBookingDeliveryNoteNo" type="text" />
            </label>
            <label style="display:flex; flex-direction:column; gap:4px; min-width:170px;">
              <span id="invBookingDeliveryNoteDateLabel">Lieferschein-Datum</span>
              <input id="invBookingDeliveryNoteDate" type="date" />
            </label>
            <label style="display:flex; flex-direction:column; gap:4px; min-width:220px;">
              <span id="invBookingSupplierLabel">Lieferant</span>
              <input id="invBookingSupplier" type="text" />
            </label>
            <label style="display:flex; flex-direction:column; gap:4px; min-width:260px;">
              <span id="invBookingDeliveryFilesLabel">Lieferschein-Datei (optional)</span>
              <input id="invBookingDeliveryFiles" type="file" accept="application/pdf,image/*" multiple />
            </label>
          </div>
          <div id="invBookingDeliveryFilesInfo" style="font-size:12px; opacity:.85; margin-top:6px;"></div>
        `;

        // möglichst vor der Notiz einfügen, sonst ans Ende
        if (bookingNoteEl && bookingNoteEl.parentElement) {
          bookingNoteEl.parentElement.insertAdjacentElement("beforebegin", wrap);
        } else {
          bookingForm.appendChild(wrap);
        }
      }

      // refs aktualisieren
      bookingDeliveryFieldsEl     = wrap;
      bookingDeliveryNoteNoEl     = byId("invBookingDeliveryNoteNo");
      bookingDeliveryNoteDateEl   = byId("invBookingDeliveryNoteDate");
      bookingSupplierEl           = byId("invBookingSupplier");
      bookingDeliveryFilesEl      = byId("invBookingDeliveryFiles");
      bookingDeliveryFilesInfoEl  = byId("invBookingDeliveryFilesInfo");

      // Labels (DE/EN)
      syncBookingDeliveryLabels();

      // Datei-Info
      if (bookingDeliveryFilesEl) {
        bookingDeliveryFilesEl.addEventListener("change", () => {
          if (!bookingDeliveryFilesInfoEl) return;
          const files = bookingDeliveryFilesEl.files ? Array.from(bookingDeliveryFilesEl.files) : [];
          if (!files.length) {
            bookingDeliveryFilesInfoEl.textContent = "";
            return;
          }
          bookingDeliveryFilesInfoEl.textContent =
            tLangSafe("Ausgewählt: ", "Selected: ") +
            files.map(f => f.name).join(", ");
        });
      }
    }

    function syncBookingDeliveryLabels() {
      const set = (id, de, en) => {
        const el = byId(id);
        if (el) el.textContent = tLangSafe(de, en);
      };
      set("invBookingDeliveryNoteNoLabel",   "Lieferschein-Nr. (Pflicht)", "Delivery note no. (required)");
      set("invBookingDeliveryNoteDateLabel", "Lieferschein-Datum",         "Delivery note date");
      set("invBookingSupplierLabel",         "Lieferant",                  "Supplier");
      set("invBookingDeliveryFilesLabel",    "Lieferschein-Datei (optional)", "Delivery note file (optional)");
    }

    function syncBookingConditionalFields() {
      const dir = bookingDirectionEl ? bookingDirectionEl.value : "";
      const reason = bookingReasonEl ? bookingReasonEl.value : "";
      const isReceipt = (dir === "in" && reason === "wareneingang");
      const isTransfer = (reason === "umlagerung");

      if (bookingTransferFieldsEl) bookingTransferFieldsEl.style.display = isTransfer ? "" : "none";
      if (bookingDeliveryFieldsEl) bookingDeliveryFieldsEl.style.display = isReceipt ? "" : "none";

      if (isReceipt) {
        const id = bookingArticleIdEl ? bookingArticleIdEl.value : "";
        const art = state.articles.find((a) => String(a.id) === String(id));
        if (art && bookingSupplierEl && !String(bookingSupplierEl.value || "").trim()) {
          bookingSupplierEl.value = art.lieferant || "";
        }
      }
    }


    if (bookingReasonEl && bookingTransferFieldsEl) {
      bookingReasonEl.addEventListener("change", () => {
        const isTransfer = bookingReasonEl.value === "umlagerung";
        bookingTransferFieldsEl.style.display = isTransfer ? "" : "none";
        syncBookingDeliveryLabels();
        syncBookingConditionalFields();

        if (isTransfer && bookingTargetLagerEl && bookingTargetLagerortEl) {
          const id = bookingArticleIdEl.value;
          const art = state.articles.find((a) => String(a.id) === String(id));
          if (art) {
            if (!bookingTargetLagerEl.value)
              bookingTargetLagerEl.value = art.lager || "";
            if (!bookingTargetLagerortEl.value)
              bookingTargetLagerortEl.value = art.lagerort || "";
          }
        }
      });
    }

    if (bookingCancelBtn) {
      bookingCancelBtn.addEventListener("click", (e) => {
        e.preventDefault();
        closeDialog(bookingDialog);
      });
    }

    if (bookingForm) {
      bookingForm.addEventListener("submit", async (e) => {
        e.preventDefault();

        const id = bookingArticleIdEl.value;
        if (!id) return;

        const idx = state.articles.findIndex((a) => String(a.id) === String(id));
        if (idx === -1) return;

        const art = state.articles[idx];

        const direction = bookingDirectionEl
          ? bookingDirectionEl.value
          : "out"; // "in" oder "out"

        if (direction === "in" && isReceiptBlocked(art)) {
          const reasonTxt = String(art.receiptBlockedReason || "").trim();
                    uiAlert(
            "Dieser Artikel ist für Zugang/BANF gesperrt.\n\n" +
                          (reasonTxt ? ("Grund: " + reasonTxt + "\n\n") : "") +
                          "Entnahme ist weiterhin möglich.",
            "This item is blocked for receiving/requisition.\n\n" +
                          (reasonTxt ? ("Reason: " + reasonTxt + "\n\n") : "") +
                          "Issue/consumption is still allowed.",
            "\ud83d\udd12 Gesperrt",
            "\ud83d\udd12 Blocked",
            "OK",
            "OK"
          );
          return;
        }

        const amount = toNumber(bookingAmountEl.value, 0);
        if (!amount || amount <= 0) {
          return; // ungültige Menge
        }

        const reason = bookingReasonEl
          ? bookingReasonEl.value || "buchung"
          : "buchung";
        const current = toNumber(art.bestand, 0);

        // Standardfall: Zugang / Entnahme
        let newBestand = current;
        if (direction === "in") {
          newBestand = current + amount;
        } else if (direction === "out") {
          newBestand = current - amount;
          if (newBestand < 0) newBestand = 0;
        }


        // Wareneingang: Lieferschein-Daten + Anlagen (nur bei Zugang + Grund=wareneingang)
        const isReceipt = (direction === "in" && reason === "wareneingang");
        let supplier = "";
        let deliveryNoteNo = "";
        let deliveryNoteDate = "";
        let attachments = [];

        if (isReceipt) {
          supplier = bookingSupplierEl ? String(bookingSupplierEl.value || "").trim() : "";
          deliveryNoteNo = bookingDeliveryNoteNoEl ? String(bookingDeliveryNoteNoEl.value || "").trim() : "";
          deliveryNoteDate = bookingDeliveryNoteDateEl ? String(bookingDeliveryNoteDateEl.value || "").trim() : "";

          if (!deliveryNoteNo) {
            uiAlert(
              "Bitte Lieferschein-Nr. eintragen.",
              "Please enter the delivery note number.",
              "⚠️ Pflichtfeld",
              "⚠️ Required",
              "OK",
              "OK"
            );
            return;
          }

          // Dateien speichern (optional)
          if (bookingDeliveryFilesEl && bookingDeliveryFilesEl.files && bookingDeliveryFilesEl.files.length) {
            const files = Array.from(bookingDeliveryFilesEl.files);
            for (const f of files) {
              try {
                const rec = await storeFileInDb(f, "delivery_note");
                if (rec && rec.id) {
                  attachments.push({
                    id: rec.id,
                    name: rec.name,
                    type: rec.type,
                    size: rec.size,
                    ts: rec.ts,
                    kind: rec.kind || "delivery_note"
                  });
                }
              } catch (e) {
                console.warn("[lager] Datei konnte nicht gespeichert werden:", e);
              }
            }
          }
        }

        const res = addBooking(
          art,
          {
            direction: direction,
            amount: amount,
            reason: reason,
            note: bookingNoteEl ? bookingNoteEl.value || "" : "",
            kostenstelle: bookingCostCenterEl
              ? bookingCostCenterEl.value || ""
              : "",
            mitarbeiter: bookingEmployeeEl ? bookingEmployeeEl.value || "" : "",
            newBestand: newBestand,
            supplier: supplier,
            deliveryNoteNo: deliveryNoteNo,
            deliveryNoteDate: deliveryNoteDate,
            attachments: attachments,
          },
          state.bookings
        );

        // Anlagen am Artikel merken (für Details)
        if (isReceipt && attachments && attachments.length) {
          const prev = Array.isArray(res.updatedArticle.attachments) ? res.updatedArticle.attachments : [];
          const meta = attachments.map((a) => Object.assign({}, a, {
            articleId: art.id,
            bookingId: res.booking ? res.booking.id : null,
            supplier: supplier,
            deliveryNoteNo: deliveryNoteNo,
            deliveryNoteDate: deliveryNoteDate
          }));
          res.updatedArticle.attachments = prev.concat(meta);
        }


        // --- Gleitender Durchschnittspreis (nur bei Zugang) ---
        if (direction === "in") {
          const oldQty   = toNumber(art.bestand, 0);
          const oldPrice = toNumber(art.wertEinzeln, 0);
          const inQty    = amount;
          const inPrice  = bookingUnitPriceEl ? toNumber(bookingUnitPriceEl.value, 0) : 0;

          if (oldQty >= 0 && inQty > 0 && inPrice > 0) {
            const newPrice = ((oldQty * oldPrice) + (inQty * inPrice)) / (oldQty + inQty);
            res.updatedArticle.wertEinzeln = Number(newPrice.toFixed(2));

            if (bookingCurrencyEl && bookingCurrencyEl.value) {
              res.updatedArticle.waehrung = bookingCurrencyEl.value.trim();
            } else if (art.waehrung) {
              res.updatedArticle.waehrung = art.waehrung;
            }
          }
        }

        // --- Audit: Buchung (Zugang/Entnahme/Inventur/Umlagerung) -------------
        try {
          addAuditEvent("booking.created", {
            bookingId: res.booking ? res.booking.id : null,
            articleId: art.id,
            artikelnummer: art.artikelnummer || "",
            bezeichnung: art.bezeichnung || "",
            direction: direction,
            amount: amount,
            reason: reason,
            bestandVorher: toNumber(art.bestand, 0),
            bestandNachher: newBestand,
            unitPrice: (direction === "in" && bookingUnitPriceEl) ? (bookingUnitPriceEl.value || "") : "",
            currency: (direction === "in" && bookingCurrencyEl) ? (bookingCurrencyEl.value || res.updatedArticle.waehrung || "") : (res.updatedArticle.waehrung || ""),
            avgPriceAfter: res.updatedArticle.wertEinzeln ?? null,
            kostenstelle: bookingCostCenterEl ? (bookingCostCenterEl.value || "") : "",
            mitarbeiter: bookingEmployeeEl ? (bookingEmployeeEl.value || "") : "",
            note: bookingNoteEl ? (bookingNoteEl.value || "") : "",
            supplier: supplier || "",
            deliveryNoteNo: deliveryNoteNo || "",
            deliveryNoteDate: deliveryNoteDate || "",
            attachmentsCount: (attachments && attachments.length) ? attachments.length : 0
          });
        } catch (e) {}

        state.articles[idx] = res.updatedArticle;
        refresh();
        closeDialog(bookingDialog);

        // ➕ Entnahme-/Warenausgangsbeleg drucken (optional)
        // Nur für echte Abgänge (nicht Umlagerung, nicht Inventur)
        if (direction === "out" && (reason === "entnahme" || reason === "warenausgang")) {
          try {
            const label = (reason === "warenausgang") ? "Warenausgang" : "Entnahme";
            uiConfirm(
              label + " wurde gebucht.\n\nBeleg jetzt drucken?",
              "Issue was posted.\n\nPrint the receipt now?",
              "🖨️ Drucken",
              "🖨️ Print",
              "Drucken",
              "Print",
              "Nicht jetzt",
              "Not now"
            ).then((yes) => {
              if (yes) {
                // für den Beleg nehmen wir den Artikelstand VOR der Buchung
                const artBefore = Object.assign({}, art);
                printEntnahmeBeleg(artBefore, res.booking);
              }
            });
          } catch (e) {
            console.warn("[lager] Entnahmebeleg konnte nicht gedruckt werden:", e);
          }
        }
      });
    }

    if (inventoryCancelBtn) {
      inventoryCancelBtn.addEventListener("click", (e) => {
        e.preventDefault();
        closeDialog(inventoryDialog);
      });
    }

    if (inventoryForm) {
      inventoryForm.addEventListener("submit", (e) => {
        e.preventDefault();
        const id = inventoryArticleIdEl.value;
        if (!id) return;
        const idx = state.articles.findIndex((a) => String(a.id) === String(id));
        if (idx === -1) return;

        const art = state.articles[idx];
        const current = toNumber(art.bestand, 0);
        const counted = toNumber(inventoryCountedEl.value, current);
        const diff = counted - current;
        if (diff === 0) {
          closeDialog(inventoryDialog);
          return;
        }

        const direction = diff > 0 ? "in" : "out";
        const amount = Math.abs(diff);

        const res = addBooking(
          art,
          {
            direction: direction,
            amount: amount,
            reason: "inventur",
            note: inventoryNoteEl.value || "",
            kostenstelle: "",
            mitarbeiter: "",
            newBestand: counted,
          },
          state.bookings
        );

        // --- Audit: Inventur -----------------------------------------------
        try {
          addAuditEvent("inventory.counted", {
            bookingId: res.booking ? res.booking.id : null,
            articleId: art.id,
            artikelnummer: art.artikelnummer || "",
            bezeichnung: art.bezeichnung || "",
            bestandVorher: current,
            bestandNachher: counted,
            diff: counted - current,
            note: inventoryNoteEl ? (inventoryNoteEl.value || "") : ""
          });
        } catch (e) {}

        state.articles[idx] = res.updatedArticle;
        refresh();
        closeDialog(inventoryDialog);
      });
    }

    if (selectAllEl) {
      selectAllEl.addEventListener("change", () => {
        const list = getFilteredArticles();
        if (selectAllEl.checked) {
          list.forEach((a) => state.selectedIds.add(String(a.id)));
        } else {
          list.forEach((a) => state.selectedIds.delete(String(a.id)));
        }
        renderTable(tableBodyEl, summaryEls, selectAllEl);
      });
    }

    if (tableBodyEl && !tableBodyEl.dataset.invBound) {
      tableBodyEl.dataset.invBound = "1";

      tableBodyEl.addEventListener("change", (e) => {
        const target = e.target;
        if (!target) return;

        // Checkboxen (Mehrfachauswahl)
        const checkbox = target.closest && target.closest("input.inv-row-select");
        if (checkbox) {
          const id = checkbox.getAttribute("data-id");
          if (!id) return;
          if (checkbox.checked) state.selectedIds.add(String(id));
          else state.selectedIds.delete(String(id));
          updateSelectAllCheckbox(selectAllEl, getFilteredArticles());
          return;
        }

        // Aktionen-Dropdown (Bearbeiten / Wareneingang / Zugang / Warenausgang / Entnahme / Inventur / Details / BANF)
        const select = target.closest && target.closest("select.inv-action-select");
        if (!select) return;

        const id = select.getAttribute("data-id");
        const action = select.value;
        if (!id || !action) return;

        const art = state.articles.find((a) => String(a.id) === String(id));
        if (!art) {
          select.value = "";
          return;
        }

        if (action === "edit") {
          openEditArticle(String(id));
        } else if (action === "block") {
          blockReceiptsDialog(art);
        } else if (action === "unblock") {
          unblockReceiptsDialog(art);
        } else if (action === "goods-in") {
          // Wareneingang (Lieferschein-Pflicht)
          openBooking(art, "in", "wareneingang");
        } else if (action === "book-in") {
          // Zugang als Korrektur (ohne Lieferschein)
          openBooking(art, "in", "korrektur");
        } else if (action === "goods-out") {
          openBooking(art, "out", "warenausgang");
        } else if (action === "book-out") {
          openBooking(art, "out", "entnahme");
        } else if (action === "inventory") {
          openInventory(art);
        } else if (action === "details") {
          openArticleDetails(art);
        } else if (action === "banf") {
          openBanfFromArticles([art]);
        }

        // wieder auf Platzhalter zurücksetzen
        select.value = "";
      });

      // Legacy-Click-Handler (für alte Action-Menüs / Buttons)
      tableBodyEl.addEventListener("click", (e) => {
        const target = e.target;
        if (!target) return;

        // 1) Klick auf "Aktionen ▼" -> Menü öffnen/schließen
        const toggleBtn = target.closest && target.closest(".inv-action-toggle");
        if (toggleBtn) {
          const menu = toggleBtn.closest(".action-menu");
          if (!menu) return;

          tableBodyEl.querySelectorAll(".action-menu.open").forEach((m) => {
            if (m !== menu) m.classList.remove("open");
          });

          menu.classList.toggle("open");
          return;
        }

        // 2) Details-Button
        const detailsBtn = target.closest && target.closest("button.btn-details");
        if (detailsBtn) {
          const id = detailsBtn.getAttribute("data-id");
          if (!id) return;
          const art = state.articles.find((a) => String(a.id) === String(id));
          if (!art) return;
          openArticleDetails(art);
          return;
        }

        // 3) Aktions-Buttons (Bearbeiten / Buchung / Inventur)
        const btn = target.closest && target.closest("button[data-action]");
        if (!btn) return;

        const id = btn.getAttribute("data-id");
        const action = btn.getAttribute("data-action");
        const art = state.articles.find((a) => String(a.id) === String(id));
        if (!art) return;

        // Menü schließen, aus dem der Klick kam
        const menu = btn.closest(".action-menu");
        if (menu) menu.classList.remove("open");

        if (action === "edit") {
          openEditArticle(String(id));
        } else if (action === "block") {
          blockReceiptsDialog(art);
        } else if (action === "unblock") {
          unblockReceiptsDialog(art);
        } else if (action === "banf") {
          openBanfFromArticles([art]);
        } else if (action === "goods-in") {
          openBooking(art, "in", "wareneingang");
        } else if (action === "book-in") {
          openBooking(art, "in", "korrektur");
        } else if (action === "goods-out") {
          openBooking(art, "out", "warenausgang");
        } else if (action === "book-out") {
          openBooking(art, "out", "entnahme");
        } else if (action === "inventory") {
          openInventory(art);
        }
      });
    }

    // Nach dem ersten Render Spalteneinblendung anwenden
    applyColumnVisibility();
    applyColumnLabels();

    console.log(
      "[lager] Lagerverwaltung initialisiert – Artikel:",
      state.articles.length,
      "Buchungen:",
      state.bookings.length
    );
  }
  
function enhanceActionMenus(container) {
  if (!container) return;

  const menus = container.querySelectorAll("details.action-menu");

  menus.forEach((menu) => {
    // Nur noch: wenn eines geöffnet wird, die anderen schließen
    menu.addEventListener("toggle", () => {
      if (!menu.open) return;
      container
        .querySelectorAll("details.action-menu[open]")
        .forEach((other) => {
          if (other !== menu) other.removeAttribute("open");
        });
    });
  });
}



  
// Artikeldetail-Ansicht (Einzeldatenblatt)
function openArticleDetails(article) {
  const dialog = byId("invDetailDialog");
  const root   = byId("artikelDetailView");
  if (!root) return;

  // kleiner Helfer, damit leere Felder als "–" erscheinen
  const setText = (id, value) => {
    const el = document.getElementById(id);
    if (!el) return;
    const v =
      value != null && String(value).trim() !== ""
        ? String(value).trim()
        : "–";
    el.textContent = v;
  };

  // Grunddaten
  setText("detail_artikelnummer", article.artikelnummer || "");
  setText("detail_interneArtikelnummer", article.interneArtikelnummer || "");
  setText("detail_bezeichnung",   article.bezeichnung   || "");
  // Status: Zugang/BANF gesperrt (Restentnahme erlaubt)
  try {
    const tbl = root.querySelector(".detail-table");
    if (tbl) {
      let row = document.getElementById("detail_receipt_block_row");
      if (!row) {
        row = document.createElement("tr");
        row.id = "detail_receipt_block_row";
        row.innerHTML = '<th>Beschaffung / Zugang</th><td id="detail_receipt_block"></td>';
        const bezeichEl = document.getElementById("detail_bezeichnung");
        const bezeichRow = bezeichEl ? bezeichEl.closest("tr") : null;
        if (bezeichRow && bezeichRow.parentNode) bezeichRow.parentNode.insertBefore(row, bezeichRow.nextSibling);
        else tbl.appendChild(row);
      }
      const td = document.getElementById("detail_receipt_block");
      if (td) {
        const blocked = isReceiptBlocked(article);
        const reason = String(article.receiptBlockedReason || "").trim();
        const since = article.receiptBlockedAt ? formatDateTime(article.receiptBlockedAt) : "";
        const by = String(article.receiptBlockedBy || "").trim();
        if (blocked) {
          td.textContent = "🔒 gesperrt" + (reason ? (" — " + reason) : "");
          td.title =
            "Gesperrt für Zugang/BANF" +
            (reason ? ("\nGrund: " + reason) : "") +
            (since ? ("\nSeit: " + since) : "") +
            (by ? ("\nVon: " + by) : "");
        } else {
          const unReason = String(article.receiptUnblockedReason || "").trim();
          const unAt = article.receiptUnblockedAt ? formatDateTime(article.receiptUnblockedAt) : "";
          const unBy = String(article.receiptUnblockedBy || "").trim();
          td.textContent = "🔓 frei";
          td.title =
            "Nicht gesperrt" +
            (unReason ? ("\nLetzte Entsperrung: " + unReason) : "") +
            (unAt ? ("\nAm: " + unAt) : "") +
            (unBy ? ("\nVon: " + unBy) : "");
        }
      }

      // 📎 Lieferscheine / Anlagen
      let row2 = document.getElementById("detail_delivery_notes_row");
      if (!row2) {
        row2 = document.createElement("tr");
        row2.id = "detail_delivery_notes_row";
        const th2 = document.createElement("th");
        th2.textContent = tLangSafe("Lieferscheine / Anlagen", "Delivery notes / attachments");
        const td2 = document.createElement("td");
        td2.id = "detail_delivery_notes";
        row2.appendChild(th2);
        row2.appendChild(td2);

        // direkt nach der Sperr-Row einfügen
        const r1 = document.getElementById("detail_receipt_block_row");
        if (r1 && r1.parentNode) {
          if (r1.nextSibling) r1.parentNode.insertBefore(row2, r1.nextSibling);
          else r1.parentNode.appendChild(row2);
        } else {
          tbl.appendChild(row2);
        }
      }

      const td2 = document.getElementById("detail_delivery_notes");
      if (td2) {
        td2.innerHTML = "";
        const atts = Array.isArray(article.attachments) ? article.attachments.filter(a => a && a.id) : [];
        if (!atts.length) {
          td2.textContent = "–";
        } else {
          const ul = document.createElement("ul");
          ul.style.listStyle = "none";
          ul.style.padding = "0";
          ul.style.margin = "0";

          // neueste zuerst
          const sorted = atts.slice().sort((a,b) => String(b.ts||"").localeCompare(String(a.ts||"")));
          for (const att of sorted) {
            const li = document.createElement("li");
            li.style.display = "flex";
            li.style.gap = "8px";
            li.style.alignItems = "center";
            li.style.marginBottom = "4px";

            const btn = document.createElement("button");
            btn.type = "button";
            btn.textContent = tLangSafe("Öffnen", "Open");
            btn.style.padding = "2px 8px";
            btn.style.border = "1px solid rgba(0,0,0,.2)";
            btn.style.borderRadius = "6px";
            btn.style.background = "white";
            btn.style.cursor = "pointer";
            btn.addEventListener("click", async () => {
              await openStoredFile(att.id);
            });

            const label = document.createElement("span");
            const dn = att.deliveryNoteNo ? (" • " + tLangSafe("LS", "DN") + ": " + att.deliveryNoteNo) : "";
            label.textContent = (att.name || tLangSafe("Datei", "File")) + dn;

            li.appendChild(btn);
            li.appendChild(label);
            ul.appendChild(li);
          }

          td2.appendChild(ul);
        }
      }

    }
  } catch (e) {
    // Detail-Status ist optional
  }

  setText("detail_hersteller",    article.hersteller    || "");
  setText("detail_lieferant",     article.lieferant     || "");
  setText("detail_kategorie",     article.kategorie     || "");  // ← NEU

  // Standortdaten
  setText("detail_lager",         article.lager         || "");
  setText("detail_regal",         article.regal         || "");  // ← NEU
  setText("detail_fach",          article.fach          || "");  // ← NEU
  setText("detail_position",      article.position      || "");  // ← NEU
  setText("detail_niveau",        article.niveau        || "");  // ← NEU
  setText("detail_lagerort",      article.lagerort      || "");

  // Zahlen
  setText("detail_bestand",         article.bestand != null ? article.bestand : 0);
  setText("detail_mindestbestand",  article.mindestbestand != null ? article.mindestbestand : "");
  setText("detail_einheit",         article.einheit || "");
  setText("detail_wertEinzeln",     article.wertEinzeln != null ? article.wertEinzeln : "");
  setText("detail_waehrung",        article.waehrung || "");

  // Gesamtpreis (Bestand * Wert/Einheit)
  try {
    const qty = toNumber(article.bestand, 0);
    const unit = toNumber(article.wertEinzeln, 0);
    const total = qty * unit;
    const waehr = article.waehrung || "";
    setText("detail_gesamtpreis", (total > 0 ? total.toFixed(2) : "") + (waehr ? (" " + waehr) : ""));
  } catch (e) {
    setText("detail_gesamtpreis", "");
  }

  // Technische Infos
  setText("detail_technischerPlatz", article.technischerPlatz || "");
  setText("detail_equipment",        article.equipment        || "");
  setText("detail_anlage",           article.anlage           || "");
  setText("detail_notizen",          article.notizen          || "");

  // Sicherheitsstufe
  setText("detail_sicherheitsstufe", article.sicherheitsstufe || "");

  // Kostenstelle (falls bei letzter Buchung vorhanden)
  let kst = "";
  if (article.lastBooking && article.lastBooking.kostenstelle) {
    kst = article.lastBooking.kostenstelle;
  }
  setText("detail_kostenstelle", kst);

  // Datum & Uhrzeit für den Ausdruck setzen
  const now = new Date();
  const dateStr = now.toLocaleDateString("de-DE");
  const timeStr = now.toLocaleTimeString("de-DE", {
    hour: "2-digit",
    minute: "2-digit",
  });

  setText("detail_print_date", dateStr);
  setText("detail_print_time", timeStr);

  // Dialog öffnen (wie bei Artikel/Inventur)
  if (dialog) {
    try { if (root) root.scrollTop = 0; } catch (e) {}
    try { dialog.scrollTop = 0; } catch (e) {}
    openDialog(dialog);
    // manche Browser merken sich Scrollpositionen in <dialog>; daher danach nochmals zurücksetzen
    try {
      requestAnimationFrame(() => {
        try { if (root) root.scrollTop = 0; } catch (e) {}
        try { dialog.scrollTop = 0; } catch (e) {}
      });
    } catch (e) {}
  } else {
    // Fallback: ohne Dialog einfach anzeigen
    root.style.display = "block";
  }
}


  // Druck-Funktion für das Artikeldetail-Blatt
  function printArtikelDetails() {
    const root = document.getElementById("artikelDetailView");
    if (!root) return;

    const win = window.open("", "_blank", "width=900,height=700");
    if (!win) {
      window.print();
      return;
    }

    const html = `
<!DOCTYPE html>
<html lang="de">
<head>
<meta charset="utf-8">
<title>Lagerartikel – Einzeldatenblatt</title>
<style>
  body {
    font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
    margin: 15mm;
  }

  .artikel-detail-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 10px;
    padding-bottom: 6px;
    border-bottom: 1px solid #444;
  }

  .detail-logo {
    width: 40mm;
    height: auto;
  }

  .detail-header-text {
    margin-left: 8px;
  }

  .detail-company {
    font-size: 10pt;
    text-transform: uppercase;
  }

  .detail-title {
    font-size: 14pt;
    font-weight: bold;
  }

  .detail-header-meta {
    font-size: 9pt;
    text-align: right;
  }

  h2 {
    margin-top: 12mm;
    margin-bottom: 4mm;
    font-size: 13pt;
  }

  .detail-table {
    width: 100%;
    border-collapse: collapse;
    font-size: 10pt;
  }

  .detail-table th {
    text-align: left;
    padding: 2px 6px 2px 0;
    font-weight: bold;
    vertical-align: top;
    white-space: nowrap;
  }

  .detail-table td {
    padding: 2px 0;
  }

  .inv-detail-actions,
  .inv-detail-actions button {
    display: none !important;
  }

</style>
</head>
<body>
${root.innerHTML}
<script>
  window.onload = function() {
    window.print();
    window.close();
  };
<\/script>
</body>
</html>
`;

    win.document.open();
    win.document.write(html);
    win.document.close();
  }


  // Entnahme-/Warenausgangsbeleg drucken (Buchung: Entnahme oder Warenausgang)
  function printEntnahmeBeleg(article, booking) {
    if (!article || !booking) return;

    const win = window.open("", "_blank", "width=900,height=700");
    if (!win) {
      window.print();
      return;
    }

    const dt = new Date(booking.ts || Date.now());
    const dateStr = dt.toLocaleDateString("de-DE");
    const timeStr = dt.toLocaleTimeString("de-DE", {
      hour: "2-digit",
      minute: "2-digit",
    });

    const cur = (article.waehrung || "EUR").toString().trim() || "EUR";
    const einzel = toNumber(article.wertEinzeln, 0);
    const menge  = toNumber(booking.amount, 0);
    const gesamt = einzel * menge;

    const fmtMoney = (val) => {
      try {
        return new Intl.NumberFormat("de-DE", {
          style: "currency",
          currency: cur,
          minimumFractionDigits: 2,
          maximumFractionDigits: 2,
        }).format(toNumber(val, 0));
      } catch (e) {
        return String(val);
      }
    };

    const esc = (v) => {
      const s = v == null ? "" : String(v);
      return s
        .replace(/&/g, "&amp;")
        .replace(/</g, "&lt;")
        .replace(/>/g, "&gt;")
        .replace(/\"/g, "&quot;");
    };

    const docTitle = (String(booking.reason || "").trim() === "warenausgang")
      ? "Warenausgangsbeleg"
      : "Entnahmebeleg";
    const sectionLabel = (String(booking.reason || "").trim() === "warenausgang")
      ? "Warenausgang"
      : "Entnahme";

    const html = `
<!DOCTYPE html>
<html lang="de">
<head>
  <meta charset="utf-8" />
  <title>${docTitle}</title>
  <style>
    body { font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif; margin: 15mm; }
    .hdr { display:flex; justify-content:space-between; align-items:flex-start; border-bottom: 1px solid #444; padding-bottom: 6px; }
    .hdr-left { display:flex; gap:10px; align-items:flex-start; }
    .logo { width: 40mm; height:auto; }
    .company { font-size: 10pt; text-transform: uppercase; }
    .title { font-size: 16pt; font-weight: 700; margin-top: 2px; }
    .meta { font-size: 9pt; text-align:right; }
    h2 { margin: 10mm 0 4mm; font-size: 12pt; }
    table { width:100%; border-collapse: collapse; font-size: 10pt; }
    th { text-align:left; padding: 2px 8px 2px 0; white-space:nowrap; vertical-align: top; }
    td { padding: 2px 0; }
    .grid { display:grid; grid-template-columns: 1fr 1fr; gap: 10mm; }
    .sign { margin-top: 14mm; display:grid; grid-template-columns: 1fr 1fr; gap: 12mm; }
    .line { margin-top: 18mm; border-top: 1px solid #222; padding-top: 3mm; font-size: 10pt; }
    .small { color:#444; font-size: 9pt; }
  </style>
</head>
<body>
  <div class="hdr">
    <div class="hdr-left">
      <img class="logo" src="ops-logo.svg" alt="Logo" onerror="this.style.display='none'" />
      <div>
        <div class="company">[OpsDeck360]</div>
        <div class="title">${docTitle}</div>
        <div class="small">Buchungs-ID: ${esc(booking.id || "")}</div>
      </div>
    </div>
    <div class="meta">
      <div>Datum: ${esc(dateStr)}</div>
      <div>Uhrzeit: ${esc(timeStr)}</div>
    </div>
  </div>

  <div class="grid">
    <div>
      <h2>Artikel</h2>
      <table>
        <tr><th>Artikelnummer</th><td>${esc(article.artikelnummer || "")}</td></tr>
        <tr><th>Interne Artikelnummer</th><td>${esc(article.interneArtikelnummer || "")}</td></tr>
        <tr><th>Bezeichnung</th><td>${esc(article.bezeichnung || "")}</td></tr>
        <tr><th>Hersteller</th><td>${esc(article.hersteller || "")}</td></tr>
        <tr><th>Lieferant</th><td>${esc(article.lieferant || "")}</td></tr>
        <tr><th>Kategorie</th><td>${esc(article.kategorie || "")}</td></tr>
      </table>
    </div>
    <div>
      <h2>Standort</h2>
      <table>
        <tr><th>Lager</th><td>${esc(article.lager || "")}</td></tr>
        <tr><th>Lagerort</th><td>${esc(article.lagerort || "")}</td></tr>
        <tr><th>Regal</th><td>${esc(article.regal || "")}</td></tr>
        <tr><th>Fach</th><td>${esc(article.fach || "")}</td></tr>
        <tr><th>Position / Fachnr.</th><td>${esc(article.position || "")}</td></tr>
        <tr><th>Niveau</th><td>${esc(article.niveau || "")}</td></tr>
      </table>
    </div>
  </div>

  <h2>${sectionLabel}</h2>
  <table>
    <tr><th>Menge</th><td>${esc(String(menge))} ${esc(article.einheit || "")}</td></tr>
    <tr><th>Grund</th><td>${esc(booking.reason || "entnahme")}</td></tr>
    <tr><th>Kostenstelle</th><td>${esc(booking.kostenstelle || "")}</td></tr>
    <tr><th>Mitarbeiter</th><td>${esc(booking.mitarbeiter || "")}</td></tr>
    <tr><th>Notiz</th><td>${esc(booking.note || "")}</td></tr>
    <tr><th>Bestand vorher</th><td>${esc(String(booking.bestandVorher ?? ""))} ${esc(article.einheit || "")}</td></tr>
    <tr><th>Bestand nachher</th><td>${esc(String(booking.bestandNachher ?? ""))} ${esc(article.einheit || "")}</td></tr>
  </table>

  <h2>Wert</h2>
  <table>
    <tr><th>Einzelpreis</th><td>${esc(fmtMoney(einzel))}</td></tr>
    <tr><th>Gesamtwert</th><td>${esc(fmtMoney(gesamt))}</td></tr>
  </table>

  <div class="sign">
    <div>
      <div class="line">Unterschrift Ausgabe</div>
      <div class="small">Name / Datum</div>
    </div>
    <div>
      <div class="line">Unterschrift Empfänger</div>
      <div class="small">Name / Datum</div>
    </div>
  </div>

  <script>
    window.onload = function(){ window.print(); window.close(); };
  <\/script>
</body>
</html>
`;

    win.document.open();
    win.document.write(html);
    win.document.close();
  }


  // ------------------------------------------------------------------
  // Tabellen-Druck (aktuell gefilterte Ansicht)
  // ------------------------------------------------------------------
  
function formatMoney(value, currency) {
  const num = toNumber(value, NaN);
  if (!isFinite(num)) return "";
  const cur = (currency || "EUR").toString().trim() || "EUR";
  try {
    return new Intl.NumberFormat("de-DE", {
      style: "currency",
      currency: cur,
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    }).format(num);
  } catch (e) {
    // Fallback, falls Währungscode ungültig ist
    return new Intl.NumberFormat("de-DE", {
      minimumFractionDigits: 2,
      maximumFractionDigits: 2,
    }).format(num) + " " + escapeHtml(cur);
  }
}

function getSelectedArticles() {
  const ids = state && state.selectedIds ? state.selectedIds : new Set();
  if (!ids.size) return [];
  const out = [];
  (state.articles || []).forEach((a) => {
    const idStr = String(a.id);
    if (ids.has(idStr)) out.push(a);
  });
  return out;
}

function printSelectedArticles() {
  const list = getSelectedArticles();
  if (!list.length) {
        uiAlert(
      "Bitte mindestens einen Artikel auswählen.",
      "Please select at least one item.",
      "\u26a0\ufe0f Hinweis",
      "\u26a0\ufe0f Note",
      "OK",
      "OK"
    );
    return;
  }

  const win = window.open("", "_blank", "width=900,height=700");
  if (!win) {
        uiAlert(
      "Popup blockiert – bitte erlauben.",
      "Popup blocked — please allow it.",
      "\u26a0\ufe0f Hinweis",
      "\u26a0\ufe0f Note",
      "OK",
      "OK"
    );
    return;
  }

  const now = new Date();
  const dateStr = now.toLocaleDateString("de-DE");
  const timeStr = now.toLocaleTimeString("de-DE", { hour: "2-digit", minute: "2-digit" });

  let pagesHtml = "";

  list.forEach((a) => {
    const bestand = toNumber(a.bestand, 0);
    const einzelpreis = toNumber(a.wertEinzeln, 0);
    const gesamtpreis = bestand * einzelpreis;
    const waehrung = a.waehrung || "EUR";

    pagesHtml += `
<section class="print-page">
  <header class="print-header">
    <div class="brand">
      <img src="icon-512.png" alt="Logo" class="logo">
      <div class="brand-text">
        <div class="company">[OpsDeck360]</div>
        <div class="title">Lagerartikel – Einzeldatenblatt</div>
      </div>
    </div>
    <div class="meta">
      <div><span class="label">Datum:</span> ${escapeHtml(dateStr)}</div>
      <div><span class="label">Uhrzeit:</span> ${escapeHtml(timeStr)}</div>
    </div>
  </header>

  <h1>${escapeHtml(a.bezeichnung || "Lagerartikel")}</h1>

  <table class="detail">
    <tr><th>Artikelnummer</th><td>${escapeHtml(a.artikelnummer || "")}</td></tr>
    <tr><th>Interne Artikelnummer</th><td>${escapeHtml(a.interneArtikelnummer || "")}</td></tr>
    <tr><th>Hersteller</th><td>${escapeHtml(a.hersteller || "")}</td></tr>
    <tr><th>Lieferant</th><td>${escapeHtml(a.lieferant || "")}</td></tr>
    <tr><th>Kategorie</th><td>${escapeHtml(a.kategorie || "")}</td></tr>
    <tr><th>Lager</th><td>${escapeHtml(a.lager || "")}</td></tr>
    <tr><th>Lagerort</th><td>${escapeHtml(a.lagerort || "")}</td></tr>
    <tr><th>Bestand</th><td>${bestand} ${escapeHtml(a.einheit || "")}</td></tr>
    <tr><th>Einzelpreis</th><td>${formatMoney(einzelpreis, waehrung)}</td></tr>
    <tr><th>Gesamtwert</th><td>${formatMoney(gesamtpreis, waehrung)}</td></tr>
  </table>
</section>`;
  });

  const html = `<!DOCTYPE html>
<html lang="de">
<head>
<meta charset="utf-8">
<title>Artikel-Druck</title>
<style>
  :root { --border:#e5e7eb; --muted:#6b7280; --bg:#f9fafb; }
  @page { margin: 16mm; }
  body { font-family: system-ui, -apple-system, Segoe UI, Roboto, Arial, sans-serif; color:#111; }
  .print-page { page-break-after: always; }
  .print-page:last-child { page-break-after: auto; }
  .print-header { display:flex; justify-content:space-between; gap:12mm; align-items:flex-start; margin-bottom:10mm; padding-bottom:6mm; border-bottom:1px solid var(--border); }
  .brand { display:flex; gap:10px; align-items:center; }
  .logo { height:42px; width:auto; }
  .company { font-weight:700; font-size:10pt; letter-spacing:.2px; }
  .title { color:var(--muted); font-size:9pt; margin-top:1mm; }
  .meta { text-align:right; font-size:9pt; color:var(--muted); line-height:1.4; }
  .meta .label { color:#111; font-weight:600; }
  h1 { font-size:16pt; margin: 0 0 6mm 0; }
  table.detail { width:100%; border-collapse:collapse; border:1px solid var(--border); border-radius:10px; overflow:hidden; }
  th, td { padding:8px 10px; border-bottom:1px solid var(--border); vertical-align:top; }
  tr:last-child th, tr:last-child td { border-bottom:none; }
  th { width:34%; background:var(--bg); text-align:left; font-weight:700; }
</style>
</head>
<body>
${pagesHtml}
<script>
  window.onload = function(){ window.print(); window.close(); };
</script>
</body>
</html>`;

  win.document.open();
  win.document.write(html);
  win.document.close();
}


function printCurrentTable() {
    const list = getFilteredArticles();
    if (!list.length) {
            uiAlert(
        "Es gibt keine Artikel für den aktuellen Filter.",
        "No items match the current filter.",
        "\u2139\ufe0f Hinweis",
        "\u2139\ufe0f Info",
        "OK",
        "OK"
      );
      return;
    }

    // Neues Fenster für den Ausdruck
    const win = window.open("", "_blank", "width=1000,height=700");
    if (!win) {
      // Fallback, falls Popups blockiert sind
      window.print();
      return;
    }

    const now = new Date();
    const dateStr = now.toLocaleDateString("de-DE");
    const timeStr = now.toLocaleTimeString("de-DE", {
      hour: "2-digit",
      minute: "2-digit",
    });

    // Tabellendaten aufbauen
    let rowsHtml = "";
    list.forEach((a) => {
      const bestand = toNumber(a.bestand, 0);
      const einheit = a.einheit || "";
      rowsHtml += `
      <tr>
        <td>${escapeHtml(a.artikelnummer || "")}</td>
        <td>${escapeHtml(a.bezeichnung   || "")}</td>
        <td>${escapeHtml(a.hersteller    || "")}</td>
        <td>${escapeHtml(a.kategorie     || "")}</td>
        <td>${escapeHtml(a.lager         || "")}</td>
        <td>${escapeHtml(a.lagerort      || "")}</td>
        <td style="text-align:right;">${bestand} ${escapeHtml(einheit)}</td>
      </tr>`;
    });

    const html = `
<!DOCTYPE html>
<html lang="de">
<head>
<meta charset="utf-8">
<title>Lager – Tabellenübersicht</title>
<style>
  body {
    font-family: system-ui, -apple-system, BlinkMacSystemFont, "Segoe UI", sans-serif;
    margin: 15mm;
    background: #ffffff;
    color: #111827;
  }

  .print-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 12mm;
    padding-bottom: 4mm;
    border-bottom: 1px solid #9ca3af;
  }

  .print-logo {
    width: 40mm;
    height: auto;
    object-fit: contain;
  }

  .print-header-text {
    margin-left: 10px;
  }

  .print-company {
    font-size: 10pt;
    text-transform: uppercase;
    letter-spacing: 0.08em;
    color: #4b5563;
  }

  .print-title {
    font-size: 16pt;
    font-weight: 700;
    margin-top: 2mm;
  }

  .print-meta {
    font-size: 9pt;
    text-align: right;
    color: #4b5563;
  }

  h2 {
    font-size: 13pt;
    margin: 0 0 6mm 0;
  }

  table {
    width: 100%;
    border-collapse: collapse;
    font-size: 9pt;
  }

  thead th {
    text-align: left;
    padding: 4px 6px;
    border-bottom: 1px solid #9ca3af;
    background: #e5e7eb;
  }

  tbody td {
    padding: 3px 6px;
    border-bottom: 1px solid #e5e7eb;
    vertical-align: top;
  }

  tbody tr:nth-child(even) td {
    background: #f9fafb;
  }

  .col-number {
    text-align: right;
    white-space: nowrap;
  }

  @page {
    margin: 15mm;
  }
</style>
</head>
<body>
  <header class="print-header">
    <div style="display:flex;align-items:center;">
      <!-- Hier dein echtes Firmenlogo eintragen -->
      <img src="firma-logo.png" alt="Firmenlogo" class="print-logo">
      <div class="print-header-text">
        <div class="print-company">Deine Firma</div>
        <div class="print-title">Lagerübersicht</div>
      </div>
    </div>
    <div class="print-meta">
      <div>Gedruckt am ${dateStr}</div>
      <div>Uhrzeit: ${timeStr}</div>
    </div>
  </header>

  <h2>Aktuelle Tabellenansicht (${list.length} Artikel)</h2>

  <table>
    <thead>
      <tr>
        <th>${escapeHtml(getColumnLabel("artikelnummer"))}</th>
        <th>${escapeHtml(getColumnLabel("bezeichnung"))}</th>
        <th>${escapeHtml(getColumnLabel("hersteller"))}</th>
        <th>${escapeHtml(getColumnLabel("kategorie"))}</th>
        <th>${escapeHtml(getColumnLabel("lager"))}</th>
        <th>${escapeHtml(getColumnLabel("lagerort"))}</th>
        <th class="col-number">${escapeHtml(getColumnLabel("bestand"))}</th>
      </tr>
    </thead>
    <tbody>
      ${rowsHtml}
    </tbody>
  </table>

  <script>
    window.onload = function() {
      window.print();
      window.close();
    };
  <\/script>
</body>
</html>
`;

    win.document.open();
    win.document.write(html);
    win.document.close();
  }


  // Funktion global verfügbar machen für das onclick="" im HTML
  window.printArtikelDetails = printArtikelDetails;


  // =====================================================================
  // BANF / Bestellung – minimal-invasiv (läuft parallel zur bestehenden Logik)
  // =====================================================================

  function computeSuggestedQty(article) {
    const best = toNumber(article.bestand, 0);
    const min  = toNumber(article.mindestbestand, 0);
    const diff = min - best;
    return diff > 0 ? diff : 0;
  }

  function openBanfFromArticles(articles) {
    // Wenn nichts übergeben: nimm alle Artikel unter Mindestbestand (Vorschlag)
    const rawList = Array.isArray(articles) && articles.length
      ? articles
      : (state.articles || []).filter(a => computeSuggestedQty(a) > 0);

    // Gesperrte Artikel dürfen nicht bestellt werden (SAP-ähnlich: Beschaffung/Zugang gesperrt)
    const blocked = rawList.filter(a => isReceiptBlocked(a));
    const list = rawList.filter(a => !isReceiptBlocked(a));

    if (blocked.length && list.length) {
            uiAlert(
        "Hinweis: " + blocked.length + " Artikel sind für Zugang/BANF gesperrt und wurden aus der BANF entfernt.",
        "Note: " + blocked.length + " item(s) are blocked for receiving/requisition and were removed from the requisition.",
        "\ud83e\uddfe BANF / Bestellung",
        "\ud83e\uddfe Requisition / Order",
        "OK",
        "OK"
      );
    }

    if (!list.length) {
      if (blocked.length) {
                uiAlert(
          "BANF nicht möglich: Alle ausgewählten Artikel sind für Zugang/BANF gesperrt.\n\n" +
                    "Tipp: Sperre im Aktionen-Menü wieder aufheben.",
          "Cannot create requisition: all selected items are blocked for receiving/requisition.\n\n" +
                    "Tip: Remove the block in the Actions menu.",
          "\ud83e\uddfe BANF / Bestellung",
          "\ud83e\uddfe Requisition / Order",
          "OK",
          "OK"
        );
      } else {
                uiAlert(
          "Keine Artikel für BANF gefunden.\n\nTipp: Markiere Artikel in der Tabelle oder setze Mindestbestand/Bestand so, dass ein Bedarf entsteht.",
          "No items found for requisition.\n\nTip: Select items in the table or adjust min stock/stock so that a demand is created.",
          "\ud83e\uddfe BANF / Bestellung",
          "\ud83e\uddfe Requisition / Order",
          "OK",
          "OK"
        );
      }
      return;
    }

    const banfDoc = {
      id: "banf_" + Date.now().toString(36) + "_" + Math.random().toString(36).slice(2, 7),
      ts: nowIso(),
      status: "Entwurf",
      requester: "",
      costCenter: "",
      supplier: "",
      needBy: "",
      note: "",
      lines: list.map(a => ({
        articleId: a.id,
        artikelnummer: a.artikelnummer || "",
        interneArtikelnummer: a.interneArtikelnummer || "",
        bezeichnung: a.bezeichnung || "",
        lieferant: a.lieferant || "",
        menge: computeSuggestedQty(a),
        einheit: a.einheit || "",
        einzelpreis: toNumber(a.wertEinzeln, 0),
        waehrung: a.waehrung || "EUR"
      }))
    };

    // Speichern + Audit
    state.banf = state.banf || [];
    state.banf.unshift(banfDoc);
    saveBanf(state.banf);
    addAuditEvent("banf.created", { banfId: banfDoc.id, lines: banfDoc.lines.length });

    // Direkt drucken (mit Logo/Datum/Unterschrift)
    printBanfDoc(banfDoc);

    // Optional JSON export
    uiConfirm(
      "BANF wurde erstellt.\n\nSoll zusätzlich ein JSON-Export (BANF + SAP-Payload Stub) heruntergeladen werden?",
      "Requisition was created.\n\nDownload an additional JSON export (requisition + SAP payload stub)?",
      "⬇️ Export",
      "⬇️ Export",
      "OK",
      "OK",
      "Abbrechen",
      "Cancel"
    ).then((yes) => {
      if (yes) {
        downloadJson("banf-" + banfDoc.id + ".json", banfDoc);
        downloadJson(
          "banf-" + banfDoc.id + "-sap-payload.json",
          buildSapPayloadFromBanf(banfDoc)
        );
      }
    });
  }

  function downloadJson(filename, obj) {
    try {
      const json = JSON.stringify(obj, null, 2);
      const blob = new Blob([json], { type: "application/json;charset=utf-8" });
      const url = URL.createObjectURL(blob);
      const a = document.createElement("a");
      a.href = url;
      a.download = filename;
      document.body.appendChild(a);
      a.click();
      document.body.removeChild(a);
      URL.revokeObjectURL(url);
    } catch (e) {
      console.warn("[lager] JSON Export fehlgeschlagen:", e);
    }
  }

  function buildSapPayloadFromBanf(banfDoc) {
    // Stub: Bitte an euer SAP-Setup anpassen (z.B. S/4HANA Purchase Requisition API via Proxy)
    return {
      documentDate: banfDoc.ts,
      requester: banfDoc.requester || "",
      costCenter: banfDoc.costCenter || "",
      supplier: banfDoc.supplier || "",
      needBy: banfDoc.needBy || "",
      note: banfDoc.note || "",
      items: (banfDoc.lines || []).map((l, i) => ({
        itemNo: String(i + 1).padStart(5, "0"),
        material: l.artikelnummer || l.interneArtikelnummer || "",
        shortText: l.bezeichnung || "",
        quantity: Number(l.menge || 0),
        unit: l.einheit || "",
        supplier: l.lieferant || banfDoc.supplier || "",
        price: Number(l.einzelpreis || 0),
        currency: l.waehrung || "EUR",
        accountAssignment: { costCenter: banfDoc.costCenter || "" }
      }))
    };
  }

  function printBanfDoc(banfDoc) {
    const d = new Date();
    const dateStr = d.toLocaleDateString("de-DE");
    const timeStr = d.toLocaleTimeString("de-DE", { hour: "2-digit", minute: "2-digit" });

    const rows = (banfDoc.lines || []).map((l, idx) => {
      const sum = toNumber(l.menge, 0) * toNumber(l.einzelpreis, 0);
      return `
        <tr>
          <td style="text-align:right;">${idx + 1}</td>
          <td>${escapeHtml(l.artikelnummer || "")}</td>
          <td>${escapeHtml(l.bezeichnung || "")}</td>
          <td>${escapeHtml(l.lieferant || "")}</td>
          <td style="text-align:right;">${escapeHtml(String(toNumber(l.menge, 0)))}</td>
          <td>${escapeHtml(l.einheit || "")}</td>
          <td style="text-align:right;">${l.einzelpreis ? escapeHtml(toNumber(l.einzelpreis,0).toFixed(2)) : ""}</td>
          <td>${escapeHtml(l.waehrung || "EUR")}</td>
          <td style="text-align:right;">${sum ? escapeHtml(sum.toFixed(2)) : ""}</td>
        </tr>`;
    }).join("");

    const html = `<!doctype html>
<html lang="de">
<head>
<meta charset="utf-8">
<title>BANF – ${escapeHtml(banfDoc.id || "")}</title>
<style>
  body { font-family: system-ui,-apple-system,BlinkMacSystemFont,"Segoe UI",sans-serif; margin: 15mm; }
  .hdr { display:flex; justify-content:space-between; align-items:flex-start; border-bottom: 1px solid #444; padding-bottom: 6px; }
  .hdr-left { display:flex; gap:10px; align-items:flex-start; }
  .logo { width: 40mm; height:auto; }
  .company { font-size: 10pt; text-transform: uppercase; }
  .title { font-size: 16pt; font-weight: 700; margin-top: 2px; }
  .meta { font-size: 9pt; text-align:right; }
  h2 { margin: 10mm 0 4mm; font-size: 12pt; }
  table { width:100%; border-collapse: collapse; font-size: 10pt; }
  thead th { text-align:left; padding: 4px 6px; border-bottom: 1px solid #9ca3af; background: #e5e7eb; }
  tbody td { padding: 3px 6px; border-bottom: 1px solid #e5e7eb; vertical-align: top; }
  tbody tr:nth-child(even) td { background: #f9fafb; }
  .sign { margin-top: 14mm; display:grid; grid-template-columns: 1fr 1fr; gap: 12mm; }
  .line { margin-top: 18mm; border-top: 1px solid #222; padding-top: 3mm; font-size: 10pt; }
</style>
</head>
<body>
  <div class="hdr">
    <div class="hdr-left">
      <img class="logo" src="${escapeHtml(APP.logo)}" alt="Logo" onerror="this.style.display='none'"/>
      <div>
        <div class="company">${escapeHtml(APP.company)}</div>
        <div class="title">BANF / Bestellung (Entwurf)</div>
        <div class="meta" style="text-align:left;font-size:9pt;opacity:.85;">ID: ${escapeHtml(banfDoc.id || "-")}</div>
      </div>
    </div>
    <div class="meta">
      <div>Datum: ${escapeHtml(dateStr)}</div>
      <div>Uhrzeit: ${escapeHtml(timeStr)}</div>
    </div>
  </div>

  <h2>Positionen</h2>
  <table>
    <thead>
      <tr>
        <th style="width:16mm;text-align:right;">Pos</th>
        <th>Artikelnummer</th>
        <th>Bezeichnung</th>
        <th>Lieferant</th>
        <th style="width:18mm;text-align:right;">Menge</th>
        <th style="width:18mm;">Einheit</th>
        <th style="width:22mm;text-align:right;">Einzel</th>
        <th style="width:18mm;">Währ.</th>
        <th style="width:26mm;text-align:right;">Summe</th>
      </tr>
    </thead>
    <tbody>${rows}</tbody>
  </table>

  <div class="sign">
    <div class="line">Datum</div>
    <div class="line">Unterschrift</div>
  </div>

  <script>
    window.onload = function(){ window.print(); window.close(); };
  </script>
</body>
</html>`;

    const win = window.open("", "_blank", "width=980,height=760");
    if (!win) {
            uiAlert(
        "Popup-Blocker aktiv – bitte Popups erlauben, um die BANF zu drucken.",
        "Popup blocker is active — please allow popups to print the requisition.",
        "\ud83e\uddfe BANF / Bestellung",
        "\ud83e\uddfe Requisition / Order",
        "OK",
        "OK"
      );
      return;
    }
    win.document.open();
    win.document.write(html);
    win.document.close();
  }



  document.addEventListener("DOMContentLoaded", () => {
    // Ensure Offline-Client finished login + initial sync before the app starts.
    const ready = window.__opsOfflineReady || (window.OpsOffline && window.OpsOffline.ensureLoggedIn ? window.OpsOffline.ensureLoggedIn() : Promise.resolve(null));
    Promise.resolve(ready)
      .catch(() => null)
      .finally(() => {
        try { init(); } catch (e) { console.error(e); }
      });
  });


})();



  

