// ops-lang-bridge.js
// Fix: Language button in header toggles DE/EN reliably.
// Works even if the header was generated by another script.

(() => {
  "use strict";

  function hasI18n() {
    return typeof window.setOpsLang === "function" && typeof window.getOpsLang === "function";
  }

  function current() {
    try { return (window.getOpsLang ? window.getOpsLang() : "de"); } catch (_) { return "de"; }
  }

  function setLang(lang) {
    if (!hasI18n()) return;
    try { window.setOpsLang(lang); } catch (_) {}
    // Some UIs render late -> ensure re-apply
    try { window.applyI18n && window.applyI18n(document); } catch (_) {}
  }

  function updateHeaderBadge() {
    const header = document.querySelector(".app-header") || document.querySelector("header");
    if (!header) return;

    const lang = String(current()).toUpperCase();

    // Try to find a short "EN/DE" badge/button in header
    const candidates = header.querySelectorAll("button, a, span, div");
    for (const el of candidates) {
      const txt = (el.textContent || "").trim();
      if (txt === "EN" || txt === "DE") {
        el.textContent = lang; // show current language
        return;
      }
    }
  }

  // Capture clicks in header on "EN"/"DE" badge
  document.addEventListener("click", (e) => {
    const header = document.querySelector(".app-header") || document.querySelector("header");
    if (!header) return;

    const el = e.target && e.target.closest ? e.target.closest("button, a, span, div") : null;
    if (!el || !header.contains(el)) return;

    const txt = (el.textContent || "").trim().toUpperCase();

    if (txt === "EN") {
      e.preventDefault();
      e.stopPropagation();
      setLang("en");
      updateHeaderBadge();
    } else if (txt === "DE") {
      e.preventDefault();
      e.stopPropagation();
      setLang("de");
      updateHeaderBadge();
    }
  }, true);

  // If your code dispatches ops:langchange, keep badge in sync
  window.addEventListener("ops:langchange", () => updateHeaderBadge());

  // Initial sync after header injected
  const boot = () => {
    updateHeaderBadge();
    // second pass for late header rendering
    setTimeout(updateHeaderBadge, 300);
    setTimeout(updateHeaderBadge, 1000);
  };

  if (document.readyState === "loading") {
    document.addEventListener("DOMContentLoaded", boot, { once: true });
  } else {
    boot();
  }
})();
