// ops-lang-fix.js — makes EN/DE toggle actually switch translations via setOpsLang()
// Safe: no changes to existing modules required.

(() => {
  "use strict";

  function ready(fn) {
    if (document.readyState === "loading") {
      document.addEventListener("DOMContentLoaded", fn, { once: true });
    } else fn();
  }

  function getLang() {
    try { return (window.getOpsLang ? window.getOpsLang() : "de"); } catch (_) { return "de"; }
  }

  function setLang(lang) {
    if (typeof window.setOpsLang !== "function") return false;
    try { window.setOpsLang(lang); } catch (_) {}
    // Safety re-apply (some UIs render late)
    try { window.applyI18n && window.applyI18n(document); } catch (_) {}
    setTimeout(() => { try { window.applyI18n && window.applyI18n(document); } catch (_) {} }, 50);
    setTimeout(() => { try { window.applyI18n && window.applyI18n(document); } catch (_) {} }, 300);
    return true;
  }

  function isLangElement(el) {
    if (!el) return false;
    const id = (el.id || "").toLowerCase();
    const cls = (el.className || "").toString().toLowerCase();
    const txt = (el.textContent || "").trim().toUpperCase();

    // Typical cases
    if (id.includes("lang")) return true;
    if (cls.includes("lang")) return true;
    if (txt === "EN" || txt === "DE") return true;

    return false;
  }

  function updateBadge(header) {
    const lang = getLang().toUpperCase();
    const candidates = header.querySelectorAll("button,a,span,div");
    for (const el of candidates) {
      const txt = (el.textContent || "").trim().toUpperCase();
      if (txt === "EN" || txt === "DE" || (el.id || "").toLowerCase().includes("lang")) {
        // show current language
        el.textContent = lang;
      }
    }
  }

  function bind(header) {
    if (!header || header.__opsLangBound) return;
    header.__opsLangBound = true;

    header.addEventListener("click", (e) => {
      const el = e.target && e.target.closest ? e.target.closest("button,a,span,div") : null;
      if (!el || !header.contains(el)) return;
      if (!isLangElement(el)) return;

      // toggle language no matter what the badge currently shows
      e.preventDefault();
      e.stopPropagation();

      const next = getLang() === "de" ? "en" : "de";
      if (setLang(next)) updateBadge(header);
    }, true);

    // keep badge synced
    window.addEventListener("ops:langchange", () => updateBadge(header));
    updateBadge(header);
  }

  ready(() => {
    const header = document.querySelector(".app-header") || document.querySelector("header");
    if (header) bind(header);

    // If header is injected later, observe and bind again
    const mo = new MutationObserver(() => {
      const h = document.querySelector(".app-header") || document.querySelector("header");
      if (h) bind(h);
    });
    try { mo.observe(document.documentElement, { childList: true, subtree: true }); } catch (_) {}
  });
})();
