/* ops-offline-client.js
   Offline-first: Local Windows Python server (FastAPI)
   - Login dialog (DE/EN)
   - Sync selected localStorage keys to server (debounced)
   - Sync server -> localStorage on load

   Load this before ops-lagerverwaltung.js so the initial state comes from the server.
*/

(() => {
  "use strict";

  if (window.__opsOfflineClientLoaded) return;
  window.__opsOfflineClientLoaded = true;

  const API_BASE = (window.OpsDeck && window.OpsDeck.apiBase) || "/api";

  const STORE_KEYS = [
    "opsdeck_lager_v1",
    "opsdeck_lager_bookings_v1",
    "opsdeck_lager_banf_v1",
    "opsdeck_lager_audit_v1",
    "opsdeck_lager_visible_cols_v1",
    "opsdeck_lager_col_labels_v1",
    "opsdeck_lager_seals_v1",
    "opsdeck_lager_receipts_v1"
  ];

  const SYNC = {
    serverRev: 0,
    initDone: false,
    dirtyKeys: new Set(),
    timer: null,
    syncing: false,
    lastErr: null
  };

  // --- i18n helpers --------------------------------------------------
  function getLang() {
    try {
      if (typeof window.getOpsLang === "function") return window.getOpsLang();
    } catch (_) {}
    const l = document.documentElement.getAttribute("lang") || "de";
    return String(l).toLowerCase().startsWith("en") ? "en" : "de";
  }

  function t(de, en) {
    return getLang() === "en" ? (en ?? de) : (de ?? en);
  }

  // --- UI helpers (match your existing style) -------------------------
  function uiAlert(de, en, titleDe, titleEn) {
    try {
      if (window.OPS_UI && typeof window.OPS_UI.alert === "function") {
        return window.OPS_UI.alert(de, en, titleDe || "ℹ️", titleEn || "ℹ️");
      }
      if (typeof window.uiAlert === "function") {
        return window.uiAlert(de, en, titleDe || "ℹ️", titleEn || "ℹ️");
      }
    } catch (_) {}
    alert((getLang() === "en" ? (titleEn || "Info") : (titleDe || "Info")) + "\n\n" + t(de, en));
  }

  function ensureLoginDialog() {
    let dlg = document.getElementById("opsLoginDialog");
    if (dlg) return dlg;

    dlg = document.createElement("dialog");
    dlg.id = "opsLoginDialog";
    dlg.className = "dialog";

    dlg.innerHTML = `
      <div class="card" style="max-width:520px;width:min(520px,96vw);padding:16px 18px 18px;">
        <div style="display:flex;align-items:flex-start;justify-content:space-between;gap:12px;">
          <div>
            <h3 style="margin:0;font-size:18px;">${t("Anmeldung", "Login")}</h3>
            <p class="tiny" style="margin:6px 0 0;opacity:.85;">${t("Lokaler Offline-Server", "Local offline server")}</p>
          </div>
          <button type="button" class="btn btn-ghost" id="opsLoginCloseX" aria-label="close">✕</button>
        </div>

        <form id="opsLoginForm" style="margin:14px 0 0;">
          <label class="tiny" style="display:block;">${t("Benutzername", "Username")}
            <input id="opsLoginUser" autocomplete="username" style="width:100%;padding:8px 10px;margin-top:6px;" />
          </label>

          <label class="tiny" style="display:block;margin-top:10px;">${t("Passwort", "Password")}
            <input id="opsLoginPass" type="password" autocomplete="current-password" style="width:100%;padding:8px 10px;margin-top:6px;" />
          </label>

          <div id="opsLoginErr" class="tiny" style="color:#b00020;margin-top:10px;display:none;"></div>

          <div style="display:flex;justify-content:flex-end;gap:10px;margin-top:14px;">
            <button type="button" class="btn btn-ghost" id="opsLoginCancel">${t("Abbrechen", "Cancel")}</button>
            <button type="submit" class="btn btn-primary" id="opsLoginBtn">${t("Anmelden", "Login")}</button>
          </div>

          <p class="tiny" style="opacity:.8;margin:12px 0 0;">
            ${t("Standard: Benutzer 'admin' / Passwort 'admin' (danach ändern).", "Default: user 'admin' / password 'admin' (change afterwards).")}
          </p>
        </form>
      </div>
    `;

    document.body.appendChild(dlg);

    const close = () => { try { dlg.close(); } catch (_) { dlg.removeAttribute("open"); } };
    dlg.querySelector("#opsLoginCloseX")?.addEventListener("click", close);
    dlg.querySelector("#opsLoginCancel")?.addEventListener("click", close);

    return dlg;
  }

  function showLoginError(dlg, msg) {
    const el = dlg.querySelector("#opsLoginErr");
    if (!el) return;
    el.textContent = msg || "";
    el.style.display = msg ? "block" : "none";
  }

  // --- API ------------------------------------------------------------
  async function apiFetch(path, options) {
    const opts = Object.assign({
      method: "GET",
      credentials: "include",
      headers: { "Content-Type": "application/json" }
    }, options || {});

    if (opts.body && typeof opts.body !== "string" && !(opts.body instanceof FormData)) {
      opts.body = JSON.stringify(opts.body);
    }

    // For FormData, let browser set boundary
    if (opts.body instanceof FormData) {
      if (opts.headers) delete opts.headers["Content-Type"]; 
    }

    const res = await fetch(API_BASE + path, opts);
    const isJson = (res.headers.get("content-type") || "").includes("application/json");
    const data = isJson ? await res.json().catch(() => ({})) : await res.text().catch(() => "");
    if (!res.ok) {
      const msg = (data && data.detail) ? data.detail : (typeof data === "string" ? data : "Request failed");
      const err = new Error(msg);
      err.status = res.status;
      err.data = data;
      throw err;
    }
    return data;
  }

  async function me() {
    try {
      return await apiFetch("/me");
    } catch (e) {
      if (e && e.status === 401) return null;
      throw e;
    }
  }

  async function login(username, password) {
    return await apiFetch("/auth/login", { method: "POST", body: { username, password } });
  }

  async function logout() {
    try {
      await apiFetch("/auth/logout", { method: "POST", body: {} });
    } catch (_) {}
  }

  // --- Sync logic ------------------------------------------------------
  function markChanged(key) {
    if (!STORE_KEYS.includes(key)) return;
    if (!SYNC.initDone) return; // avoid syncing during initial hydration
    SYNC.dirtyKeys.add(key);
    scheduleSync();
  }

  function scheduleSync() {
    if (SYNC.timer) clearTimeout(SYNC.timer);
    SYNC.timer = setTimeout(() => { syncToServer().catch(() => {}); }, 700);
  }

  function readKey(key) {
    try { return localStorage.getItem(key); } catch (_) { return null; }
  }

  function writeKey(key, value) {
    try { localStorage.setItem(key, value); } catch (_) {}
  }

  async function syncFromServer() {
    const payload = await apiFetch("/store/all");
    if (!payload || !payload.data) return;

    // Avoid triggering write events during hydration
    SYNC.initDone = false;

    try {
      const data = payload.data;
      for (const k of STORE_KEYS) {
        if (!(k in data)) continue;
        writeKey(k, JSON.stringify(data[k]));
      }
      SYNC.serverRev = payload.rev || 0;
    } finally {
      // next tick: allow sync
      setTimeout(() => { SYNC.initDone = true; }, 0);
    }
  }

  function collectLocalData() {
    const out = {};
    for (const k of STORE_KEYS) {
      const raw = readKey(k);
      if (raw == null || raw === "") {
        out[k] = k.includes("col_labels") || k.includes("visible_cols") ? {} : [];
        continue;
      }
      try { out[k] = JSON.parse(raw); }
      catch { out[k] = raw; }
    }
    return out;
  }

  async function isServerEmpty() {
    const payload = await apiFetch("/store/all");
    const data = payload && payload.data ? payload.data : {};
    const hasAny = STORE_KEYS.some((k) => {
      const v = data[k];
      if (Array.isArray(v)) return v.length > 0;
      if (v && typeof v === "object") return Object.keys(v).length > 0;
      return false;
    });
    return { empty: !hasAny, rev: payload.rev || 0, data };
  }

  async function migrateLocalToServerIfNeeded() {
    // If server is empty but local has data -> push local to server (one-time migration)
    const server = await isServerEmpty();
    const local = collectLocalData();

    const localHas = STORE_KEYS.some((k) => {
      const v = local[k];
      if (Array.isArray(v)) return v.length > 0;
      if (v && typeof v === "object") return Object.keys(v).length > 0;
      return false;
    });

    if (server.empty && localHas) {
      try {
        await apiFetch("/store/all", {
          method: "PUT",
          body: { rev: server.rev || 0, data: local, force: false }
        });
        // Refresh serverRev
        const newest = await apiFetch("/store/all");
        SYNC.serverRev = newest.rev || 0;
      } catch (e) {
        console.warn("[offline] migration failed:", e);
      }
    }
  }

  async function syncToServer() {
    if (SYNC.syncing) return;
    if (!SYNC.initDone) return;
    if (!SYNC.dirtyKeys.size) return;

    SYNC.syncing = true;
    const keys = Array.from(SYNC.dirtyKeys);
    SYNC.dirtyKeys.clear();

    try {
      const data = collectLocalData();
      const res = await apiFetch("/store/all", {
        method: "PUT",
        body: { rev: SYNC.serverRev || 0, data, force: false }
      });
      SYNC.serverRev = (res && res.rev) ? res.rev : (SYNC.serverRev + 1);
      SYNC.lastErr = null;
    } catch (e) {
      SYNC.lastErr = e;
      // If conflict: pull server and re-apply local by retrying once
      if (e && e.status === 409) {
        try {
          await syncFromServer();
          // re-mark changed keys to re-send
          keys.forEach((k) => SYNC.dirtyKeys.add(k));
          scheduleSync();
        } catch (_) {}
      } else if (e && e.status === 401) {
        // session expired
        try { await ensureLoggedIn(); } catch (_) {}
      }
    } finally {
      SYNC.syncing = false;
    }
  }

  // --- Intercept localStorage writes (minimal-intrusive) ---------------
  function patchLocalStorage() {
    try {
      if (!window.localStorage) return;
      const ls = window.localStorage;
      if (ls.__opsPatched) return;
      ls.__opsPatched = true;

      const origSet = ls.setItem.bind(ls);
      const origRem = ls.removeItem.bind(ls);
      const origClr = ls.clear.bind(ls);

      ls.setItem = function (k, v) {
        const key = String(k);
        origSet(key, String(v));
        try {
          window.dispatchEvent(new CustomEvent("opsLocalStorageChanged", { detail: { key } }));
        } catch (_) {}
      };

      ls.removeItem = function (k) {
        const key = String(k);
        origRem(key);
        try { window.dispatchEvent(new CustomEvent("opsLocalStorageChanged", { detail: { key } })); } catch (_) {}
      };

      ls.clear = function () {
        origClr();
        try { window.dispatchEvent(new CustomEvent("opsLocalStorageChanged", { detail: { key: "*" } })); } catch (_) {}
      };
    } catch (e) {
      console.warn("[offline] localStorage patch failed:", e);
    }
  }

  async function ensureLoggedIn() {
    const u = await me();
    if (u) return u;

    const dlg = ensureLoginDialog();
    showLoginError(dlg, "");

    const userEl = dlg.querySelector("#opsLoginUser");
    const passEl = dlg.querySelector("#opsLoginPass");
    const form = dlg.querySelector("#opsLoginForm");
    const btn = dlg.querySelector("#opsLoginBtn");

    const doLogin = async () => {
      const username = (userEl?.value || "").trim();
      const password = (passEl?.value || "");

      if (!username || !password) {
        showLoginError(dlg, t("Bitte Benutzername und Passwort eingeben.", "Please enter username and password."));
        return;
      }

      if (btn) {
        btn.disabled = true;
        btn.textContent = t("Anmelden…", "Logging in…");
      }

      try {
        await login(username, password);
        try { dlg.close(); } catch (_) { dlg.removeAttribute("open"); }
      } catch (e) {
        showLoginError(dlg, t("Login fehlgeschlagen.", "Login failed."));
      } finally {
        if (btn) {
          btn.disabled = false;
          btn.textContent = t("Anmelden", "Login");
        }
      }
    };

    if (form && !form.dataset.bound) {
      form.dataset.bound = "1";
      form.addEventListener("submit", (e) => {
        e.preventDefault();
        doLogin();
      });
    }

    // open and wait for close
    try { dlg.showModal(); } catch (_) { dlg.setAttribute("open", ""); }
    userEl && userEl.focus && userEl.focus();

    // Wait until user is authenticated or dialog closed
    return await new Promise((resolve, reject) => {
      const start = Date.now();
      const tick = async () => {
        try {
          const u2 = await me();
          if (u2) return resolve(u2);
        } catch (e) {
          return reject(e);
        }
        const isOpen = dlg.open || dlg.hasAttribute("open");
        if (!isOpen) return resolve(null);
        if (Date.now() - start > 5 * 60 * 1000) return resolve(null);
        setTimeout(tick, 250);
      };
      tick();
    });
  }

  async function boot() {
    patchLocalStorage();

    window.addEventListener("opsLocalStorageChanged", (ev) => {
      const key = ev && ev.detail ? ev.detail.key : "";
      if (key === "*") {
        STORE_KEYS.forEach((k) => markChanged(k));
      } else {
        markChanged(String(key || ""));
      }
    });

    const user = await ensureLoggedIn();
    if (!user) {
      uiAlert(
        "Nicht angemeldet – Offline-Sync ist deaktiviert.",
        "Not logged in – offline sync is disabled.",
        "🔒 Login",
        "🔒 Login"
      );
      return;
    }

    // Pull from server BEFORE other scripts read localStorage
    await migrateLocalToServerIfNeeded();

    // If server empty but browser had data (first run migration)
   await syncFromServer();

    // Expose a tiny API for other scripts
    window.OpsOffline = {
      apiBase: API_BASE,
      apiFetch,
      ensureLoggedIn,
      me,
      logout,
      syncNow: syncToServer,
      pullNow: syncFromServer,
      get serverRev() { return SYNC.serverRev; },
      get lastError() { return SYNC.lastErr; }
    }
  }

document.addEventListener("DOMContentLoaded", () => {
  try { init(); } catch (e) { console.error(e); }
});

})();
