// ops-ui-dialogs.js
(() => {
  "use strict";

  const t = (de, en) => {
    try {
      if (typeof window.tLang === "function") return window.tLang(de, en);
    } catch (_) {}
    const isEn = String(document.documentElement.lang || "").toLowerCase().startsWith("en");
    return isEn ? (en ?? de) : (de ?? en);
  };

  function openDialog(dlg) {
    try { dlg.showModal(); return; } catch (_) {}
    try { dlg.setAttribute("open", ""); } catch (_) {}
  }
  function closeDialog(dlg) {
    try { dlg.close(); } catch (_) {}
    try { dlg.removeAttribute("open"); } catch (_) {}
  }

  function ensureDlg() {
    let dlg = document.getElementById("opsUiDialog");
    if (dlg) return dlg;

    dlg = document.createElement("dialog");
    dlg.id = "opsUiDialog";
    dlg.className = "dialog";
    dlg.innerHTML = `
      <div class="card" style="max-width:560px;width:92vw;">
        <h3 id="opsUiTitle" style="margin:0 0 10px 0;font-size:18px;"></h3>
        <div id="opsUiMsg" style="white-space:pre-wrap;line-height:1.45;"></div>

        <div id="opsUiInputWrap" style="margin-top:10px; display:none;">
          <label id="opsUiInputLabel" style="display:block; font-size:13px; opacity:.9; margin-bottom:6px;"></label>
          <input id="opsUiInput" class="input" style="width:100%;" />
          <div id="opsUiError" style="display:none;margin-top:6px;font-size:12px;opacity:.9;"></div>
        </div>

        <div style="display:flex;justify-content:flex-end;gap:8px;margin-top:14px;">
          <button type="button" class="btn" id="opsUiCancel">Cancel</button>
          <button type="button" class="btn btn-primary" id="opsUiOk">OK</button>
        </div>
      </div>
    `;
    document.body.appendChild(dlg);

    // backdrop click = cancel
    dlg.addEventListener("click", (e) => {
      if (e.target === dlg) {
        dlg.__opsResolve && dlg.__opsResolve(dlg.__opsCancelValue);
        dlg.__opsResolve = null;
        closeDialog(dlg);
      }
    });

    // ESC = cancel
    dlg.addEventListener("cancel", (e) => {
      e.preventDefault();
      dlg.__opsResolve && dlg.__opsResolve(dlg.__opsCancelValue);
      dlg.__opsResolve = null;
      closeDialog(dlg);
    });

    return dlg;
  }

  function runDialog({ mode, deMsg, enMsg, titleDe, titleEn, okDe, okEn, cancelDe, cancelEn, inputLabelDe, inputLabelEn, placeholderDe, placeholderEn, defaultValue, inputType, validate }) {
    const dlg = ensureDlg();

    const titleEl = dlg.querySelector("#opsUiTitle");
    const msgEl = dlg.querySelector("#opsUiMsg");
    const okBtn = dlg.querySelector("#opsUiOk");
    const cancelBtn = dlg.querySelector("#opsUiCancel");

    const wrap = dlg.querySelector("#opsUiInputWrap");
    const inLabel = dlg.querySelector("#opsUiInputLabel");
    const input = dlg.querySelector("#opsUiInput");
    const err = dlg.querySelector("#opsUiError");

    titleEl.textContent = t(titleDe || "ℹ️ Hinweis", titleEn || "ℹ️ Info");
    msgEl.textContent = t(deMsg || "", enMsg || "");
    okBtn.textContent = t(okDe || "OK", okEn || "OK");
    cancelBtn.textContent = t(cancelDe || "Abbrechen", cancelEn || "Cancel");

    const showInput = mode === "prompt";
    wrap.style.display = showInput ? "" : "none";
    cancelBtn.style.display = (mode === "alert") ? "none" : "";

    dlg.__opsCancelValue = (mode === "prompt") ? null : false;

    if (showInput) {
      inLabel.textContent = t(inputLabelDe || "Eingabe", inputLabelEn || "Input");
      input.type = inputType || "text";
      input.placeholder = t(placeholderDe || "", placeholderEn || "");
      input.value = (defaultValue ?? "");
      err.style.display = "none";
      err.textContent = "";
      setTimeout(() => { try { input.focus(); input.select(); } catch (_) {} }, 0);
    }

    const cleanup = () => {
      okBtn.onclick = null;
      cancelBtn.onclick = null;
      dlg.__opsResolve = null;
    };

    return new Promise((resolve) => {
      // wenn ein alter Dialog noch offen ist: sauber beenden
      if (dlg.__opsResolve) {
        try { dlg.__opsResolve(dlg.__opsCancelValue); } catch (_) {}
        dlg.__opsResolve = null;
      }

      dlg.__opsResolve = (val) => {
        cleanup();
        resolve(val);
      };

      cancelBtn.onclick = (e) => {
        e.preventDefault();
        dlg.__opsResolve && dlg.__opsResolve(dlg.__opsCancelValue);
        closeDialog(dlg);
      };

      okBtn.onclick = (e) => {
        e.preventDefault();

        if (mode !== "prompt") {
          dlg.__opsResolve && dlg.__opsResolve(true);
          closeDialog(dlg);
          return;
        }

        const val = String(input.value ?? "");
        const vres = (typeof validate === "function") ? validate(val) : true;
        if (vres !== true) {
          err.style.display = "";
          err.textContent = String(vres || t("Ungültige Eingabe.", "Invalid input."));
          return;
        }
        dlg.__opsResolve && dlg.__opsResolve(val);
        closeDialog(dlg);
      };

      openDialog(dlg);
    });
  }

  const OPS_UI = window.OPS_UI || {};

  OPS_UI.alert = (deMsg, enMsg, titleDe, titleEn, okDe, okEn) =>
    runDialog({ mode: "alert", deMsg, enMsg, titleDe, titleEn, okDe, okEn });

  OPS_UI.confirm = (deMsg, enMsg, titleDe, titleEn, okDe, okEn, cancelDe, cancelEn) =>
    runDialog({ mode: "confirm", deMsg, enMsg, titleDe, titleEn, okDe, okEn, cancelDe, cancelEn });

  OPS_UI.prompt = (deMsg, enMsg, defaultValue, titleDe, titleEn, okDe, okEn, cancelDe, cancelEn, inputLabelDe, inputLabelEn, placeholderDe, placeholderEn, inputType, validate) =>
    runDialog({
      mode: "prompt",
      deMsg, enMsg, titleDe, titleEn, okDe, okEn, cancelDe, cancelEn,
      inputLabelDe, inputLabelEn, placeholderDe, placeholderEn,
      defaultValue, inputType, validate
    });

  window.OPS_UI = OPS_UI;
})();
