/* ops-warenausgang-addon.js
   Add-on: "Warenausgang buchen" (Beleg + mehrere Positionen)

   Idee:
   - Interceptet die Aktion "📤 Warenausgang" aus dem Aktionen-Dropdown (value="goods-out").
   - Öffnet einen eigenen Warenausgang-Dialog (Beleg + mehrere Positionen).
   - Bucht jede Position nacheinander über den vorhandenen Buchungsdialog (invBookingForm)
     => Bestand, Audit usw. bleiben exakt wie im bestehenden System.
   - Speichert einen Beleg-Eintrag in localStorage: opsdeck_lager_receipts_v1
     und annotiert neu entstandene Buchungen in localStorage mit receipt-Metadaten.

   Keine Abhängigkeit von internen (IIFE) Funktionen.
*/
(() => {
  "use strict";

  const STORAGE_ARTICLES = "opsdeck_lager_v1";
  const STORAGE_BOOKINGS = "opsdeck_lager_bookings_v1";
  const STORAGE_RECEIPTS = "opsdeck_lager_receipts_v1";

  // Bypass-Flag, um während der automatisierten Buchung nicht selbst zu intercepten
  const BYPASS_FLAG = "__opsWoBypass";

  // ---- Sprache ---------------------------------------------------------
  function getLang() {
    try {
      if (typeof window.getOpsLang === "function") return window.getOpsLang();
    } catch (_) {}
    const l = document.documentElement.getAttribute("lang") || "de";
    return String(l).toLowerCase().startsWith("en") ? "en" : "de";
  }
  function t(de, en) {
    return getLang() === "en" ? en : de;
  }

  // ---- kleine Helpers --------------------------------------------------
  function nowIso() {
    return new Date().toISOString();
  }
  function toNumber(v, fallback) {
    const n = Number(String(v || "").replace(",", "."));
    return Number.isFinite(n) ? n : (fallback || 0);
  }
  function safeJSONParse(s, fallback) {
    try {
      const v = JSON.parse(s);
      return v == null ? fallback : v;
    } catch (_) {
      return fallback;
    }
  }
  function loadArticles() {
    return safeJSONParse(localStorage.getItem(STORAGE_ARTICLES) || "[]", []);
  }
  function loadBookings() {
    return safeJSONParse(localStorage.getItem(STORAGE_BOOKINGS) || "[]", []);
  }
  function saveBookings(arr) {
    localStorage.setItem(STORAGE_BOOKINGS, JSON.stringify(arr));
  }
  function loadReceipts() {
    return safeJSONParse(localStorage.getItem(STORAGE_RECEIPTS) || "[]", []);
  }
  function saveReceipts(arr) {
    localStorage.setItem(STORAGE_RECEIPTS, JSON.stringify(arr));
  }

  function uiInfoCompat(de, en, titleDe, titleEn) {
    const msg = t(de, en);
    const title = t(titleDe || "Info", titleEn || "Info");
    try {
      if (typeof window.uiInfo === "function") {
        window.uiInfo(de, en, titleDe || "ℹ️ Info", titleEn || "ℹ️ Info");
        return;
      }
    } catch (_) {}
if (window.OPS_UI && window.OPS_UI.alert) {
  window.OPS_UI.alert(de, en, titleDe || "ℹ️ Info", titleEn || "ℹ️ Info");
} else {
  alert(`${title}\n\n${msg}`);
}

  }


  // ---- UI helpers ------------------------------------------------------
  function ensureStyleTag(styleId, cssText) {
    try {
      if (document.getElementById(styleId)) return;
      const st = document.createElement("style");
      st.id = styleId;
      st.textContent = cssText;
      document.head.appendChild(st);
    } catch (_) {}
  }

  function createArticleOptions(articles, selectedId) {
    const opts = [`<option value="">${t("Bitte wählen…", "Select…")}</option>`];
    for (const a of (articles || [])) {
      const id = a && a.id != null ? String(a.id) : "";
      const label = [a.artikelnummer || "", a.bezeichnung || ""].filter(Boolean).join(" – ") || ("#" + id);
      const sel = selectedId && String(selectedId) === id ? " selected" : "";
      opts.push(`<option value="${escapeHtml(id)}"${sel}>${escapeHtml(label)}</option>`);
    }
    return opts.join("");
  }

  function escapeHtml(s) {
    return String(s || "")
      .replace(/&/g, "&amp;")
      .replace(/</g, "&lt;")
      .replace(/>/g, "&gt;")
      .replace(/"/g, "&quot;")
      .replace(/'/g, "&#39;");
  }

  // ---- UI: Dialog erstellen -------------------------------------------
  function ensureWoDialog() {
    let dlg = document.getElementById("invWoDialog");
    if (dlg) return dlg;

    ensureStyleTag(
      "invWoAddonStyle",
      `
      #invWoDialog{ padding:0; border:0; }
      #invWoDialog::backdrop{ background: rgba(0,0,0,.35); }
      #invWoDialog .inv-wo-card{ max-width:900px; width:min(900px,96vw); padding:16px 18px 18px; }
      #invWoDialog .inv-wo-head{ display:flex; align-items:flex-start; justify-content:space-between; gap:12px; }
      #invWoDialog .inv-wo-head h3{ margin:0; font-size:18px; line-height:1.2; }
      #invWoDialog .inv-wo-grid{ display:grid; grid-template-columns:1fr 1fr; gap:10px; margin-top:14px; }
      #invWoDialog .inv-wo-grid label{ display:flex; flex-direction:column; gap:6px; font-size:12px; }
      #invWoDialog .inv-wo-grid input,
      #invWoDialog .inv-wo-grid textarea,
      #invWoDialog .inv-wo-lines input,
      #invWoDialog .inv-wo-lines select{ width:100%; padding:8px 10px; }
      #invWoDialog .inv-wo-lines-head{ display:flex; justify-content:space-between; align-items:center; gap:10px; margin-top:14px; }
      #invWoDialog .inv-wo-lines{ margin-top:10px; display:flex; flex-direction:column; gap:8px; }
      #invWoDialog .inv-wo-line{ display:grid; grid-template-columns:1.7fr .6fr  auto; gap:8px; align-items:end; }
      #invWoDialog .inv-wo-line label{ display:flex; flex-direction:column; gap:6px; font-size:12px; }
      #invWoDialog .inv-wo-rm{ min-width:36px; padding:8px 10px; cursor:pointer; }
      #invWoDialog .inv-wo-error{ margin-top:10px; font-size:12px; color:#b00020; display:none; }
      #invWoDialog .inv-wo-actions{ display:flex; justify-content:flex-end; gap:10px; margin-top:14px; }
      @media (max-width: 640px){
        #invWoDialog .inv-wo-grid{ grid-template-columns:1fr; }
        #invWoDialog .inv-wo-line{ grid-template-columns:1fr 1fr; }
        #invWoDialog .inv-wo-line .inv-wo-rm{ grid-column:2; justify-self:end; }
      }
      `
    );

    dlg = document.createElement("dialog");
    dlg.id = "invWoDialog";
    dlg.className = "dialog";

    dlg.innerHTML = `
      <div class="card inv-wo-card">
        <div class="inv-wo-head">
          <div>
            <h3>${t("Warenausgang buchen", "Post goods issue")}</h3>
            <p class="tiny" style="margin:6px 0 0; opacity:.85;">
              ${t("Beleg + mehrere Positionen", "Document + multiple line items")}
            </p>
          </div>
          <button type="button" id="invWoCloseX" class="btn btn-ghost" aria-label="close" title="${t("Schließen", "Close")}">✕</button>
        </div>

        <form id="invWoForm" style="margin:0;">
          <div class="inv-wo-grid">
            <label>
              <span>${t("Beleg-Nr. (optional)", "Document no. (optional)")}</span>
              <input id="invWoDocNo" placeholder="WA-..." />
            </label>

            <label>
              <span>${t("Datum", "Date")}</span>
              <input id="invWoDate" type="date" />
            </label>

            <label>
              <span>${t("Empfänger / Bereich", "Recipient / department")}</span>
              <input id="invWoRecipient" placeholder="${t("optional", "optional")}" />
            </label>

            <label>
              <span>${t("Name / Kürzel (Pflicht)", "Name / initials (required)")}</span>
              <input id="invWoBy" required placeholder="AB" />
            </label>

            <label style="grid-column:1 / -1;">
              <span>${t("Grund (Pflicht)", "Reason (required)")}</span>
              <input id="invWoReason" required placeholder="${t("z.B. Versand / Montage", "e.g. shipment / installation")}" />
            </label>

            <label style="grid-column:1 / -1;">
              <span>${t("Notiz", "Note")}</span>
              <textarea id="invWoNote" rows="2" placeholder="${t("optional", "optional")}"></textarea>
            </label>
          </div>

          <hr style="margin:14px 0; opacity:.25;" />

          <div class="inv-wo-lines-head">
            <strong>${t("Positionen", "Line items")}</strong>
            <button type="button" id="invWoAddLine" class="btn btn-ghost">➕ ${t("Position hinzufügen", "Add line")}</button>
          </div>

          <div id="invWoLines" class="inv-wo-lines"></div>

          <div id="invWoError" class="inv-wo-error"></div>

          <div class="inv-wo-actions">
            <button type="button" id="invWoCancel" class="btn btn-ghost">${t("Abbrechen", "Cancel")}</button>
            <button type="submit" id="invWoPost" class="btn btn-primary">${t("Warenausgang buchen", "Post goods issue")}</button>
          </div>
        </form>
      </div>
    `;

    document.body.appendChild(dlg);

    // Close handlers
    const close = () => {
      try { dlg.close(); } catch (_) { dlg.removeAttribute("open"); }
    };
    dlg.querySelector("#invWoCancel")?.addEventListener("click", close);
    dlg.querySelector("#invWoCloseX")?.addEventListener("click", close);

    return dlg;
  }

  function showError(dlg, msg) {
    const el = dlg ? dlg.querySelector("#invWoError") : null;
    if (!el) return;
    el.textContent = msg || "";
    el.style.display = msg ? "block" : "none";
  }

  function createLineRow(articles, prefillArticleId) {
    const row = document.createElement("div");
    row.className = "inv-wo-line";

    row.innerHTML = `
      <label>
        <span>${t("Artikel", "Item")}</span>
        <select class="inv-wo-article">${createArticleOptions(articles, prefillArticleId)}</select>
      </label>

      <label>
        <span>${t("Menge", "Qty")}</span>
        <input class="inv-wo-qty" type="number" min="0" step="1" value="1" />
      </label>

      <button type="button" class="btn btn-ghost inv-wo-rm" title="${t("Entfernen", "Remove")}">✕</button>
    `;

    row.querySelector(".inv-wo-rm")?.addEventListener("click", () => row.remove());
    return row;
  }

  function openWoDialog(prefillArticleId) {
    const dlg = ensureWoDialog();

    // default date
    const dateEl = dlg.querySelector("#invWoDate");
    if (dateEl && !dateEl.value) dateEl.value = new Date().toISOString().slice(0, 10);

    // lines
    const linesBox = dlg.querySelector("#invWoLines");
    const addBtn = dlg.querySelector("#invWoAddLine");
    const arts = loadArticles();

    if (linesBox) linesBox.innerHTML = "";
    if (linesBox) linesBox.appendChild(createLineRow(arts, prefillArticleId));

    if (addBtn && !addBtn.dataset.bound) {
      addBtn.dataset.bound = "1";
      addBtn.addEventListener("click", () => {
        const a2 = loadArticles();
        const box = dlg.querySelector("#invWoLines");
        if (box) box.appendChild(createLineRow(a2, null));
      });
    }

    showError(dlg, "");

    // bind submit once
    const form = dlg.querySelector("#invWoForm");
    if (form && !form.dataset.bound) {
      form.dataset.bound = "1";
      form.addEventListener("submit", (e) => {
        e.preventDefault();
        postReceiptFromDialog(dlg);
      });
    }

    try {
      dlg.showModal();
    } catch (_) {
      dlg.setAttribute("open", "");
    }
  }

  // ---- Filter Snapshot/Restore (damit Zeilen gefunden werden) ----------
  function snapshotFilters() {
    const s = {};
    try {
      s.search = document.getElementById("invSearch")?.value || "";
      s.lager = document.getElementById("invFilterLager")?.value || "";
      s.kat = document.getElementById("invFilterKategorie")?.value || "";
      s.bestand = (document.getElementById("invFilterStatus") || document.getElementById("invFilterBestand"))?.value || "";
      s.regal = document.getElementById("invRegal")?.value || "";
      s.fach = document.getElementById("invFach")?.value || "";
      s.pos = document.getElementById("invPosition")?.value || "";
      s.niveau = document.getElementById("invNiveau")?.value || "";
    } catch (_) {}
    return s;
  }
  function setFiltersAll() {
    const fire = (el) => { try { el.dispatchEvent(new Event("input", { bubbles: true })); el.dispatchEvent(new Event("change", { bubbles: true })); } catch (_) {} };
    try {
      const search = document.getElementById("invSearch");
      if (search) { search.value = ""; fire(search); }
      const lager = document.getElementById("invFilterLager");
      if (lager) { lager.value = ""; fire(lager); }
      const kat = document.getElementById("invFilterKategorie");
      if (kat) { kat.value = ""; fire(kat); }
      const best = (document.getElementById("invFilterStatus") || document.getElementById("invFilterBestand"));
      if (best) { best.value = ""; fire(best); }
      const regal = document.getElementById("invRegal");
      if (regal) { regal.value = ""; fire(regal); }
      const fach = document.getElementById("invFach");
      if (fach) { fach.value = ""; fire(fach); }
      const pos = document.getElementById("invPosition");
      if (pos) { pos.value = ""; fire(pos); }
      const niveau = document.getElementById("invNiveau");
      if (niveau) { niveau.value = ""; fire(niveau); }
    } catch (_) {}
  }
  function restoreFilters(snap) {
    if (!snap) return;
    const fire = (el) => { try { el.dispatchEvent(new Event("input", { bubbles: true })); el.dispatchEvent(new Event("change", { bubbles: true })); } catch (_) {} };
    try {
      const search = document.getElementById("invSearch");
      if (search) { search.value = snap.search || ""; fire(search); }
      const lager = document.getElementById("invFilterLager");
      if (lager) { lager.value = snap.lager || ""; fire(lager); }
      const kat = document.getElementById("invFilterKategorie");
      if (kat) { kat.value = snap.kat || ""; fire(kat); }
      const best = (document.getElementById("invFilterStatus") || document.getElementById("invFilterBestand"));
      if (best) { best.value = snap.bestand || ""; fire(best); }
      const regal = document.getElementById("invRegal");
      if (regal) { regal.value = snap.regal || ""; fire(regal); }
      const fach = document.getElementById("invFach");
      if (fach) { fach.value = snap.fach || ""; fire(fach); }
      const pos = document.getElementById("invPosition");
      if (pos) { pos.value = snap.pos || ""; fire(pos); }
      const niveau = document.getElementById("invNiveau");
      if (niveau) { niveau.value = snap.niveau || ""; fire(niveau); }
    } catch (_) {}
  }

  function sleep(ms) {
    return new Promise((r) => setTimeout(r, ms));
  }

  async function waitForDialogOpen(id, timeoutMs) {
    const t0 = Date.now();
    while (Date.now() - t0 < (timeoutMs || 3000)) {
      const dlg = document.getElementById(id);
      if (dlg && (dlg.open || dlg.hasAttribute("open"))) return dlg;
      await sleep(30);
    }
    return null;
  }

  async function waitForDialogClose(id, timeoutMs) {
    const t0 = Date.now();
    while (Date.now() - t0 < (timeoutMs || 3000)) {
      const dlg = document.getElementById(id);
      if (!dlg || (!dlg.open && !dlg.hasAttribute("open"))) return true;
      await sleep(50);
    }
    return false;
  }

  async function postOneLine({ articleId, qty, noteText, employee, receiptMeta }) {
    // 1) ensure row exists
    const actionSelect = document.querySelector(`select.inv-action-select[data-id="${CSS.escape(String(articleId))}"]`);
    if (!actionSelect) {
      return { ok: false, error: t("Artikel in Tabelle nicht gefunden (Filter?)", "Item not found in table (filters?)") };
    }

    // 2) Open existing booking dialog
    window[BYPASS_FLAG] = true;
    actionSelect.value = "goods-out";
    actionSelect.dispatchEvent(new Event("change", { bubbles: true }));
    window[BYPASS_FLAG] = false;

    const dlg = await waitForDialogOpen("invBookingDialog", 4000);
    if (!dlg) {
      return { ok: false, error: t("Buchungsdialog konnte nicht geöffnet werden.", "Booking dialog could not be opened.") };
    }

    // 3) Fill booking fields
    const amountEl = document.getElementById("invBookingAmount");
    const employeeEl = document.getElementById("invBookingEmployee");
    const noteEl = document.getElementById("invBookingNote");
    const reasonEl = document.getElementById("invBookingReason");

    if (amountEl) amountEl.value = String(qty);
    if (employeeEl) employeeEl.value = employee || "";

    if (reasonEl) {
      const hasWO = Array.from(reasonEl.options || []).some((o) => o.value === "warenausgang");
      reasonEl.value = hasWO ? "warenausgang" : (reasonEl.value || "warenausgang");
      reasonEl.dispatchEvent(new Event("change", { bubbles: true }));
    }

    if (noteEl) noteEl.value = noteText || "";

    // 4) Submit form
    const form = document.getElementById("invBookingForm");
    const beforeBookings = loadBookings();
    const beforeLen = beforeBookings.length;

    if (form && typeof form.requestSubmit === "function") {
      form.requestSubmit();
    } else if (form) {
      form.dispatchEvent(new Event("submit", { bubbles: true, cancelable: true }));
    }

    const closed = await waitForDialogClose("invBookingDialog", 8000);
    if (!closed) {
      return { ok: false, error: t("Buchung konnte nicht abgeschlossen werden.", "Booking could not be completed.") };
    }

    // 5) annotate newly created booking(s)
    const after = loadBookings();
    const added = after.slice(beforeLen);
    const addedIds = [];

    if (!added.length) {
      return { ok: false, error: t("Keine Buchung erzeugt – evtl. wurde der Buchungsdialog abgebrochen.", "No booking created – dialog may have been cancelled.") };
    }

    for (let i = after.length - added.length; i < after.length; i++) {
      const b = after[i];
      if (!b || typeof b !== "object") continue;

      b.receiptId = receiptMeta.id;
      b.receiptKind = receiptMeta.kind;
      b.docNo = receiptMeta.docNo;
      b.docDate = receiptMeta.docDate;
      b.recipient = receiptMeta.recipient;
      b.receiptReason = receiptMeta.reason;
      b.receiptBookedBy = receiptMeta.bookedBy;

      if (b.id != null) addedIds.push(b.id);
    }
    saveBookings(after);

    return { ok: true, bookingIds: addedIds };
  }

  async function postReceiptFromDialog(dlg) {
    showError(dlg, "");

    const bookedBy = String(dlg.querySelector("#invWoBy")?.value || "").trim();
    const reason = String(dlg.querySelector("#invWoReason")?.value || "").trim();
    const note = String(dlg.querySelector("#invWoNote")?.value || "").trim();
    const date = String(dlg.querySelector("#invWoDate")?.value || "").trim() || new Date().toISOString().slice(0, 10);
    const recipient = String(dlg.querySelector("#invWoRecipient")?.value || "").trim();

    let docNo = String(dlg.querySelector("#invWoDocNo")?.value || "").trim();
    if (!docNo) {
      docNo = `WA-${date.replace(/-/g, "")}-${Math.random().toString(36).slice(2, 6).toUpperCase()}`;
    }

    if (!bookedBy) {
      showError(dlg, t("Bitte Name/Kürzel eingeben.", "Please enter name/initials."));
      try { dlg.querySelector("#invWoBy")?.focus(); } catch (_) {}
      return;
    }
    if (!reason) {
      showError(dlg, t("Bitte Grund eingeben.", "Please enter a reason."));
      try { dlg.querySelector("#invWoReason")?.focus(); } catch (_) {}
      return;
    }

    const lineNodes = Array.from(dlg.querySelectorAll(".inv-wo-line"));
    if (!lineNodes.length) {
      showError(dlg, t("Bitte mindestens eine Position hinzufügen.", "Please add at least one line item."));
      return;
    }

    const articles = loadArticles();
    const lines = [];
    for (const node of lineNodes) {
      const aid = String(node.querySelector("select.inv-wo-article")?.value || "").trim();
      const qty = toNumber(node.querySelector("input.inv-wo-qty")?.value, 0);
      if (!aid) continue;
      if (!qty || qty <= 0) continue;
      const art = articles.find((a) => String(a.id) === String(aid));
      if (!art) continue;
      lines.push({ articleId: String(aid), amount: qty });
    }

    if (!lines.length) {
      showError(dlg, t("Bitte gültige Positionen (Artikel + Menge) eintragen.", "Please add valid lines (item + qty)."));
      return;
    }

    const rid = `wo_${date}_${Date.now().toString(36)}`;

    const postBtn = dlg.querySelector("#invWoPost");
    const prevText = postBtn ? postBtn.textContent : "";
    if (postBtn) {
      postBtn.disabled = true;
      postBtn.textContent = t("Buchen…", "Posting…");
    }

    const receipt = {
      id: rid,
      ts: nowIso(),
      kind: "warenausgang",
      docNo: docNo,
      docDate: date,
      recipient: recipient,
      bookedBy: bookedBy,
      reason: reason,
      note: note,
      lines: lines.map((l) => ({ articleId: l.articleId, amount: l.amount })),
      bookingIds: [],
    };

    const snap = snapshotFilters();
    setFiltersAll();
    await sleep(30);

    try {
      for (let i = 0; i < lines.length; i++) {
        const l = lines[i];
        if (postBtn) postBtn.textContent = t(`Buchen… (${i + 1}/${lines.length})`, `Posting… (${i + 1}/${lines.length})`);

        const bookingNote = [
          `WA ${docNo}`,
          recipient ? `Empfänger ${recipient}` : "",
          reason,
          note ? note : ""
        ].filter(Boolean).join(" | ");

        const res = await postOneLine({
          articleId: l.articleId,
          qty: l.amount,
          noteText: bookingNote,
          employee: bookedBy,
          receiptMeta: receipt,
        });

        if (!res.ok) throw new Error(res.error || "WO failed");
        if (res.bookingIds && res.bookingIds.length) receipt.bookingIds.push(...res.bookingIds);
      }

      const receipts = loadReceipts();
      receipts.push(receipt);
      saveReceipts(receipts);

      dlg.close();

      uiInfoCompat(
        `Warenausgang gebucht.\n\nBeleg: ${docNo}\nPositionen: ${lines.length}`,
        `Goods issue posted.\n\nDocument: ${docNo}\nLines: ${lines.length}`,
        "✅ Erfolg",
        "✅ Success"
      );
    } catch (err) {
      console.error("[WO add-on] failed:", err);
      showError(dlg, String(err && err.message ? err.message : err));
    } finally {
      restoreFilters(snap);
      if (postBtn) {
        postBtn.disabled = false;
        postBtn.textContent = prevText || t("Warenausgang buchen", "Post goods issue");
      }
    }
  }

  // ---- Öffnen / Intercept ---------------------------------------------
  function bindIntercept() {
    if (window.__opsWoAddonBound) return;
    window.__opsWoAddonBound = true;

    document.addEventListener(
      "change",
      (e) => {
        try {
          if (window[BYPASS_FLAG]) return;
          const target = e.target;
          if (!target) return;

          const select = target.closest && target.closest("select.inv-action-select");
          if (!select) return;
          if (select.value !== "goods-out") return;

          e.preventDefault();
          e.stopPropagation();

          const id = select.getAttribute("data-id");
          select.value = ""; // reset
          openWoDialog(id ? String(id) : null);
        } catch (err) {
          console.error("[WO add-on] intercept error:", err);
        }
      },
      true
    );
  }

  function bindOpenEvent() {
    if (window.__opsWoOpenEventBound) return;
    window.__opsWoOpenEventBound = true;

    window.addEventListener("opsWoOpenDialog", (ev) => {
      try {
        const prefill = ev && ev.detail ? ev.detail.prefillArticleId : null;
        openWoDialog(prefill ? String(prefill) : null);
      } catch (_) {
        try { openWoDialog(null); } catch (_) {}
      }
    });
  }

  function boot() {
    try {
      bindIntercept();
      bindOpenEvent();
      window.__opsWoAddon = { version: "v1", ts: Date.now() };
    } catch (e) {
      console.error("[WO add-on] boot failed:", e);
    }
  }

  if (document.readyState === "loading") {
    document.addEventListener("DOMContentLoaded", boot);
  } else {
    boot();
  }
})();
