/* ops-wareneingang-addon.js
   Add-on: "Wareneingang buchen" (Beleg + mehrere Positionen) ohne Core-Änderungen.

   Idee:
   - Interceptet die Aktion "📥 Wareneingang" aus dem Aktionen-Dropdown (value="goods-in").
   - Öffnet einen eigenen Wareneingang-Dialog (Lieferschein + mehrere Positionen).
   - Bucht jede Position nacheinander über den vorhandenen Buchungsdialog (invBookingForm)
     => Bestand, Gleitpreis, Audit usw. bleiben exakt wie im bestehenden System.
   - Speichert einen Beleg-Eintrag in localStorage: opsdeck_lager_receipts_v1
     und annotiert die neu entstandenen Buchungen in localStorage mit receipt-Metadaten.
   - Optional: Lieferschein-Dateien werden im Browser (IndexedDB) gespeichert und als Metadaten
     am Beleg, an Buchungen und am Artikel hinterlegt (falls UI das später anzeigen möchte).

   Keine Abhängigkeit von internen (IIFE) Funktionen.
*/
(() => {
  "use strict";

  // ---- Konfiguration / Keys -------------------------------------------
  const STORAGE_ARTICLES = "opsdeck_lager_v1";
  const STORAGE_BOOKINGS = "opsdeck_lager_bookings_v1";
  const STORAGE_RECEIPTS = "opsdeck_lager_receipts_v1";

  // File Store (IndexedDB) – gleicher Name wie im Core (falls vorhanden)
  const FILES_DB_NAME  = "opsdeck_lager_files_v1";
  const FILES_DB_STORE = "files";
  let __filesDbPromise = null;

  // Bypass-Flag, um während der automatisierten Buchung nicht selbst zu intercepten
  const BYPASS_FLAG = "__opsWeBypass";

  // ---- Sprache ---------------------------------------------------------
  function getLang() {
    try {
      if (typeof window.getOpsLang === "function") return window.getOpsLang();
    } catch (_) {}
    const l = document.documentElement.getAttribute("lang") || "de";
    return String(l).toLowerCase().startsWith("en") ? "en" : "de";
  }
  function t(de, en) {
    return getLang() === "en" ? en : de;
  }

  // ---- kleine Helpers --------------------------------------------------
  function nowIso() {
    return new Date().toISOString();
  }

  // ---- Storage helpers -------------------------------------------------
  function safeJSONParse(s, fallback) {
    try {
      const v = JSON.parse(s);
      return v == null ? fallback : v;
    } catch (_) {
      return fallback;
    }
  }
  function loadArticles() {
    return safeJSONParse(localStorage.getItem(STORAGE_ARTICLES) || "[]", []);
  }
  function saveArticles(arr) {
    localStorage.setItem(STORAGE_ARTICLES, JSON.stringify(arr));
  }
  function loadBookings() {
    return safeJSONParse(localStorage.getItem(STORAGE_BOOKINGS) || "[]", []);
  }
  function saveBookings(arr) {
    localStorage.setItem(STORAGE_BOOKINGS, JSON.stringify(arr));
  }
  function loadReceipts() {
    return safeJSONParse(localStorage.getItem(STORAGE_RECEIPTS) || "[]", []);
  }
  function saveReceipts(arr) {
    localStorage.setItem(STORAGE_RECEIPTS, JSON.stringify(arr));
  }

  // ---- IndexedDB: Datei speichern -------------------------------------
  function openFilesDb() {
    if (!("indexedDB" in window)) return Promise.resolve(null);
    if (__filesDbPromise) return __filesDbPromise;

    __filesDbPromise = new Promise((resolve) => {
      try {
        const req = indexedDB.open(FILES_DB_NAME, 1);
        req.onupgradeneeded = () => {
          try {
            const db = req.result;
            if (!db.objectStoreNames.contains(FILES_DB_STORE)) {
              db.createObjectStore(FILES_DB_STORE, { keyPath: "id" });
            }
          } catch (_) {}
        };
        req.onsuccess = () => resolve(req.result);
        req.onerror = () => resolve(null);
      } catch (_) {
        resolve(null);
      }
    });

    return __filesDbPromise;
  }

  async function storeFileInDb(file, kind) {
    const db = await openFilesDb();
    if (!db || !file) return null;

    const rec = {
      id: "f" + Math.random().toString(36).slice(2),
      ts: nowIso(),
      kind: kind || "",
      name: file.name || "file",
      type: file.type || "",
      size: file.size || 0,
      blob: file
    };

    return await new Promise((resolve) => {
      try {
        const tx = db.transaction(FILES_DB_STORE, "readwrite");
        tx.objectStore(FILES_DB_STORE).put(rec);
        tx.oncomplete = () => resolve(rec);
        tx.onerror = () => resolve(null);
      } catch (_) {
        resolve(null);
      }
    });
  }

  // ---- IndexedDB: Datei lesen/öffnen (für UI-Links) -------------------
  async function getFileFromDb(fileId) {
    const db = await openFilesDb();
    if (!db || !fileId) return null;

    return await new Promise((resolve) => {
      try {
        const tx = db.transaction(FILES_DB_STORE, "readonly");
        const req = tx.objectStore(FILES_DB_STORE).get(String(fileId));
        req.onsuccess = () => resolve(req.result || null);
        req.onerror = () => resolve(null);
      } catch (_) {
        resolve(null);
      }
    });
  }

  async function openStoredFileCompat(fileId) {
    try {
      const rec = await getFileFromDb(fileId);
      if (!rec || !rec.blob) {
     if (window.OPS_UI && window.OPS_UI.alert) {
  window.OPS_UI.alert(
    "Datei nicht gefunden (evtl. Browser-Daten gelöscht).",
    "File not found (browser data may have been cleared).",
    "📎 Datei",
    "📎 File"
  );
} else {
  alert(t(
    "Datei nicht gefunden (evtl. Browser-Daten gelöscht).",
    "File not found (browser data may have been cleared)."
  ));
}
return;

      }
      const url = URL.createObjectURL(rec.blob);
      const a = document.createElement("a");
      a.href = url;
      a.target = "_blank";
      a.rel = "noopener";
      document.body.appendChild(a);
      a.click();
      a.remove();
      setTimeout(() => URL.revokeObjectURL(url), 30000);
    } catch (e) {
      console.warn("[WE add-on] openStoredFileCompat failed:", e);
    }
  }

  // ---- UI helpers ------------------------------------------------------
  function ensureStyleTag(styleId, cssText) {
    try {
      if (document.getElementById(styleId)) return;
      const st = document.createElement("style");
      st.id = styleId;
      st.textContent = cssText;
      document.head.appendChild(st);
    } catch (_) {}
  }

  // ---- UI: Dialog erstellen -------------------------------------------
  function ensureWeDialog() {
    let dlg = document.getElementById("invWeDialog");
    if (dlg) return dlg;

    // Ein kleines CSS nur für Layout/Fallback (damit wir nicht "alles kaputt machen")
    ensureStyleTag(
      "invWeAddonStyle",
      `
      #invWeDialog{ padding:0; border:0; }
      #invWeDialog::backdrop{ background: rgba(0,0,0,.35); }
      #invWeDialog .inv-we-card{ max-width:900px; width:min(900px,96vw); padding:16px 18px 18px; }
      #invWeDialog .inv-we-head{ display:flex; align-items:flex-start; justify-content:space-between; gap:12px; }
      #invWeDialog .inv-we-head h3{ margin:0; font-size:18px; line-height:1.2; }
      #invWeDialog .inv-we-grid{ display:grid; grid-template-columns:1fr 1fr; gap:10px; margin-top:14px; }
      #invWeDialog .inv-we-grid label{ display:flex; flex-direction:column; gap:6px; font-size:12px; }
      #invWeDialog .inv-we-grid input,
      #invWeDialog .inv-we-grid textarea,
      #invWeDialog .inv-we-lines input,
      #invWeDialog .inv-we-lines select{ width:100%; padding:8px 10px; }
      #invWeDialog .inv-we-lines-head{ display:flex; justify-content:space-between; align-items:center; gap:10px; margin-top:14px; }
      #invWeDialog .inv-we-lines{ margin-top:10px; display:flex; flex-direction:column; gap:8px; }
      #invWeDialog .inv-we-line{ display:grid; grid-template-columns:1.7fr .6fr .7fr auto; gap:8px; align-items:end; }
      #invWeDialog .inv-we-line label{ display:flex; flex-direction:column; gap:6px; font-size:12px; }
      #invWeDialog .inv-we-rm{ min-width:36px; padding:8px 10px; cursor:pointer; }
      #invWeDialog .inv-we-error{ margin-top:10px; font-size:12px; color:#b00020; display:none; }
      #invWeDialog .inv-we-actions{ display:flex; justify-content:flex-end; gap:10px; margin-top:14px; }
      #invWeDialog .inv-we-files-info{ opacity:.8; font-size:12px; margin-top:6px; }
      @media (max-width: 640px){
        #invWeDialog .inv-we-grid{ grid-template-columns:1fr; }
        #invWeDialog .inv-we-line{ grid-template-columns:1fr 1fr; }
        #invWeDialog .inv-we-line .inv-we-rm{ grid-column:2; justify-self:end; }
      }
    `
    );

    dlg = document.createElement("dialog");
    dlg.id = "invWeDialog";
    // Diese Klassen nutzen deine bestehende "moderne" Optik (falls vorhanden)
    dlg.className = "dialog";

    dlg.innerHTML = `
      <div class="card inv-we-card">
        <div class="inv-we-head">
          <div>
            <h3 id="invWeTitle">${t("Wareneingang buchen", "Post goods receipt")}</h3>
            <p id="invWeSub" class="tiny" style="margin:6px 0 0; opacity:.85;">
              ${t("Beleg + mehrere Positionen", "Document + multiple line items")}
            </p>
          </div>
          <button type="button" id="invWeCloseX" class="btn btn-ghost" aria-label="close" title="${t("Schließen", "Close")}">✕</button>
        </div>

        <form id="invWeForm" style="margin:0;">
          <div class="inv-we-grid">
            <label>
              <span>${t("Lieferscheinnr. (Pflicht)", "Delivery note no. (required)")}</span>
              <input id="invWeDocNo" required placeholder="LS-..." />
            </label>

            <label>
              <span>${t("Auftragsnummer / PO", "Order no. / PO")}</span>
              <input id="invWeOrderNo" placeholder="${t("optional", "optional")}" />
            </label>

            <label>
              <span>${t("Lieferant", "Supplier")}</span>
              <input id="invWeSupplier" placeholder="${t("optional", "optional")}" />
            </label>

            <label>
              <span>${t("Name / Kürzel (Pflicht)", "Name / initials (required)")}</span>
              <input id="invWeBy" required placeholder="AB" />
            </label>

            <label>
              <span>${t("Lieferschein-Datum", "Delivery note date")}</span>
              <input id="invWeDate" type="date" />
            </label>

            <label style="grid-column:1 / -1;">
              <span>${t("Lieferschein hochladen (optional)", "Upload delivery note (optional)")}</span>
              <input id="invWeFiles" type="file" accept="application/pdf,image/*" multiple />
              <div id="invWeFilesInfo" class="inv-we-files-info">${t("Keine Datei ausgewählt.", "No file selected.")}</div>
            </label>

            <label style="grid-column:1 / -1;">
              <span>${t("Grund (Pflicht)", "Reason (required)")}</span>
              <input id="invWeReason" required placeholder="${t("z.B. Auffüllung Lager", "e.g. restock")}" />
            </label>

            <label style="grid-column:1 / -1;">
              <span>${t("Notiz", "Note")}</span>
              <textarea id="invWeNote" rows="2" placeholder="${t("optional", "optional")}"></textarea>
            </label>
          </div>

          <hr style="margin:14px 0; opacity:.25;" />

          <div class="inv-we-lines-head">
            <div style="font-weight:700;">${t("Positionen", "Line items")}</div>
            <button type="button" id="invWeAddLine" class="btn">${t("+ Position hinzufügen", "+ Add line")}</button>
          </div>

          <div id="invWeLines" class="inv-we-lines"></div>

          <div id="invWeError" class="inv-we-error"></div>

          <div class="inv-we-actions">
            <button type="button" id="invWeCancel" class="btn btn-ghost">${t("Abbrechen", "Cancel")}</button>
            <button type="submit" id="invWePost" class="btn btn-primary">${t("Wareneingang buchen", "Post goods receipt")}</button>
          </div>
        </form>
      </div>
    `;

    document.body.appendChild(dlg);

    // Close actions
    const closeX = dlg.querySelector("#invWeCloseX");
    const cancel = dlg.querySelector("#invWeCancel");
    if (closeX) closeX.addEventListener("click", () => dlg.close());
    if (cancel) cancel.addEventListener("click", () => dlg.close());

    // Datei-Info
    const files = dlg.querySelector("#invWeFiles");
    if (files) {
      files.addEventListener("change", () => updateWeFilesInfo(dlg));
      updateWeFilesInfo(dlg);
    }

    return dlg;
  }

  function updateWeFilesInfo(dlg) {
    const input = dlg ? dlg.querySelector("#invWeFiles") : null;
    const info  = dlg ? dlg.querySelector("#invWeFilesInfo") : null;
    if (!info) return;

    const files = input && input.files ? Array.from(input.files) : [];
    if (!files.length) {
      info.textContent = t("Keine Datei ausgewählt.", "No file selected.");
      return;
    }
    info.textContent = t("Ausgewählt: ", "Selected: ") + files.map(f => f.name).join(", ");
  }

  function formatArticleLabel(a) {
    const nr = a.artikelnummer || a.interneArtikelnummer || a.id || "";
    const bez = a.bezeichnung || "";
    const lager = a.lager != null ? String(a.lager) : "";
    const best = a.bestand != null ? String(a.bestand) : "0";
    return `${nr} — ${bez}${lager ? ` · ${t("Lager", "Warehouse")}: ${lager}` : ""}${best ? ` · ${t("Bestand", "Stock")}: ${best}` : ""}`;
  }

  function createLineRow(articles, preselectId) {
    const row = document.createElement("div");
    row.className = "inv-we-line";

    const selWrap = document.createElement("label");
    selWrap.innerHTML = `<span>${t("Artikel", "Item")}</span>`;

    const sel = document.createElement("select");
    sel.className = "inv-we-article";

    const ph = document.createElement("option");
    ph.value = "";
    ph.textContent = t("– bitte wählen –", "– select –");
    sel.appendChild(ph);

    articles.forEach((a) => {
      const opt = document.createElement("option");
      opt.value = String(a.id);
      opt.textContent = formatArticleLabel(a);
      sel.appendChild(opt);
    });

    if (preselectId != null) sel.value = String(preselectId);
    selWrap.appendChild(sel);

    const qtyWrap = document.createElement("label");
    qtyWrap.innerHTML = `<span>${t("Menge", "Qty")}</span>`;

    const qty = document.createElement("input");
    qty.type = "number";
    qty.min = "0";
    qty.step = "1";
    qty.value = "1";
    qty.className = "inv-we-qty";
    qtyWrap.appendChild(qty);

    const priceWrap = document.createElement("label");
    priceWrap.innerHTML = `<span>${t("Einzelpreis", "Unit price")}</span>`;

    const price = document.createElement("input");
    price.type = "text";
    price.placeholder = t("optional", "optional");
    price.className = "inv-we-price";
    priceWrap.appendChild(price);

    const rm = document.createElement("button");
    rm.type = "button";
    rm.className = "btn btn-ghost inv-we-rm";
    rm.textContent = "✕";
    rm.title = t("Position entfernen", "Remove line");
    rm.addEventListener("click", () => row.remove());

    row.appendChild(selWrap);
    row.appendChild(qtyWrap);
    row.appendChild(priceWrap);
    row.appendChild(rm);

    return row;
  }

  function showError(dlg, msg) {
    const box = dlg.querySelector("#invWeError");
    if (!box) return;
    if (!msg) {
      box.style.display = "none";
      box.textContent = "";
    } else {
      box.style.display = "block";
      box.textContent = msg;
    }
  }
  // ---- Modern Info Dialog (reuses #invInfoDialog if present) ----------
function ensureInfoDialog() {
  let dlg = document.getElementById("invInfoDialog");
  if (dlg) return dlg;

  dlg = document.createElement("dialog");
  dlg.id = "invInfoDialog";
  dlg.className = "dialog";
  dlg.innerHTML = `
    <div class="card" style="max-width:560px;width:92vw;">
      <h3 id="invInfoTitle" style="margin:0 0 10px 0;font-size:18px;"></h3>
      <div id="invInfoMsg" style="white-space:pre-wrap;line-height:1.45;"></div>
      <div style="display:flex;justify-content:flex-end;gap:8px;margin-top:14px;">
        <button type="button" class="btn btn-primary" id="invInfoOk">OK</button>
      </div>
    </div>
  `;
  document.body.appendChild(dlg);
  return dlg;
}

function uiInfo(deMsg, enMsg, deTitle, enTitle, deOk, enOk) {
  const dlg = ensureInfoDialog();
  const titleEl = dlg.querySelector("#invInfoTitle");
  const msgEl   = dlg.querySelector("#invInfoMsg");
  const okBtn   = dlg.querySelector("#invInfoOk");

  if (titleEl) titleEl.textContent = t(deTitle || "ℹ️ Hinweis", enTitle || "ℹ️ Info");
  if (msgEl)   msgEl.textContent   = t(deMsg, enMsg);
  if (okBtn)   okBtn.textContent   = t(deOk || "OK", enOk || "OK");

  const onOk = (ev) => {
    ev.preventDefault();
    try { okBtn && okBtn.removeEventListener("click", onOk); } catch (_) {}
    try { dlg.close(); } catch (_) { dlg.removeAttribute("open"); }
  };
  try { okBtn && okBtn.addEventListener("click", onOk); } catch (_) {}

  try { if (!dlg.open) dlg.showModal(); } catch (_) { dlg.setAttribute("open",""); }
}

function uiWarnRequired(deMsg, enMsg) {
  uiInfo(deMsg, enMsg, "⚠️ Pflichtfeld", "⚠️ Required");
}


  // ---- Attachments: in Details/Artikel-Dialog anzeigen ----------------
  function normalizeText(v) {
    const s = String(v == null ? "" : v).trim();
    if (!s || s === "–" || s === "-") return "";
    return s;
  }

  function findArticleByNumbers(interneArtikelnummer, artikelnummer) {
    const internal = normalizeText(interneArtikelnummer);
    const external = normalizeText(artikelnummer);
    const arts = loadArticles();

    if (internal) {
      const a = arts.find(x => normalizeText(x.interneArtikelnummer) === internal);
      if (a) return a;
    }
    if (external) {
      const a = arts.find(x => normalizeText(x.artikelnummer) === external);
      if (a) return a;
    }
    return null;
  }

  function renderAttachmentsInto(container, attachments) {
    if (!container) return;
    const atts = Array.isArray(attachments) ? attachments.filter(a => a && a.id) : [];

    container.innerHTML = "";

    if (!atts.length) {
      container.textContent = "–";
      return;
    }

    const ul = document.createElement("ul");
    ul.style.listStyle = "none";
    ul.style.padding = "0";
    ul.style.margin = "0";

    const sorted = atts.slice().sort((a, b) => String(b.ts || "").localeCompare(String(a.ts || "")));
    for (const att of sorted) {
      const li = document.createElement("li");
      li.style.display = "flex";
      li.style.gap = "8px";
      li.style.alignItems = "center";
      li.style.marginBottom = "4px";

      const btn = document.createElement("button");
      btn.type = "button";
      btn.textContent = t("Öffnen", "Open");
      btn.className = "btn btn-ghost";
      btn.style.padding = "2px 10px";
      btn.addEventListener("click", async () => {
        await openStoredFileCompat(att.id);
      });

      const label = document.createElement("span");
      const parts = [];
      parts.push(att.name || t("Datei", "File"));
      const dn = normalizeText(att.deliveryNoteNo);
      const po = normalizeText(att.orderNo);
      if (dn) parts.push(`${t("LS", "DN")}: ${dn}`);
      if (po) parts.push(`${t("Auftrag", "PO")}: ${po}`);
      label.textContent = parts.join(" • ");

      li.appendChild(btn);
      li.appendChild(label);
      ul.appendChild(li);
    }

    container.appendChild(ul);
  }

  function updateDetailDialogAttachments() {
    const td = document.getElementById("detail_delivery_notes");
    if (!td) return;

    const interne = document.getElementById("detail_interneArtikelnummer")?.textContent || "";
    const artno = document.getElementById("detail_artikelnummer")?.textContent || "";
    const art = findArticleByNumbers(interne, artno);
    const atts = art && Array.isArray(art.attachments) ? art.attachments : [];
    renderAttachmentsInto(td, atts);
  }

  function updateArticleDialogAttachments() {
    const box = document.getElementById("invArticleWeAttachmentsBox");
    if (!box) return;

    const interne = document.getElementById("invInterneArtikelnummer")?.value || "";
    const artno = document.getElementById("invArtikelnummer")?.value || "";
    const art = findArticleByNumbers(interne, artno);
    const atts = art && Array.isArray(art.attachments) ? art.attachments : [];
    renderAttachmentsInto(box, atts);
  }

  function ensureArticleDialogAttachmentsSection() {
    const form = document.getElementById("invArticleForm");
    if (!form) return;
    if (document.getElementById("invArticleWeAttachmentsSection")) return;

    const section = document.createElement("div");
    section.id = "invArticleWeAttachmentsSection";
    section.style.marginTop = "14px";
    section.style.paddingTop = "12px";
    section.style.borderTop = "1px solid rgba(0,0,0,.08)";

    section.innerHTML = `
      <div style="font-weight:600;margin-bottom:8px;">${t("Lieferscheine / Anlagen", "Delivery notes / attachments")}</div>
      <div id="invArticleWeAttachmentsBox"></div>
    `;

    form.appendChild(section);
  }

  function hookDialogOpen(dialogId, onOpen) {
    const dlg = document.getElementById(dialogId);
    if (!dlg || dlg.__opsWeHooked) return !!dlg;
    dlg.__opsWeHooked = true;

    const run = () => {
      try {
        if (dlg.open) setTimeout(onOpen, 0);
      } catch (_) {}
    };

    const mo = new MutationObserver(run);
    mo.observe(dlg, { attributes: true, attributeFilter: ["open"] });
    run();
    return true;
  }

  function setupAttachmentHooks() {
    // Detail dialog
    const okDetail = hookDialogOpen("invDetailDialog", updateDetailDialogAttachments);

    // Article dialog
    const okArt = hookDialogOpen("invArticleDialog", () => {
      ensureArticleDialogAttachmentsSection();
      updateArticleDialogAttachments();
    });

    if (okDetail && okArt) return;
    // try again later (dialogs may be created after init)
    setTimeout(setupAttachmentHooks, 600);
  }

  // ---- Orchestrierung: Buchungen nacheinander -------------------------
  function sleep(ms) {
    return new Promise((r) => setTimeout(r, ms));
  }

  async function waitForDialogOpen(id, timeoutMs = 4000) {
    const start = Date.now();
    while (Date.now() - start < timeoutMs) {
      const dlg = document.getElementById(id);
      if (dlg && dlg.open) return dlg;
      await sleep(25);
    }
    return null;
  }

  async function waitForDialogClose(id, timeoutMs = 8000) {
    const start = Date.now();
    while (Date.now() - start < timeoutMs) {
      const dlg = document.getElementById(id);
      if (dlg && !dlg.open) return true;
      await sleep(25);
    }
    return false;
  }

  function dispatchInput(el) {
    if (!el) return;
    el.dispatchEvent(new Event("input", { bubbles: true }));
    el.dispatchEvent(new Event("change", { bubbles: true }));
  }

  function pickAllValue(sel) {
    if (!sel || !sel.options || !sel.options.length) return null;
    const opts = Array.from(sel.options);
    const preferred = opts.find((o) => o.value === "" || o.value === "all");
    return (preferred ? preferred.value : opts[0].value);
  }

  function snapshotFilters() {
    return {
      search: document.getElementById("invSearch")?.value ?? null,
      lager: document.getElementById("invFilterLager")?.value ?? null,
      kat: document.getElementById("invFilterKategorie")?.value ?? null,
      bestand: document.getElementById("invFilterBestand")?.value ?? null,
      sperre: document.getElementById("invFilterSperre")?.value ?? null,
    };
  }

  function setFiltersAll() {
    const search = document.getElementById("invSearch");
    if (search) {
      search.value = "";
      dispatchInput(search);
    }

    const lager = document.getElementById("invFilterLager");
    const kat = document.getElementById("invFilterKategorie");
    const bestand = document.getElementById("invFilterBestand");
    const sperre = document.getElementById("invFilterSperre");

    for (const sel of [lager, kat, bestand, sperre]) {
      if (!sel) continue;
      const v = pickAllValue(sel);
      if (v != null) {
        sel.value = v;
        sel.dispatchEvent(new Event("change", { bubbles: true }));
      }
    }
  }

  function restoreFilters(snap) {
    if (!snap) return;
    const search = document.getElementById("invSearch");
    if (search && snap.search != null) {
      search.value = snap.search;
      dispatchInput(search);
    }

    const mapping = [
      ["invFilterLager", snap.lager],
      ["invFilterKategorie", snap.kat],
      ["invFilterBestand", snap.bestand],
      ["invFilterSperre", snap.sperre],
    ];

    for (const [id, val] of mapping) {
      const sel = document.getElementById(id);
      if (sel && val != null) {
        sel.value = val;
        sel.dispatchEvent(new Event("change", { bubbles: true }));
      }
    }
  }

  function toNumber(v, fallback = 0) {
    if (v == null) return fallback;
    const s = String(v).trim().replace(/\./g, "").replace(",", ".");
    const n = Number(s);
    return Number.isFinite(n) ? n : fallback;
  }

  function normalizeAttachmentsList(list) {
    const arr = Array.isArray(list) ? list : [];
    return arr.filter(a => a && a.id).map(a => ({
      id: String(a.id),
      ts: a.ts || nowIso(),
      kind: a.kind || "delivery_note",
      name: a.name || "file",
      type: a.type || "",
      size: a.size || 0
    }));
  }

  function addAttachmentsToArticle(articleId, bookingIds, receipt) {
    try {
      const arts = loadArticles();
      const idx = arts.findIndex(a => String(a.id) === String(articleId));
      if (idx === -1) return;
      const art = arts[idx];
      const prev = Array.isArray(art.attachments) ? art.attachments : [];
      const base = normalizeAttachmentsList(receipt.attachments);

      const meta = [];
      for (const bId of (Array.isArray(bookingIds) ? bookingIds : [])) {
        for (const a of base) {
          meta.push({
            ...a,
            articleId: String(articleId),
            bookingId: bId != null ? String(bId) : null,
            supplier: receipt.supplier || "",
            deliveryNoteNo: receipt.deliveryNoteNo || "",
            orderNo: receipt.orderNo || "",
            deliveryNoteDate: receipt.deliveryNoteDate || ""
          });
        }
      }
      // Falls keine bookingIds bekannt: trotzdem einmal pro Artikel speichern
      if (!meta.length && base.length) {
        for (const a of base) {
          meta.push({
            ...a,
            articleId: String(articleId),
            bookingId: null,
            supplier: receipt.supplier || "",
            deliveryNoteNo: receipt.deliveryNoteNo || "",
            orderNo: receipt.orderNo || "",
            deliveryNoteDate: receipt.deliveryNoteDate || ""
          });
        }
      }

      art.attachments = prev.concat(meta);
      arts[idx] = art;
      saveArticles(arts);
    } catch (_) {}
  }

  async function storeSelectedFilesFromDialog(dlg) {
    const input = dlg ? dlg.querySelector("#invWeFiles") : null;
    const files = input && input.files ? Array.from(input.files) : [];
    if (!files.length) return [];

    const saved = [];
    for (const f of files) {
      try {
        const rec = await storeFileInDb(f, "delivery_note");
        if (rec && rec.id) {
          saved.push({
            id: rec.id,
            ts: rec.ts,
            kind: rec.kind || "delivery_note",
            name: rec.name,
            type: rec.type,
            size: rec.size
          });
        }
      } catch (_) {}
    }
    return saved;
  }

  async function postOneLine({ articleId, qty, unitPriceText, noteText, employee, receiptMeta }) {
    // 1) ensure row exists (filters all)
    const actionSelect = document.querySelector(`select.inv-action-select[data-id="${CSS.escape(String(articleId))}"]`);
    if (!actionSelect) {
      return { ok: false, error: t("Artikel in Tabelle nicht gefunden (Filter?)", "Item not found in table (filters?)") };
    }

    // 2) Open existing booking dialog by triggering the original handler
    window[BYPASS_FLAG] = true;
    actionSelect.value = "goods-in";
    actionSelect.dispatchEvent(new Event("change", { bubbles: true }));
    window[BYPASS_FLAG] = false;

    const dlg = await waitForDialogOpen("invBookingDialog", 4000);
    if (!dlg) {
      return { ok: false, error: t("Buchungsdialog konnte nicht geöffnet werden.", "Booking dialog could not be opened.") };
    }

    // 3) Fill booking fields
    const amountEl = document.getElementById("invBookingAmount");
    const employeeEl = document.getElementById("invBookingEmployee");
    const noteEl = document.getElementById("invBookingNote");
    const reasonEl = document.getElementById("invBookingReason");
    const priceEl = document.getElementById("invBookingUnitPrice");

    const dnNoEl = document.getElementById("invBookingDeliveryNoteNo");
    const dnDateEl = document.getElementById("invBookingDeliveryNoteDate");
    const supplierEl = document.getElementById("invBookingSupplier");


    if (amountEl) amountEl.value = String(qty);
    if (employeeEl) employeeEl.value = employee || "";

    // ensure reason = wareneingang if available
    if (reasonEl) {
      const hasWE = Array.from(reasonEl.options || []).some((o) => o.value === "wareneingang");
      reasonEl.value = hasWE ? "wareneingang" : (reasonEl.value || "wareneingang");
      reasonEl.dispatchEvent(new Event("change", { bubbles: true }));
    }

    if (noteEl) noteEl.value = noteText || "";

    // delivery note meta (required for reason=wareneingang)
    if (dnNoEl && receiptMeta && receiptMeta.deliveryNoteNo) dnNoEl.value = String(receiptMeta.deliveryNoteNo);
    if (dnDateEl && receiptMeta && receiptMeta.deliveryNoteDate) dnDateEl.value = String(receiptMeta.deliveryNoteDate);
    if (supplierEl && receiptMeta && receiptMeta.supplier) supplierEl.value = String(receiptMeta.supplier);

    // unit price optional
    if (priceEl && unitPriceText && String(unitPriceText).trim()) {
      priceEl.value = String(unitPriceText).trim();
    }

    // 4) Submit form
    const form = document.getElementById("invBookingForm");
    const beforeBookings = loadBookings();
    const beforeLen = beforeBookings.length;

    if (form && typeof form.requestSubmit === "function") {
      form.requestSubmit();
    } else if (form) {
      form.dispatchEvent(new Event("submit", { bubbles: true, cancelable: true }));
    }

    // wait close
    const closed = await waitForDialogClose("invBookingDialog", 8000);
    if (!closed) {
      return { ok: false, error: t("Buchung konnte nicht abgeschlossen werden.", "Booking could not be completed.") };
    }

    // 5) annotate newly created booking(s) in localStorage
    const after = loadBookings();
    const added = after.slice(beforeLen);
    const addedIds = [];

    if (!added.length) {
      return { ok: false, error: t("Keine Buchung erzeugt – evtl. wurde der Buchungsdialog abgebrochen oder Pflichtfelder fehlen.", "No booking created – the booking dialog may have been cancelled or required fields are missing.") };
    }

    if (added.length) {
      for (let i = after.length - added.length; i < after.length; i++) {
        const b = after[i];
        if (!b || typeof b !== "object") continue;

        b.receiptId = receiptMeta.id;
        b.deliveryNoteNo = receiptMeta.deliveryNoteNo;
        b.orderNo = receiptMeta.orderNo;
        b.supplier = receiptMeta.supplier;
        b.deliveryNoteDate = receiptMeta.deliveryNoteDate;
        b.receiptReason = receiptMeta.reason;
        b.receiptBookedBy = receiptMeta.bookedBy;

        // Lieferschein-Anlagen (Metadaten)
        if (Array.isArray(receiptMeta.attachments) && receiptMeta.attachments.length) {
          b.attachments = normalizeAttachmentsList(receiptMeta.attachments);
        }

        if (b.id != null) addedIds.push(b.id);
      }
      saveBookings(after);
    }

    return { ok: true, bookingIds: addedIds };
  }

  async function postReceiptFromDialog(dlg) {
    showError(dlg, "");

    const docNo = String(dlg.querySelector("#invWeDocNo")?.value || "").trim();
    const orderNo = String(dlg.querySelector("#invWeOrderNo")?.value || "").trim();
    const supplier = String(dlg.querySelector("#invWeSupplier")?.value || "").trim();
    const bookedBy = String(dlg.querySelector("#invWeBy")?.value || "").trim();
    const date = String(dlg.querySelector("#invWeDate")?.value || "").trim();
    const reason = String(dlg.querySelector("#invWeReason")?.value || "").trim();
    const note = String(dlg.querySelector("#invWeNote")?.value || "").trim();

if (!docNo) {
  showError(dlg, t("Bitte Lieferschein-Nr. eintragen.", "Please enter the delivery note number."));
  uiWarnRequired("Bitte Lieferschein-Nr. eintragen.", "Please enter the delivery note number.");
  try { dlg.querySelector("#invWeDocNo")?.focus(); } catch (_) {}
  return;
}

    if (!bookedBy) {
      showError(dlg, t("Bitte Name/Kürzel eingeben.", "Please enter name/initials."));
      return;
    }
    if (!reason) {
      showError(dlg, t("Bitte Grund eingeben.", "Please enter a reason."));
      return;
    }

    const lineNodes = Array.from(dlg.querySelectorAll(".inv-we-line"));
    if (!lineNodes.length) {
      showError(dlg, t("Bitte mindestens eine Position hinzufügen.", "Please add at least one line item."));
      return;
    }

    const articles = loadArticles();
    const lines = [];
    for (const node of lineNodes) {
      const aid = String(node.querySelector("select.inv-we-article")?.value || "").trim();
      const qty = toNumber(node.querySelector("input.inv-we-qty")?.value, 0);
      const price = String(node.querySelector("input.inv-we-price")?.value || "").trim();

      if (!aid) continue;
      if (!qty || qty <= 0) continue;

      const art = articles.find((a) => String(a.id) === String(aid));
      if (!art) continue;

      lines.push({
        articleId: String(aid),
        amount: qty,
        unitPrice: price,
      });
    }

    if (!lines.length) {
      showError(dlg, t("Bitte gültige Positionen wählen (Artikel + Menge > 0).", "Please choose valid line items (item + qty > 0)."));
      return;
    }

    // receipt id
    const rid = `we_${new Date().toISOString().slice(0,10)}_${Date.now().toString(36)}`;

    // Dateien VOR dem Buchen speichern (optional)
    const postBtn = dlg.querySelector("#invWePost");
    const prevText = postBtn ? postBtn.textContent : "";
    if (postBtn) {
      postBtn.disabled = true;
      postBtn.textContent = t("Speichern…", "Saving…");
    }

    const savedAttachments = await storeSelectedFilesFromDialog(dlg);

    const receipt = {
      id: rid,
      kind: "wareneingang",
      ts: nowIso(),
      deliveryNoteNo: docNo,
      orderNo: orderNo,
      supplier: supplier,
      deliveryNoteDate: date,
      bookedBy: bookedBy,
      reason: reason,
      note: note,
      attachments: savedAttachments, // nur Metadaten, Datei selbst liegt in IndexedDB
      lines: lines.map((l) => ({ articleId: l.articleId, amount: l.amount, unitPrice: l.unitPrice })),
      bookingIds: [],
    };

    if (postBtn) {
      postBtn.textContent = t("Buchen…", "Posting…");
    }

    // Snapshot + set filters to show all
    const snap = snapshotFilters();
    setFiltersAll();
    await sleep(30);

    try {
      for (let i = 0; i < lines.length; i++) {
        const l = lines[i];

        if (postBtn) {
          postBtn.textContent = t(`Buchen… (${i + 1}/${lines.length})`, `Posting… (${i + 1}/${lines.length})`);
        }

        // note for the booking itself
        const bookingNote = [
          `WE ${docNo}`,
          orderNo ? `PO ${orderNo}` : "",
          supplier ? supplier : "",
          reason,
          note ? note : ""
        ].filter(Boolean).join(" | ");

        const res = await postOneLine({
          articleId: l.articleId,
          qty: l.amount,
          unitPriceText: l.unitPrice,
          noteText: bookingNote,
          employee: bookedBy,
          receiptMeta: receipt,
        });

        if (!res.ok) {
          throw new Error(res.error || "WE failed");
        }

        if (res.bookingIds && res.bookingIds.length) {
          receipt.bookingIds.push(...res.bookingIds);

          // Anlagen an Artikel hängen (damit Details das ggf. anzeigen kann)
          if (receipt.attachments && receipt.attachments.length) {
            addAttachmentsToArticle(l.articleId, res.bookingIds, receipt);
          }
        } else if (receipt.attachments && receipt.attachments.length) {
          // wenn keine bookingId ermittelt: trotzdem einmal pro Artikel speichern
          addAttachmentsToArticle(l.articleId, [], receipt);
        }
      }

      // Save receipt
      const receipts = loadReceipts();
      receipts.push(receipt);
      saveReceipts(receipts);

dlg.close();

const extraDe = orderNo ? `\nAuftragsnr.: ${orderNo}` : "";
const extraEn = orderNo ? `\nOrder no.: ${orderNo}` : "";

uiInfo(
  `Wareneingang gebucht.\n\nLieferschein: ${docNo}${extraDe}\nPositionen: ${lines.length}`,
  `Goods receipt posted.\n\nDelivery note: ${docNo}${extraEn}\nLines: ${lines.length}`,
  "✅ Erfolg",
  "✅ Success"
);

    } catch (err) {
      console.error("[WE add-on] failed:", err);
      showError(dlg, String(err && err.message ? err.message : err));
    } finally {
      restoreFilters(snap);
      if (postBtn) {
        postBtn.disabled = false;
        postBtn.textContent = prevText || t("Wareneingang buchen", "Post goods receipt");
      }
    }
  }

  function openWeDialog(prefillArticleId) {
    const dlg = ensureWeDialog();

    // Ensure at least one line
    const linesBox = dlg.querySelector("#invWeLines");
    const addBtn = dlg.querySelector("#invWeAddLine");
    const articles = loadArticles();

    // clear existing lines
    if (linesBox) linesBox.innerHTML = "";
    if (linesBox) linesBox.appendChild(createLineRow(articles, prefillArticleId));

    // wire add button
    if (addBtn && !addBtn.dataset.bound) {
      addBtn.dataset.bound = "1";
      addBtn.addEventListener("click", () => {
        const arts = loadArticles();
        const box = dlg.querySelector("#invWeLines");
        if (box) box.appendChild(createLineRow(arts, null));
      });
    }

    // Default date
    const dateEl = dlg.querySelector("#invWeDate");
    if (dateEl && !dateEl.value) {
      dateEl.value = new Date().toISOString().slice(0, 10);
    }

    showError(dlg, "");
    updateWeFilesInfo(dlg);

    // bind submit once
    const form = dlg.querySelector("#invWeForm");
    if (form && !form.dataset.bound) {
      form.dataset.bound = "1";
      form.addEventListener("submit", (e) => {
        e.preventDefault();
        postReceiptFromDialog(dlg);
      });
    }

    // show
    try {
      dlg.showModal();
    } catch (_) {
      dlg.setAttribute("open", "");
    }
  }

  // ---- Intercept Aktion "goods-in" ------------------------------------
  function bindIntercept() {
    if (window.__opsWeAddonBound) return;
    window.__opsWeAddonBound = true;

    document.addEventListener(
      "change",
      (e) => {
        try {
          if (window[BYPASS_FLAG]) return;
          const target = e.target;
          if (!target) return;

          const select = target.closest && target.closest("select.inv-action-select");
          if (!select) return;
          if (select.value !== "goods-in") return;

          // stop existing handler
          e.preventDefault();
          e.stopPropagation();
          // NOTE: do not call stopImmediatePropagation() so other add-ons can still react

          const id = select.getAttribute("data-id");
          select.value = ""; // reset to placeholder

          if (!id) return;
          openWeDialog(String(id));
        } catch (err) {
          console.error("[WE add-on] intercept error:", err);
        }
      },
      true // capture
    );
  }

  // ---- Boot ------------------------------------------------------------
  // ---- Boot ------------------------------------------------------------
  function boot() {
    try {
      bindIntercept();
      setupAttachmentHooks();

      // Öffnen über festen Button (oder andere UI): window.dispatchEvent(new CustomEvent("opsWeOpenDialog"))
      if (!window.__opsWeOpenEventBound) {
        window.__opsWeOpenEventBound = true;
        window.addEventListener("opsWeOpenDialog", (ev) => {
          try {
            const prefill = ev && ev.detail ? ev.detail.prefillArticleId : null;
            openWeDialog(prefill ? String(prefill) : null);
          } catch (e) {
            try { openWeDialog(null); } catch (_) {}
          }
        });
      }

      window.__opsWeAddon = { version: "v3-files", ts: Date.now() };
    } catch (e) {
      console.error("[WE add-on] boot failed:", e);
    }
  }

  if (document.readyState === "loading") {
    document.addEventListener("DOMContentLoaded", boot);
  } else {
    boot();
  }
})();

