// translations.js (DE/EN) — persistent language + DOM translations + dynamic updates
// Exposes: getOpsLang(), setOpsLang(lang), tLang(de,en), applyI18n(root?)
(() => {
  'use strict';

  const KEY = 'ops_lang';
  const normalize = (l) => String(l || '').toLowerCase().startsWith('en') ? 'en' : 'de';

  function getOpsLang() {
    try {
      const ls = localStorage.getItem(KEY);
      if (ls) return normalize(ls);
    } catch (_) {}
    return normalize(document.documentElement.getAttribute('lang') || 'de');
  }

  function tLang(de, en) {
    return getOpsLang() === 'en' ? (en ?? de) : (de ?? en);
  }

  function dispatchLangChange() {
    try {
      window.dispatchEvent(new CustomEvent('ops:langchange', { detail: { lang: getOpsLang() } }));
    } catch (_) {
      try { window.dispatchEvent(new Event('ops:langchange')); } catch (_) {}
    }
  }

  // --- Dictionary for elements that already use data-i18n keys ---------
  const DICT = {
    de: {
      headline: 'Lagerverwaltung',
      subline:
        'Ersatzteile & Materialien in mehreren Lagern verwalten –\n' +
        'mit Buchungen, Inventur und Sicherheitskennzeichnung.',
      navPortal: 'Portal',
      navDashboard: 'Dashboard',
      navShiftlog: 'Schichtbuch',
      navTechnicians: 'Techniker',
      navEngineering: 'Engineering',
      btnCreate: '➕ Artikel anlegen',
      btnPrint: '🖨️ Tabelle drucken',
      btnExport: '📤 CSV-Export',
      btnImport: '📥 CSV-Import',
      btnInventory: '📦 Inventur',
      btnColumns: '🧱 Spalten',
      btnDelete: '🗑️ Tabelle löschen',
      btnHistory: '📜 Historie'

    },
    en: {
      headline: 'Inventory',
      subline:
        'Manage spare parts & materials across multiple warehouses –\n' +
        'with postings, stocktaking and safety labeling.',
      navPortal: 'Portal',
      navDashboard: 'Dashboard',
      navShiftlog: 'Shift log',
      navTechnicians: 'Technicians',
      navEngineering: 'Engineering',
      btnCreate: '➕ Create item',
      btnPrint: '🖨️ Print table',
      btnExport: '📤 CSV export',
      btnImport: '📥 CSV import',
      btnInventory: '📦 Stocktaking',
      btnColumns: '🧱 Columns',
      btnDelete: '🗑️ Delete table',
      btnHistory: '📜 History'
    }
  };

  // --- Selector-based translations (covers static HTML + dynamic inserts)
  // NOTE: we intentionally translate only UI labels/options, never user-entered values.
  const RULES = [
    // Top actions
    { sel: '#invWeTopBtn', de: '📥 Wareneingang buchen', en: '📥 Post goods receipt' },
    { sel: '#invWoTopBtn', de: '📤 Warenausgang buchen', en: '📤 Post goods issue' },
    { sel: '#invReceiptsBtn', de: '🧾 Belegliste', en: '🧾 Receipts' },
    { sel: '#invBanfBtn', de: '🧾 BANF / Bestellung', en: '🧾 Requisition / Order' },
    { sel: '#invPermsBtn', de: '🔐 Berechtigungen', en: '🔐 Permissions' },
    { sel: '#invAdminBtn', de: '🔒 Admin', en: '🔒 Admin' },
	
	// Details dialog (invDetailDialog)
{ sel: '#invDetailDialog .detail-title', de: 'Lagerartikel – Einzeldatenblatt', en: 'Inventory item – datasheet' },
{ sel: '#invDetailDialog h2', de: 'Artikeldetails', en: 'Item details' },

{ sel: '#invDetailDialog .detail-header-meta > div:nth-child(1) .meta-label', de: 'Datum:', en: 'Date:' },
{ sel: '#invDetailDialog .detail-header-meta > div:nth-child(2) .meta-label', de: 'Uhrzeit:', en: 'Time:' },

{ sel: '#detailPrintBtn', de: '🖨 Drucken', en: '🖨 Print' },
{ sel: '#detailCloseBtn', de: '✖ Schließen', en: '✖ Close' },

// Table labels in details sheet
{ sel: '#invDetailDialog .detail-table tr:nth-child(1) th', de: 'Artikelnummer', en: 'Item number' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(2) th', de: 'Interne Artikelnummer', en: 'Internal item number' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(3) th', de: 'Bezeichnung', en: 'Description' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(4) th', de: 'Hersteller', en: 'Manufacturer' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(5) th', de: 'Lieferant', en: 'Supplier' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(6) th', de: 'Kategorie', en: 'Category' },

{ sel: '#invDetailDialog .detail-table tr:nth-child(7) th', de: 'Lager', en: 'Warehouse' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(8) th', de: 'Regal', en: 'Rack' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(9) th', de: 'Fach', en: 'Compartment' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(10) th', de: 'Position', en: 'Position' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(11) th', de: 'Niveau', en: 'Level' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(12) th', de: 'Lagerort', en: 'Storage location' },

{ sel: '#invDetailDialog .detail-table tr:nth-child(13) th', de: 'Bestand', en: 'Stock' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(14) th', de: 'Mindestbestand', en: 'Minimum stock' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(15) th', de: 'Einheit', en: 'Unit' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(16) th', de: 'Wert/Einheit', en: 'Value/unit' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(17) th', de: 'Währung', en: 'Currency' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(18) th', de: 'Gesamtpreis', en: 'Total price' },

{ sel: '#invDetailDialog .detail-table tr:nth-child(19) th', de: 'Technischer Platz', en: 'Functional location' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(20) th', de: 'Equipment', en: 'Equipment' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(21) th', de: 'Anlage', en: 'Plant' },

{ sel: '#invDetailDialog .detail-table tr:nth-child(22) th', de: 'Notiz', en: 'Note' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(23) th', de: 'Sicherheitsstufe', en: 'Safety level' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(24) th', de: 'Kostenstelle', en: 'Cost center' },


    // Optional buttons (inserted by JS)
    { sel: '#invSealBtn', de: '🔐 Monatsabschluss / Seal', en: '🔐 Month close / Seal' },
    { sel: '#invMonthExportBtn', de: '📦 Monatsmappe exportieren', en: '📦 Export monthly package' },
    { sel: '#invMonthProtocolBtn', de: '🧾 Monatsabschluss-Protokoll', en: '🧾 Month close protocol' },
    { sel: '#invMonthExcelTemplateBtn', de: '📄 Excel-Template (Monatsabschluss)', en: '📄 Excel template (month close)' },
    { sel: '#invReportingBtn', de: '📊 Reporting', en: '📊 Reporting' },

    // Filters / search
    { sel: 'label[for="invSearch"]', de: 'Suche', en: 'Search' },
    { sel: '#invSearch', attr: 'placeholder', de: 'Artikelnummer, Bezeichnung, Hersteller...', en: 'Item no., description, manufacturer...' },

    { sel: 'label[for="invFilterLager"]', de: 'Lager', en: 'Warehouse' },
    { sel: 'label[for="invFilterKategorie"]', de: 'Kategorie', en: 'Category' },
    { sel: 'label[for="invFilterBestand"]', de: 'Bestand', en: 'Stock' },

    // Dynamic selects: translate the "All" option
    { sel: '#invFilterLager option[value=""]', de: 'Alle', en: 'All' },
    { sel: '#invFilterKategorie option[value=""]', de: 'Alle', en: 'All' },

    // Stock filter options
    { sel: '#invFilterBestand option[value=""]', de: 'Alle', en: 'All' },
    { sel: '#invFilterBestand option[value="unterMin"]', de: 'Unter Mindestbestand', en: 'Below minimum' },
    { sel: '#invFilterBestand option[value="gleichMin"]', de: '= Mindestbestand', en: '= Minimum' },
    { sel: '#invFilterBestand option[value="ueberMin"]', de: 'Über Mindestbestand', en: 'Above minimum' },
    { sel: '#invFilterBestand option[value="null"]', de: 'Bestand = 0', en: 'Stock = 0' },
    { sel: '#invFilterBestand option[value="positiv"]', de: 'Bestand > 0', en: 'Stock > 0' },

    // Blocked filter (injected by JS)
    { sel: 'label[for="invFilterBlocked"]', de: '🔒 Sperre', en: '🔒 Block' },
    { sel: '#invFilterBlocked option[value=""]', de: 'Alle', en: 'All' },
    { sel: '#invFilterBlocked option[value="blocked"]', de: 'Nur gesperrt', en: 'Blocked only' },
    { sel: '#invFilterBlocked option[value="unblocked"]', de: 'Nur aktiv', en: 'Active only' },

    // Location filters
    { sel: '#invLagerLocationRow .inv-location-group-label', de: 'Lagerplatz', en: 'Location' },
    { sel: 'label[for="invRegal"]', de: 'Regal', en: 'Rack' },
    { sel: '#invRegal', attr: 'placeholder', de: 'z.B. R01', en: 'e.g. R01' },
    { sel: 'label[for="invFach"]', de: 'Fach', en: 'Compartment' },
    { sel: '#invFach', attr: 'placeholder', de: 'z.B. F03', en: 'e.g. F03' },
    { sel: 'label[for="invPosition"]', de: 'Position / Fachnummer', en: 'Position / Bin' },
    { sel: '#invPosition', attr: 'placeholder', de: 'z.B. 03-12', en: 'e.g. 03-12' },
    { sel: 'label[for="invNiveau"]', de: 'Niveau', en: 'Level' },
    { sel: '#invNiveau option[value=""]', de: 'Alle', en: 'All' },
    { sel: '#invNiveau option[value="unten"]', de: 'unten', en: 'bottom' },
    { sel: '#invNiveau option[value="mitte"]', de: 'mitte', en: 'middle' },
    { sel: '#invNiveau option[value="oben"]', de: 'oben', en: 'top' },

    // KPI cards
    { sel: '.inv-kpi .kpi-card:nth-child(1) .kpi-label', de: 'Artikel', en: 'Items' },
    { sel: '.inv-kpi .kpi-card:nth-child(1) .kpi-sub', de: 'Gesamt im Filter', en: 'Total in filter' },

    { sel: '.inv-kpi .kpi-card:nth-child(2) .kpi-label', de: 'Unter Mindestbestand', en: 'Below minimum' },
    { sel: '.inv-kpi .kpi-card:nth-child(2) .kpi-sub', de: 'Handlungsbedarf', en: 'Action needed' },

    { sel: '.inv-kpi .kpi-card:nth-child(3) .kpi-label', de: 'Bestand = 0', en: 'Stock = 0' },
    { sel: '.inv-kpi .kpi-card:nth-child(3) .kpi-sub', de: 'kritische Leerlager', en: 'Critical out-of-stock' },

    { sel: '.inv-kpi .kpi-card:nth-child(4) .kpi-label', de: 'Materialwert (ca.)', en: 'Material value (approx.)' },
    { sel: '.inv-kpi .kpi-card:nth-child(4) .kpi-sub', de: 'Bestand × Einzelpreis', en: 'Stock × unit price' },

    // Table headers
    { sel: 'th[data-col="aktionen"]', de: 'Aktionen', en: 'Actions' },
    { sel: 'th[data-col="artikelnummer"]', de: 'Artikelnummer', en: 'Item no.' },
    { sel: 'th[data-col="interneArtikelnummer"]', de: 'Interne Artikelnr.', en: 'Internal item no.' },
    { sel: 'th[data-col="bezeichnung"]', de: 'Bezeichnung', en: 'Description' },
    { sel: 'th[data-col="hersteller"]', de: 'Hersteller', en: 'Manufacturer' },
    { sel: 'th[data-col="lieferant"]', de: 'Lieferant', en: 'Supplier' },
    { sel: 'th[data-col="kategorie"]', de: 'Kategorie', en: 'Category' },
    { sel: 'th[data-col="lager"]', de: 'Lager', en: 'Warehouse' },
    { sel: 'th[data-col="mindestbestand"]', de: 'Mindestbestand', en: 'Min' },
    { sel: 'th[data-col="bestand"]', de: 'Bestand', en: 'Stock' },
    { sel: 'th[data-col="einheit"]', de: 'Einheit', en: 'Unit' },
    { sel: 'th[data-col="einzelpreis"]', de: 'Einzelpreis', en: 'Unit price' },
    { sel: 'th[data-col="gesamtpreis"]', de: 'Gesamtpreis', en: 'Total' },
    { sel: 'th[data-col="notizen"]', de: 'Notizen', en: 'Notes' },
    { sel: 'th[data-col="technischerPlatz"]', de: 'TechnischerPlatz', en: 'Technical location' },
    { sel: 'th[data-col="equipment"]', de: 'Equipment', en: 'Equipment' },
    { sel: 'th[data-col="anlage"]', de: 'Anlage', en: 'Plant/Machine' },
    { sel: 'th[data-col="regal"]', de: 'Regal', en: 'Rack' },
    { sel: 'th[data-col="fach"]', de: 'Fach', en: 'Compartment' },
    { sel: 'th[data-col="position"]', de: 'Position / Fachnummer', en: 'Position / Bin' },
    { sel: 'th[data-col="niveau"]', de: 'Niveau', en: 'Level' },
    { sel: 'th[data-col="lagerort"]', de: 'Lagerort', en: 'Storage location' },

    // Action dropdown (per row)
    { sel: 'select.inv-action-select option[value=""]', de: 'Aktionen …', en: 'Actions …' },
    { sel: 'select.inv-action-select option[value="edit"]', de: '✏️ Bearbeiten', en: '✏️ Edit' },
    { sel: 'select.inv-action-select option[value="unblock"]', de: '🔓 Sperre aufheben', en: '🔓 Unblock' },
    { sel: 'select.inv-action-select option[value="block"]', de: '🔒 Für Zugang/BANF sperren', en: '🔒 Block receipts / requisition' },
    { sel: 'select.inv-action-select option[value="goods-in"]', de: '📥 Wareneingang', en: '📥 Goods receipt' },
    { sel: 'select.inv-action-select option[value="book-in"]', de: '➕ Zugang (Korrektur)', en: '➕ Receipt (correction)' },
    { sel: 'select.inv-action-select option[value="goods-out"]', de: '📤 Warenausgang', en: '📤 Goods issue' },
    { sel: 'select.inv-action-select option[value="book-out"]', de: '➖ Entnahme', en: '➖ Withdrawal' },
    { sel: 'select.inv-action-select option[value="inventory"]', de: '📦 Inventur', en: '📦 Stocktaking' },
    { sel: 'select.inv-action-select option[value="details"]', de: '📄 Details', en: '📄 Details' },
    { sel: 'select.inv-action-select option[value="banf"]', de: '🧾 BANF / Bestellung', en: '🧾 Requisition / Order' },

    // Detail dialog header
// Detail dialog (invDetailDialog)
{ sel: '#invDetailDialog .detail-title', de: 'Lagerartikel – Einzeldatenblatt', en: 'Inventory item – datasheet' },
{ sel: '#invDetailDialog h2', de: 'Artikeldetails', en: 'Item details' },
{ sel: '#invDetailDialog .detail-header-meta > div:nth-child(1) .meta-label', de: 'Datum:', en: 'Date:' },
{ sel: '#invDetailDialog .detail-header-meta > div:nth-child(2) .meta-label', de: 'Uhrzeit:', en: 'Time:' },
{ sel: '#detailPrintBtn', de: '🖨 Drucken', en: '🖨 Print' },
{ sel: '#detailCloseBtn', de: '✖ Schließen', en: '✖ Close' },

{ sel: '#invDetailDialog .detail-table tr:nth-child(1) th', de: 'Artikelnummer', en: 'Item number' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(2) th', de: 'Interne Artikelnummer', en: 'Internal item number' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(3) th', de: 'Bezeichnung', en: 'Description' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(4) th', de: 'Hersteller', en: 'Manufacturer' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(5) th', de: 'Lieferant', en: 'Supplier' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(6) th', de: 'Kategorie', en: 'Category' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(7) th', de: 'Lager', en: 'Warehouse' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(8) th', de: 'Regal', en: 'Rack' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(9) th', de: 'Fach', en: 'Compartment' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(10) th', de: 'Position', en: 'Position' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(11) th', de: 'Niveau', en: 'Level' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(12) th', de: 'Lagerort', en: 'Storage location' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(13) th', de: 'Bestand', en: 'Stock' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(14) th', de: 'Mindestbestand', en: 'Minimum stock' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(15) th', de: 'Einheit', en: 'Unit' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(16) th', de: 'Wert/Einheit', en: 'Value/unit' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(17) th', de: 'Währung', en: 'Currency' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(18) th', de: 'Gesamtpreis', en: 'Total price' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(19) th', de: 'Technischer Platz', en: 'Functional location' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(20) th', de: 'Equipment', en: 'Equipment' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(21) th', de: 'Anlage', en: 'Plant/Machine' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(22) th', de: 'Notiz', en: 'Note' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(23) th', de: 'Sicherheitsstufe', en: 'Safety level' },
{ sel: '#invDetailDialog .detail-table tr:nth-child(24) th', de: 'Kostenstelle', en: 'Cost center' },


    // Article dialog
    { sel: '#invArticleDialog h3', de: 'Artikel', en: 'Item' },
    { sel: 'label[for="invArtikelnummer"]', de: 'Artikelnummer', en: 'Item no.' },
    { sel: 'label[for="invInterneArtikelnummer"]', de: 'Interne Artikelnummer', en: 'Internal item no.' },
    { sel: 'label[for="invLieferant"]', de: 'Lieferant', en: 'Supplier' },
    { sel: 'label[for="invEinheit"]', de: 'Einheit', en: 'Unit' },
    { sel: 'label[for="invBezeichnung"]', de: 'Bezeichnung', en: 'Description' },
    { sel: 'label[for="invHersteller"]', de: 'Hersteller', en: 'Manufacturer' },
    { sel: 'label[for="invKategorie"]', de: 'Kategorie', en: 'Category' },
    { sel: 'label[for="invLager"]', de: 'Lager', en: 'Warehouse' },
    { sel: '#invLager option[value=""]', de: '– wählen –', en: '– select –' },
    { sel: 'label[for="invLagerart"]', de: 'Lagerart (optional)', en: 'Warehouse type (optional)' },
    { sel: '#invLagerart option[value=""]', de: '– optional –', en: '– optional –' },

    { sel: 'label[for="invArticleRegal"]', de: 'Regal', en: 'Rack' },
    { sel: 'label[for="invArticleFach"]', de: 'Fach', en: 'Compartment' },
    { sel: 'label[for="invArticlePosition"]', de: 'Position / Fachnummer', en: 'Position / Bin' },
    { sel: 'label[for="invArticleNiveau"]', de: 'Niveau', en: 'Level' },
    { sel: '#invArticleNiveau option[value=""]', de: '– wählen –', en: '– select –' },
    { sel: '#invArticleNiveau option[value="unten"]', de: 'unten', en: 'bottom' },
    { sel: '#invArticleNiveau option[value="mitte"]', de: 'mitte', en: 'middle' },
    { sel: '#invArticleNiveau option[value="oben"]', de: 'oben', en: 'top' },

    { sel: 'label[for="invLagerplatztyp"]', de: 'Lagerplatz-Typ', en: 'Storage type' },
    { sel: '#invLagerplatztyp option[value=""]', de: '– wählen –', en: '– select –' },
    { sel: '#invLagerplatztyp option[value="Bodenlager"]', de: 'Bodenlager / Fläche', en: 'Floor storage / area' },
    { sel: '#invLagerplatztyp option[value="Regal"]', de: 'Regal', en: 'Rack' },
    { sel: '#invLagerplatztyp option[value="Schwerlastregal"]', de: 'Schwerlastregal', en: 'Heavy-duty rack' },
    { sel: '#invLagerplatztyp option[value="Werkzeugschrank"]', de: 'Werkzeugschrank', en: 'Tool cabinet' },
    { sel: '#invLagerplatztyp option[value="Gefahrstoffschrank"]', de: 'Gefahrstoffschrank', en: 'Hazard cabinet' },
    { sel: '#invLagerplatztyp option[value="Aussenfläche"]', de: 'Außenfläche', en: 'Outdoor area' },
    { sel: '#invLagerplatztyp option[value="Sonstiges"]', de: 'Sonstiges', en: 'Other' },

    { sel: 'label[for="invSicherheitsstufe"]', de: 'Sicherheitsstufe', en: 'Safety class' },
    { sel: '#invSicherheitsstufe option[value=""]', de: 'Normal', en: 'Normal' },
    { sel: '#invSicherheitsstufe option[value="Gefahrstoff"]', de: 'Gefahrstoff', en: 'Hazardous' },
    { sel: '#invSicherheitsstufe option[value="Umweltgefährlich"]', de: 'Umweltgefährlich', en: 'Environmental hazard' },
    { sel: '#invSicherheitsstufe option[value="Sicherheitsrelevant"]', de: 'Sicherheitsrelevant', en: 'Safety critical' },

    { sel: 'label[for="invLagerort"]', de: 'Lagerort', en: 'Storage location' },
    { sel: 'label[for="invMindestbestand"]', de: 'Mindestbestand', en: 'Minimum stock' },
    { sel: 'label[for="invBestand"]', de: 'Ist-Bestand', en: 'Current stock' },
    { sel: 'label[for="invWareneingang"]', de: 'Wareneingangsdatum (optional)', en: 'Goods receipt date (optional)' },
    { sel: 'label[for="invWertEinzeln"]', de: 'Wert pro Einheit (optional)', en: 'Unit value (optional)' },
    { sel: 'label[for="invWaehrung"]', de: 'Währung', en: 'Currency' },
    { sel: 'label[for="invNotizen"]', de: 'Notizen', en: 'Notes' },
    { sel: 'label[for="invTechnischerPlatz"]', de: 'Technischer Platz', en: 'Technical location' },
    { sel: 'label[for="invAnlage"]', de: 'Anlage / Maschine', en: 'Plant / Machine' },

    { sel: '#invArticleCancel', de: 'Abbrechen', en: 'Cancel' },
    { sel: '#invArticleDialog button[type="submit"]', de: 'Speichern', en: 'Save' },

    // Booking dialog
    { sel: '#invBookingArticleLabel', de: 'Buchung', en: 'Posting' },
    { sel: 'label[for="invBookingUnitPrice"]', de: 'Einzelpreis', en: 'Unit price' },
    { sel: 'label[for="invBookingCurrency"]', de: 'Währung', en: 'Currency' },
    { sel: 'label[for="invBookingAmount"]', de: 'Menge', en: 'Quantity' },
    { sel: 'label[for="invBookingReason"]', de: 'Grund', en: 'Reason' },
    { sel: '#invBookingReason option[value="wareneingang"]', de: 'Wareneingang', en: 'Goods receipt' },
    { sel: '#invBookingReason option[value="entnahme"]', de: 'Entnahme', en: 'Withdrawal' },
    { sel: '#invBookingReason option[value="warenausgang"]', de: 'Warenausgang', en: 'Goods issue' },
    { sel: '#invBookingReason option[value="umlagerung"]', de: 'Umlagerung', en: 'Transfer' },
    { sel: '#invBookingReason option[value="korrektur"]', de: 'Korrektur', en: 'Correction' },

    { sel: 'label[for="invBookingTargetLager"]', de: 'Ziel-Lager (optional)', en: 'Target warehouse (optional)' },
    { sel: '#invBookingTargetLager option[value=""]', de: '– wählen –', en: '– select –' },
    { sel: 'label[for="invBookingTargetLagerort"]', de: 'Ziel-Lagerort (optional)', en: 'Target location (optional)' },

    { sel: 'label[for="invBookingCostCenter"]', de: 'Kostenstelle (optional)', en: 'Cost center (optional)' },
    { sel: 'label[for="invBookingEmployee"]', de: 'Mitarbeiter (optional)', en: 'Employee (optional)' },
    { sel: 'label[for="invBookingNote"]', de: 'Notiz (optional)', en: 'Note (optional)' },
    { sel: '#invBookingCancel', de: 'Abbrechen', en: 'Cancel' },
    { sel: '#invBookingDialog button[type="submit"]', de: 'Buchen', en: 'Post' },

    // Inventory dialog
    { sel: '#invInventoryArticleLabel', de: 'Inventur', en: 'Stocktaking' },
    { sel: '#invInventoryDialog p.tiny', de: 'Systembestand (Soll):', en: 'System stock (expected):' },
    { sel: 'label[for="invInventoryCounted"]', de: 'Gezählter Bestand (Ist)', en: 'Counted stock (actual)' },
    { sel: 'label[for="invInventoryNote"]', de: 'Notiz / Begründung (optional)', en: 'Note / justification (optional)' },
    { sel: '#invInventoryCancel', de: 'Abbrechen', en: 'Cancel' },
    { sel: '#invInventoryDialog button[type="submit"]', de: 'Inventur buchen', en: 'Post stocktaking' },

    // Columns dialog
    { sel: '#invColumnsDialog h3', de: 'Spalten auswählen', en: 'Select columns' },
    { sel: '#invColumnsReset', de: 'Alle einblenden', en: 'Show all' },
    { sel: '#invColumnsClose', de: 'Schließen', en: 'Close' },

    // Audit dialog
    { sel: '#auditHistoryDialog h3', de: '📜 Audit-Historie (read-only)', en: '📜 Audit history (read-only)' },
    { sel: '#invAuditExportBtn', de: 'Audit exportieren (JSON)', en: 'Export audit (JSON)' },
    { sel: '#auditHistoryDialog label.btn', de: 'Audit importieren (JSON)', en: 'Import audit (JSON)' },
    { sel: '#auditHistoryDialog th:nth-child(1)', de: 'Zeit', en: 'Time' },
    { sel: '#auditHistoryDialog th:nth-child(2)', de: 'Typ', en: 'Type' },
    { sel: '#auditHistoryDialog th:nth-child(3)', de: 'Artikel', en: 'Item' },
    { sel: '#auditHistoryDialog th:nth-child(4)', de: 'Menge', en: 'Qty' },
    { sel: '#auditHistoryDialog th:nth-child(5)', de: 'Preis', en: 'Price' },
    { sel: '#auditHistoryDialog th:nth-child(6)', de: 'Bearbeiter', en: 'User' },
    { sel: '#auditHistoryDialog th:nth-child(7)', de: 'Details', en: 'Details' },
    { sel: '#auditHistoryDialog button.btn', de: 'Schließen', en: 'Close' },

    // Detail dialog logo alt
    { sel: '#invDetailDialog img.detail-logo', attr: 'alt', de: 'Firmenlogo', en: 'Company logo' },
  ];

  function applyDict(root) {
    const lang = getOpsLang();
    const map = DICT[lang] || {};
    (root || document).querySelectorAll('[data-i18n]').forEach((el) => {
      const key = el.getAttribute('data-i18n');
      if (!key) return;
      const val = map[key];
      if (val == null) return;

      // Do not touch input values (only placeholder where it makes sense)
      if (el instanceof HTMLInputElement || el instanceof HTMLTextAreaElement) {
        if (el.hasAttribute('placeholder')) el.setAttribute('placeholder', val);
        return;
      }
      el.textContent = val;
    });
  }

  function applyRules(root) {
    const scope = root || document;
    for (const r of RULES) {
      let nodes;
      try { nodes = scope.querySelectorAll(r.sel); } catch (_) { continue; }
      if (!nodes || !nodes.length) continue;
      const val = tLang(r.de, r.en);
      nodes.forEach((el) => {
        try {
          if (r.attr) el.setAttribute(r.attr, val);
          else {
            // Special case: inventory dialog tiny line starts with a prefix and contains a span.
            if (r.sel === '#invInventoryDialog p.tiny') {
              // keep the current number span
              const span = el.querySelector('span');
              el.textContent = val + ' ';
              if (span) el.appendChild(span);
              return;
            }
            el.textContent = val;
          }
        } catch (_) {}
      });
    }

    // Translate status tiny labels rendered in rows (optional)
    try {
      const lang = getOpsLang();
      scope.querySelectorAll('div.tiny').forEach((el) => {
        const txt = (el.textContent || '').trim();
        if (!txt) return;
        if (lang === 'en') {
          if (txt === '❗ Bestand = 0') el.textContent = '❗ Stock = 0';
          if (txt === '⚠️ unter Mindestbestand') el.textContent = '⚠️ below minimum';
        } else {
          if (txt === '❗ Stock = 0') el.textContent = '❗ Bestand = 0';
          if (txt === '⚠️ below minimum') el.textContent = '⚠️ unter Mindestbestand';
        }
      });
    } catch (_) {}
  }

  function applyI18n(root) {
    applyDict(root);
    applyRules(root);
  }

  function setOpsLang(lang) {
    const l = normalize(lang);
    try { localStorage.setItem(KEY, l); } catch (_) {}
    document.documentElement.setAttribute('lang', l);
    try { applyI18n(document); } catch (_) {}
    dispatchLangChange();
    return l;
  }

  // Expose globals
  window.getOpsLang = getOpsLang;
  window.setOpsLang = setOpsLang;
  window.tLang = tLang;
  window.applyI18n = applyI18n;

  // Initialize ASAP
  try {
    const l = getOpsLang();
    document.documentElement.setAttribute('lang', l);
  } catch (_) {}

const schedule = (() => {
  let scheduled = false;
  const pending = new Set();

  const request = (cb) => {
    if (window.requestIdleCallback) return requestIdleCallback(cb, { timeout: 250 });
    return setTimeout(() => cb({ didTimeout: true, timeRemaining: () => 0 }), 50);
  };

  function compactRoots(roots) {
    // Keep only the outermost roots (remove children if parent is already included)
    const list = Array.from(roots).filter((n) => n && n.nodeType === 1);
    const compact = [];
    for (const r of list) {
      if (compact.some((c) => c.contains(r))) continue;
      // remove any existing roots contained in r
      for (let i = compact.length - 1; i >= 0; i--) {
        if (r.contains(compact[i])) compact.splice(i, 1);
      }
      compact.push(r);
    }
    return compact;
  }

  function runQueue(queue) {
    scheduled = true;

    const step = (deadline) => {
      try {
        while (queue.length) {
          // process while we have time (or timeout fired)
          const enoughTime = deadline.didTimeout || deadline.timeRemaining() > 8;
          if (!enoughTime) break;

          const root = queue.shift();
          try { applyI18n(root); } catch (_) {}
        }
      } finally {
        if (queue.length) {
          request(step); // continue later
        } else {
          scheduled = false;
        }
      }
    };

    request(step);
  }

  return (mutations) => {
    // collect roots that changed
    try {
      if (Array.isArray(mutations)) {
        for (const m of mutations) {
          if (m && m.target && m.target.nodeType === 1) pending.add(m.target);
        }
      } else if (mutations && mutations.nodeType === 1) {
        pending.add(mutations);
      } else {
        // fallback
        pending.add(document.body || document.documentElement);
      }
    } catch (_) {
      pending.add(document.body || document.documentElement);
    }

    if (scheduled) return;

    // flush pending -> build compact queue
    const roots = compactRoots(pending);
    pending.clear();

    // if nothing collected, just do a safe single apply on body
    const queue = roots.length ? roots : [document.body || document.documentElement];
    runQueue(queue);
  };
})();


  // Initial apply
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', () => {
      try { applyI18n(document); } catch (_) {}

      // Watch for dynamically added nodes/options (table rows, dialogs, injected filters/buttons)
      try {
       const mo = new MutationObserver((muts) => schedule(muts));
        mo.observe(document.body, { childList: true, subtree: true });
      } catch (_) {}

      // If some scripts run after load and create UI late, re-apply once more.
     setTimeout(() => { try { schedule(document.body || document.documentElement); } catch (_) {} }, 300);
    }, { once: true });
  } else {
    try { applyI18n(document); } catch (_) {}
    try {
      const mo = new MutationObserver((muts) => schedule(muts));
      mo.observe(document.body, { childList: true, subtree: true });
    } catch (_) {}
    setTimeout(() => { try { schedule(document.body || document.documentElement); } catch (_) {} }, 300);
  }
})();
